#!/bin/sh
###
# @file
# Shell script to assemble and dump the fake Int10h handler from NASM source to
# a C array.
#
# Copyright (C) 2014, Red Hat, Inc.
# Copyright (c) 2013 - 2014, Intel Corporation. All rights reserved.<BR>
#
# SPDX-License-Identifier: BSD-2-Clause-Patent
#
###

set -e -u

STEM=$(dirname -- "$0")/$(basename -- "$0" .sh)

#
# Install exit handler -- remove temporary files.
#
exit_handler()
{
  rm -f -- "$STEM".bin "$STEM".disasm "$STEM".offsets "$STEM".insns \
      "$STEM".bytes
}
trap exit_handler EXIT

#
# Assemble the source file.
#
nasm -o "$STEM".bin -- "$STEM".asm

#
# Disassemble it, in order to get a binary dump associated with the source.
# (ndisasm doesn't recognize the "--" end-of-options delimiter.)
#
ndisasm "$STEM".bin >"$STEM".disasm

#
# Create three files, each with one column of the disassembly.
#
# The first column contains the offsets, and it starts the comment.
#
cut -c 1-8 -- "$STEM".disasm \
| sed -e 's,^,  /* ,' >"$STEM".offsets

#
# The second column contains the assembly-language instructions, and it closes
# the comment. We first pad it to 30 characters.
#
cut -c 29- -- "$STEM".disasm \
| sed -e 's,$,                              ,' \
      -e 's,^\(.\{30\}\).*$,\1 */,' >"$STEM".insns

#
# The third column contains the bytes corresponding to the instruction,
# represented as C integer constants. First strip trailing whitespace from the
# middle column of the input disassembly, then process pairs of nibbles.
#
cut -c 11-28 -- "$STEM".disasm \
| sed -e 's, \+$,,' -e 's/\(..\)/ 0x\1,/g' >"$STEM".bytes

#
# Write the output file, recombining the columns. The output should have CRLF
# line endings.
#
{
  printf '//\n'
  printf '// THIS FILE WAS GENERATED BY "%s". DO NOT EDIT.\n' \
      "$(basename -- "$0")"
  printf '//\n'
  printf '#ifndef _VBE_SHIM_H_\n'
  printf '#define _VBE_SHIM_H_\n'
  printf 'STATIC CONST UINT8 mVbeShim[] = {\n'
  paste -d ' ' -- "$STEM".offsets "$STEM".insns "$STEM".bytes
  printf '};\n'
  printf '#endif\n'
} \
| unix2dos >"$STEM".h
