/*
 * Copyright 2022-2025 The OpenSSL Project Authors. All Rights Reserved.
 *
 * Licensed under the Apache License 2.0 (the "License").  You may not use
 * this file except in compliance with the License.  You can obtain a copy
 * in the file LICENSE in the source distribution or at
 * https://www.openssl.org/source/license.html
 */

#include <openssl/crypto.h>
#include "internal/e_os.h"
#include "internal/time.h"

/* system-specific variants defining OSSL_sleep() */
#if (defined(OPENSSL_SYS_UNIX) || defined(__DJGPP__)) \
    && !defined(OPENSSL_USE_SLEEP_BUSYLOOP)
# include <unistd.h>

static void ossl_sleep_millis(uint64_t millis)
{
# ifdef OPENSSL_SYS_VXWORKS
    struct timespec ts;

    ts.tv_sec = (long int) (millis / 1000);
    ts.tv_nsec = (long int) (millis % 1000) * 1000000ul;
    nanosleep(&ts, NULL);
# elif defined(__TANDEM) && !defined(_REENTRANT)
#   include <cextdecs.h(PROCESS_DELAY_)>

    /* HPNS does not support usleep for non threaded apps */
    PROCESS_DELAY_(millis * 1000);
# else
    unsigned int s = (unsigned int)(millis / 1000);
    unsigned int us = (unsigned int)((millis % 1000) * 1000);

    sleep(s);
    usleep(us);
# endif
}
#elif defined(_WIN32) && !defined(OPENSSL_SYS_UEFI)
# include <windows.h>

static void ossl_sleep_millis(uint64_t millis)
{
    /*
     * Windows' Sleep() takes a DWORD argument, which is smaller than
     * a uint64_t, so we need to limit it to 49 days, which should be enough.
     */
    DWORD limited_millis = (DWORD)-1;

    if (millis < limited_millis)
        limited_millis = (DWORD)millis;
    Sleep(limited_millis);
}

#else
/* Fallback to a busy wait */
# define USE_SLEEP_SECS

static void ossl_sleep_secs(uint64_t secs)
{
    /*
     * sleep() takes an unsigned int argument, which is smaller than
     * a uint64_t, so it needs to be limited to 136 years which
     * should be enough even for Sleeping Beauty.
     */
    unsigned int limited_secs = UINT_MAX;

    if (secs < limited_secs)
        limited_secs = (unsigned int)secs;
    sleep(limited_secs);
}

static void ossl_sleep_millis(uint64_t millis)
{
    const OSSL_TIME finish
        = ossl_time_add(ossl_time_now(), ossl_ms2time(millis));

    while (ossl_time_compare(ossl_time_now(), finish) < 0)
        /* busy wait */ ;
}
#endif /* defined(OPENSSL_SYS_UNIX) || defined(__DJGPP__) */

void OSSL_sleep(uint64_t millis)
{
    OSSL_TIME now = ossl_time_now();
    OSSL_TIME finish = ossl_time_add(now, ossl_ms2time(millis));
    uint64_t left = millis;

#if defined(USE_SLEEP_SECS)
    do {
        ossl_sleep_secs(left / 1000);
        now = ossl_time_now();
        left = ossl_time2ms(ossl_time_subtract(finish, now));
    } while (ossl_time_compare(now, finish) < 0 && left > 1000);

    if (ossl_time_compare(now, finish) >= 0)
        return;
#endif

    do {
        ossl_sleep_millis(left);
        now = ossl_time_now();
        left = ossl_time2ms(ossl_time_subtract(finish, now));
    } while (ossl_time_compare(now, finish) < 0);
}
