/*
 * Copyright (C) 2014 David Goulet <dgoulet@efficios.com>
 * Copyright (C) 2019 Jérémie Galarneau <jeremie.galarneau@efficios.com>
 *
 * SPDX-License-Identifier: LGPL-2.1-only
 *
 */

#ifndef LTTNG_SESSION_H
#define LTTNG_SESSION_H

#ifdef __cplusplus
extern "C" {
#endif

#include <lttng/constant.h>

struct lttng_handle;
struct lttng_session_descriptor;
struct lttng_destruction_handle;

/*
 * Basic session information.
 *
 * The "enabled" field is only used when listing the sessions which indicate if
 * it's started or not.
 *
 * The structures should be initialized to zero before use.
 */
#define LTTNG_SESSION_PADDING1             8
struct lttng_session {
	char name[LTTNG_NAME_MAX];
	/*
	 * Human-readable representation of the trace's destination.
	 * In the case of a local tracing session, a path is provided:
	 *     /path/to/the/output
	 *
	 * In the case of a remote (network) tracing session, the string has
	 * the following format:
	 *     net://hostname/path:ctrl_port [data: data_port]
	 */
	char path[PATH_MAX];
	uint32_t enabled;	/* enabled/started: 1, disabled/stopped: 0 */
	uint32_t snapshot_mode;
	unsigned int live_timer_interval;	/* usec */

	/*
	 * End of public attributes.
	 * The remaining fields are used to deal with ABI management concerns.
	 */

	/*
	 * 32-bit architectures are already naturally aligned on 4 bytes after
	 * 'live_timer_interval'. However, the offset does not result in a
	 * natural alignment on 64-bit architectures. Adding 4 bytes of
	 * padding here results in an aligned offset after 'alignement_padding'
	 * for both bitnesses.
	 *
	 * This was added since not all compilers appear to align unions in the
	 * same way. Some (e.g. MSVC) do not seem to impose an alignement
	 * constraint while others (e.g. gcc, clang, icc) seem to align it to
	 * ensure 'ptr' is naturally aligned.
	 */
	char alignment_padding[4];
	union {
		/*
		 * Ensure the 'extended' union has the same size for both
		 * 32-bit and 64-bit builds.
		 */
		char padding[LTTNG_SESSION_PADDING1];
		void *ptr;
	} extended;
};

/*
 * Create a session on the session daemon from a session descriptor.
 *
 * See the session descriptor API description in session-descriptor.h
 *
 * Note that unspecified session descriptor parameters, such as a session's
 * name, are updated in the session descriptor if the creation of the session
 * succeeds. This allows users to query the session's auto-generated name
 * after its creation. Note that other attributes can be queried using the
 * session listing API.
 *
 * Returns LTTNG_OK on success. See lttng-error.h for the meaning of the other
 * return codes.
 */
extern enum lttng_error_code lttng_create_session_ext(
		struct lttng_session_descriptor *session_descriptor);

/*
 * Create a tracing session using a name and an optional URL.
 *
 * If _url_ is NULL, no consumer is created for the session. The name can't be
 * NULL here.
 *
 * Return 0 on success else a negative LTTng error code.
 */
extern int lttng_create_session(const char *name, const char *url);

/*
 * Create a tracing session that will exclusively be used for snapshot meaning
 * the session will be in no output mode and every channel enabled for that
 * session will be set in overwrite mode and in mmap output since splice is not
 * supported.
 *
 * Name can't be NULL. If an url is given, it will be used to create a default
 * snapshot output using it as a destination. If NULL, no output will be
 * defined and an add-output call will be needed.
 *
 * Return 0 on success else a negative LTTng error code.
 */
extern int lttng_create_session_snapshot(const char *name,
		const char *snapshot_url);

/*
 * Create a session exclusively used for live reading.
 *
 * In this mode, the switch-timer parameter is forced for each UST channel, a
 * live-switch-timer is enabled for kernel channels, manually setting
 * switch-timer is forbidden. Synchronization beacons are sent to the relayd,
 * indexes are sent and metadata is checked for each packet.
 *
 * Name can't be NULL. If no URL is given, the default is to send the data to
 * net://127.0.0.1. The timer_interval is in usec.
 *
 * Return 0 on success else a negative LTTng error code.
 */
extern int lttng_create_session_live(const char *name, const char *url,
		unsigned int timer_interval);

/*
 * Destroy a tracing session.
 *
 * The session will not be usable, tracing will be stopped thus buffers will be
 * flushed.
 *
 * This call will wait for data availability for each domain of the session,
 * which can take an arbitrary amount of time. However, when returning the
 * tracing data is guaranteed to be ready to be read and analyzed.
 *
 * lttng_destroy_session_no_wait() may be used if such a guarantee is not
 * needed.
 *
 * The name can't be NULL here.
 *
 * Return 0 on success else a negative LTTng error code.
 */
extern int lttng_destroy_session(const char *name);

/*
 * Destroy a tracing session.
 *
 * Performs the same function as lttng_destroy_session(), but provides
 * an lttng_destruction_handle which can be used to wait for the completion
 * of the session's destruction. The lttng_destroy_handle can also be used
 * obtain the status and archive location of any implicit session
 * rotation that may have occurred during the session's destruction.
 *
 * Returns LTTNG_OK on success. The returned handle is owned by the caller
 * and must be free'd using lttng_destruction_handle_destroy().
 */
extern enum lttng_error_code lttng_destroy_session_ext(const char *session_name,
		struct lttng_destruction_handle **handle);

/*
 * Behaves exactly like lttng_destroy_session but does not wait for data
 * availability.
 */
extern int lttng_destroy_session_no_wait(const char *name);

/*
 * List all the tracing sessions.
 *
 * Return the number of entries of the "lttng_session" array. The caller
 * must free the returned sessions array directly using free().
 *
 * On error, a negative LTTng error code is returned.
 */
extern int lttng_list_sessions(struct lttng_session **sessions);

/*
 * Get the creation time of an lttng_session object on the session daemon.
 *
 * This function must only be used with lttng_session objects returned
 * by lttng_list_sessions() or lttng_session_create().
 *
 * The creation time returned is a UNIX timestamp; the number of seconds since
 * Epoch (1970-01-01 00:00:00 +0000 (UTC)).
 *
 * Returns LTTNG_OK on success. See lttng-error.h for the meaning of the other
 * return codes.
 */
extern enum lttng_error_code lttng_session_get_creation_time(
		const struct lttng_session *session, uint64_t *creation_time);

/*
 * Set the shared memory path for a session.
 *
 * Sets the (optional) file system path where shared memory buffers will
 * be created for the session. This is useful for buffer extraction on
 * crash, when used with filesystems like pramfs.
 *
 * Return 0 on success else a negative LTTng error code.
 */
extern int lttng_set_session_shm_path(const char *session_name,
		const char *shm_path);

#ifdef __cplusplus
}
#endif

#endif /* LTTNG_SESSION_H */
