/*
 * Copyright 2020 NXP.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef IMX_SW_PDM_H
#define IMX_SW_PDM_H

#include <stdint.h>
#include <stdbool.h>

#define SWPDM_OK	(0)
#define SWPDM_ERR	(-1)

/*
 * CIC pdm tp pcm decoder avaialble types
 *  cic 5th order
 *  9 tap 1/2 down sampling FIR
 *  1/2 down sampling IIR
 *  total down sampling 12*2*2 = 48
 */

typedef enum CIC_pdmToPcmType
{
    /* cic down sampling by 12
     * total down sampling 12*2*2 = 48 */
    CIC_pdmToPcmType_cic_order_5_cic_downsample_12,
    /* cic down sampling by 16
     * total down sampling 16*2*2 = 64 */
    CIC_pdmToPcmType_cic_order_5_cic_downsample_16,
    /* cic down sampling by 24
     * total down sampling 24*2*2 = 96 */
    CIC_pdmToPcmType_cic_order_5_cic_downsample_24,
    /* cic down sampling by 32
     * total down sampling 32*2*2 = 128 */
    CIC_pdmToPcmType_cic_order_5_cic_downsample_32,
    /* cic down sampling by 48
     * total down sampling 48*2*2 = 192 */
    CIC_pdmToPcmType_cic_order_5_cic_downsample_48,
    CIC_pdmToPcmType_cic_order_5_cic_downsample_max,
    CIC_pdmToPcmType_cic_order_5_cic_downsample_unavailable = CIC_pdmToPcmType_cic_order_5_cic_downsample_max
} cic_t;

/*
 * C interface control block
 * cicDecoderType:		CIC decoder type
 * afePdmToPcmCicDecoder:	CIC decoder object pointer
 * cicOrder:			CIC decoder order
 * cicDownsamplingeRatio:	CIC down sampling ratio
 * inputBufferSizePerChannel:	Input buffer size per channel
 * inputBuffer:			Input buffer pointer
 *				Number of sample sets, sample set is 1 channel sample x N channels
 *				each set is interleved unsinged 32 bits words containing 32 PDM
 *				sample per channel
 * outputBufferSizePerChannel:	Output buffer size per channel
 * outputBuffer:		Output buffer pointer
 *				Number of sample sets, sample set is 1 channel sample x N channels
 *				each set is interleved singed 32 bits words containing 1
 *				PCM sample per channel
 * outputGainFactor:		nominal scale factor of CIC decoder
 * numberOfChannels:		Number of channels
 */

typedef struct AfeCicDecoder_C
{
    enum CIC_pdmToPcmType cicDecoderType;
    void *afePdmToPcmCicDecoder;
    unsigned cicOrder;
    unsigned cicDownsamplingeRatio;
    unsigned inputBufferSizePerChannel;
    uint32_t *inputBuffer;
    unsigned outputBufferSizePerChannel;
    int32_t *outputBuffer;
    float outputGainFactor;
    unsigned numberOfChannels;
} afe_t;

/*
 * construct C interface afe cic
 *
 * input :
 *	cicDecoderType: one of CIC_pdmToPcmType
 *	afeCicDecoder_C_cb: pointer to cic pdm to pcm control block tp be filled
 *	outputGainFactor: sample output multipler scale factor if == 0 it get set to default
 *	outputSamplesPerRunPERChannel_divi_16 : number samples to output per channel divided by 16
 * output:
 *	afeCicDecoder_C_cb: filled cic pdm to pcm control block
 *
 * return :
 *	true  : if succesessful in constructiuon the object
 *	false : if unsuccesessful in constructiuon the object
 *		(deleteAfeCicDecoder should be call to free any allocated memory)
 */

extern bool constructAfeCicDecoder(cic_t cicDecoderType,
	struct AfeCicDecoder_C *afeCicDecoder_C_cb, float outputGainFactor,
	unsigned outputSamplesPerRunPERChannel_divi_16);

/*
 * process input samples and generate oitput samples
 *
 *    input:
 *	afeCicDecoder_C_cb: pointer to cic pdm to pcm control block tp be filled
 *	afeCicDecoder_C_cb->inputBuffer: filled with feCicDecoder_C_cb->inputBufferSizePerChannel
 *		input samples
 *    output:
 *	afeCicDecoder_C_cb->outputBuffer: filled with feCicDecoder_C_cb->outputBufferSizePerChannel
 *		output samples
 */

extern void processAfeCic(struct AfeCicDecoder_C *afeCicDecoder_C_cb);

/*
 * delete decoder object and all buffers
 *    input:
 *	afeCicDecoder_C_cb: filled cic pdm to pcm control block
 */

extern void deleteAfeCicDecoder(struct AfeCicDecoder_C *afeCicDecoder_C_cb);

/*
 * CIC pdm tp pcm decoder available types
 *  :cic 5th order
 *  :16 tap 1/2 down sampling FIR
 *  :1/2 down sampling IIR
 *  :total down sampling cic_downsample*2*2
 */

extern cic_t getPdmTpPcmType(unsigned outputToInputSampleRateRatio);

#endif /* IMX_SW_PDM_H */
