package org.maplibre.android.maps

import android.content.Context
import android.graphics.PointF
import android.view.View
import androidx.test.annotation.UiThreadTest
import androidx.test.internal.runner.junit4.AndroidJUnit4ClassRunner
import androidx.test.platform.app.InstrumentationRegistry
import org.maplibre.android.AppCenter
import org.maplibre.android.MapLibre
import org.maplibre.android.WellKnownTileServer
import org.maplibre.android.camera.CameraPosition
import org.maplibre.android.geometry.LatLng
import org.maplibre.android.geometry.LatLngBounds
import org.maplibre.android.geometry.ProjectedMeters
import org.maplibre.android.maps.renderer.MapRenderer
import org.maplibre.android.style.layers.TransitionOptions
import org.maplibre.android.testapp.utils.TestConstants
import org.junit.After
import org.junit.Assert
import org.junit.Assert.*
import org.junit.Before
import org.junit.Test
import org.junit.runner.RunWith
import org.maplibre.android.testapp.styles.TestStyles

@RunWith(AndroidJUnit4ClassRunner::class)
class NativeMapViewTest : AppCenter() {

    private lateinit var nativeMapView: NativeMap

    companion object {
        const val DELTA = 0.000001
        const val DELTA_BIG = 1.0
        const val BEARING_TEST = 60.0
        const val PITCH_TEST = 40.0
        const val ZOOM_TEST = 16.0
        val PADDING_TEST = doubleArrayOf(80.0, 150.0, 0.0, 0.0)
        const val WIDTH = 500
        const val HEIGHT = WIDTH
        val LATLNG_TEST = LatLng(12.0, 34.0)
    }

    @Before
    @UiThreadTest
    fun before() {
        val context = InstrumentationRegistry.getInstrumentation().context
        val apiKey = MapLibre.getApiKey()
        val options = NativeMapOptions(2.0f, false)

        MapLibre.getInstance(context, apiKey, WellKnownTileServer.MapTiler)
        nativeMapView = NativeMapView(context, options, null, null, DummyRenderer(context))
        nativeMapView.resizeView(WIDTH, HEIGHT)
    }

    @After
    @UiThreadTest
    fun after() {
        val context = InstrumentationRegistry.getInstrumentation().context
        val apiKey = MapLibre.getApiKey()
        MapLibre.getInstance(context)
        nativeMapView.destroy()
    }

    @Test
    @UiThreadTest
    fun testSetStyleUrl() {
        val expected = TestStyles.getPredefinedStyleWithFallback("Pastel")
        nativeMapView.styleUri = expected
        val actual = nativeMapView.styleUri
        assertEquals("Style URL should match", expected, actual)
    }

    @Test
    @UiThreadTest
    fun testSetStyleJson() {
        val expected = "{}"
        nativeMapView.styleJson = expected
        val actual = nativeMapView.styleJson
        assertEquals("Style JSON should match", expected, actual)
    }

    @Test
    @UiThreadTest
    fun testBearing() {
        val expected = BEARING_TEST
        nativeMapView.setBearing(expected, 0)
        val actual = nativeMapView.bearing
        assertEquals("Bearing should match", expected, actual, DELTA)
    }

    @Test
    @UiThreadTest
    fun testLatLng() {
        val expected = LATLNG_TEST
        nativeMapView.setLatLng(expected, 0)
        val actual = nativeMapView.latLng
        assertEquals("Latitude should match", expected.latitude, actual.latitude, DELTA)
        assertEquals("Longitude should match", expected.longitude, actual.longitude, DELTA)
    }

    @Test
    @UiThreadTest
    fun testLatLngPadding() {
        val expected = LATLNG_TEST
        nativeMapView.contentPadding = PADDING_TEST
        nativeMapView.setLatLng(expected, 0)
        val actual = nativeMapView.latLng
        assertEquals("Latitude should match", expected.latitude, actual.latitude, DELTA)
        assertEquals("Longitude should match", expected.longitude, actual.longitude, DELTA)
        assertArrayEquals(PADDING_TEST, nativeMapView.cameraPosition.padding, DELTA)
    }

    @Test
    @UiThreadTest
    fun testLatLngDefault() {
        val expected = LatLng()
        val actual = nativeMapView.latLng
        assertEquals("Latitude should match", expected.latitude, actual.latitude, DELTA)
        assertEquals("Longitude should match", expected.longitude, actual.longitude, DELTA)
    }

    @Test
    @UiThreadTest
    fun testBearingDefault() {
        val expected = 0.0
        val actual = nativeMapView.bearing
        assertEquals("Bearing should match", expected, actual, DELTA)
    }

    @Test
    @UiThreadTest
    fun testPitch() {
        val expected = PITCH_TEST
        nativeMapView.setPitch(expected, 0)
        val actual = nativeMapView.pitch
        assertEquals("Pitch should match", expected, actual, DELTA)
    }

    @Test
    @UiThreadTest
    fun testPitchDefault() {
        val expected = 0.0
        val actual = nativeMapView.pitch
        assertEquals("Pitch should match", expected, actual, DELTA)
    }

    @Test
    @UiThreadTest
    fun testZoom() {
        val expected = ZOOM_TEST
        nativeMapView.setZoom(expected, PointF(0.0f, 0.0f), 0)
        val actual = nativeMapView.zoom
        assertEquals("Zoom should match", expected, actual, DELTA)
    }

    @Test
    @UiThreadTest
    fun testZoomDefault() {
        val expected = 0.0
        val actual = nativeMapView.zoom
        assertEquals("Zoom should match", expected, actual, DELTA)
    }

    @Test
    @UiThreadTest
    fun testJumpTo() {
        val expected = CameraPosition.Builder()
            .bearing(BEARING_TEST)
            .target(LATLNG_TEST)
            .tilt(PITCH_TEST)
            .zoom(ZOOM_TEST)
            .padding(PADDING_TEST)
            .build()
        // verify that the lazily set padding is ignored when a value is provided with the camera
        nativeMapView.contentPadding = doubleArrayOf(1.0, 2.0, 3.0, 4.0)
        nativeMapView.jumpTo(LATLNG_TEST, ZOOM_TEST, PITCH_TEST, BEARING_TEST, PADDING_TEST)
        val actual = nativeMapView.cameraPosition
        assertEquals("Latitude should match", expected.target!!.latitude, actual.target!!.latitude, DELTA)
        assertEquals("Longitude should match", expected.target!!.longitude, actual.target!!.longitude, DELTA)
        assertEquals("Bearing should match", expected.bearing, actual.bearing, DELTA)
        assertEquals("Pitch should match", expected.tilt, actual.tilt, DELTA)
        assertEquals("Zoom should match", expected.zoom, actual.zoom, DELTA)
        assertArrayEquals(expected.padding, actual.padding, DELTA)
    }

    @Test
    @UiThreadTest
    fun testLatLngForPixel() {
        val expected = LATLNG_TEST
        nativeMapView.setLatLng(LATLNG_TEST, 0)
        val actual = nativeMapView.latLngForPixel(
            PointF((WIDTH / 2).toFloat(), (HEIGHT / 2).toFloat())
        )
        assertEquals("Latitude should match", expected.latitude, actual.latitude, DELTA_BIG)
        assertEquals("Longitude should match", expected.longitude, actual.longitude, DELTA_BIG)
    }

    @Test
    @UiThreadTest
    fun testPixelForLatLng() {
        val expected = PointF((WIDTH / 2).toFloat(), (HEIGHT / 2).toFloat())
        nativeMapView.setLatLng(LATLNG_TEST, 0)
        val actual = nativeMapView.pixelForLatLng(LATLNG_TEST)
        assertEquals("X should match", expected.x.toDouble(), actual.x.toDouble(), DELTA_BIG)
        assertEquals("Y should match", expected.y.toDouble(), actual.y.toDouble(), DELTA_BIG)
    }

    @Test
    @UiThreadTest
    fun testPrefetchTilesTrue() {
        val expected = true
        nativeMapView.prefetchTiles = true
        val actual = nativeMapView.prefetchTiles
        assertEquals("Flag should match", expected, actual)
    }

    @Test
    @UiThreadTest
    fun testPrefetchTilesFalse() {
        val expected = false
        nativeMapView.prefetchTiles = false
        val actual = nativeMapView.prefetchTiles
        assertEquals("Flag should match", expected, actual)
    }

    @Test
    @UiThreadTest
    fun testPrefetchTilesDefault() {
        val expected = true
        val actual = nativeMapView.prefetchTiles
        assertEquals("Flag should match", expected, actual)
    }

    @Test
    @UiThreadTest
    fun testPrefetchZoomDelta() {
        val expected = 2
        nativeMapView.prefetchZoomDelta = 2
        val actual = nativeMapView.prefetchZoomDelta
        assertEquals("Prefetch zoom delta should match", expected, actual)
    }

    @Test
    @UiThreadTest
    fun testPrefetchZoomDeltaDefault() {
        val expected = 4
        val actual = nativeMapView.prefetchZoomDelta
        assertEquals("Prefetch zoom delta should match", expected, actual)
    }

    @Test
    @UiThreadTest
    fun testSetContentPadding() {
        val expected = doubleArrayOf(1.0, 2.0, 3.0, 4.0)
        nativeMapView.contentPadding = expected
        val actual = nativeMapView.contentPadding
        assertEquals("Left should match", expected[0], actual[0], 0.0)
        assertEquals("Top should match", expected[1], actual[1], 0.0)
        assertEquals("Right should match", expected[2], actual[2], 0.0)
        assertEquals("Bottom should match", expected[3], actual[3], 0.0)
    }

    @Test
    @UiThreadTest
    fun testSetMinZoom() {
        val expected = 12.0
        nativeMapView.minZoom = expected
        val actual = nativeMapView.minZoom
        assertEquals("Min zoom should match", expected, actual, 0.0)
    }

    @Test
    @UiThreadTest
    fun testSetMaxZoom() {
        val expected = 12.0
        nativeMapView.maxZoom = expected
        val actual = nativeMapView.maxZoom
        assertEquals("Max zoom should match", expected, actual, 0.0)
    }

    @Test
    @UiThreadTest
    fun testSetMinPitch() {
        val expected = 60.0
        nativeMapView.minPitch = expected
        val actual = nativeMapView.minPitch
        assertEquals("Min Pitch should match", expected, actual, 0.01)
    }

    @Test
    @UiThreadTest
    fun testSetMaxPitch() {
        val expected = 60.0
        nativeMapView.maxPitch = expected
        val actual = nativeMapView.maxPitch
        assertEquals("Max Pitch should match", expected, actual, 0.01)
    }

    @Test
    @UiThreadTest
    fun testGetProjectedMetersAtLatitude() {
        val expected = 77973.67021115532
        val actual = nativeMapView.getMetersPerPixelAtLatitude(5.0)
        assertEquals("Get projected meters should match", expected, actual, DELTA)
    }

    @Test
    @UiThreadTest
    fun testLatLngForProjectedMeters() {
        val expected = LatLng(0.01796630538796444, 0.02694945852363162)
        val actual = nativeMapView.latLngForProjectedMeters(ProjectedMeters(2000.0, 3000.0))
        assertEquals("Lat for projected meters", expected.latitude, actual.latitude, DELTA)
        assertEquals("Lng for projected meters", expected.longitude, actual.longitude, DELTA)
    }

    @Test
    @UiThreadTest
    fun testFlyTo() {
        val expected = CameraPosition.Builder()
            .zoom(12.0)
            .tilt(30.0)
            .target(LatLng(12.0, 14.0))
            .bearing(20.0)
            .padding(PADDING_TEST)
            .build()
        // verify that the lazily set padding is ignored when a value is provided with the camera
        nativeMapView.contentPadding = doubleArrayOf(1.0, 2.0, 3.0, 4.0)
        nativeMapView.flyTo(expected.target!!, expected.zoom, expected.bearing, expected.tilt, PADDING_TEST, 0)
        val actual = nativeMapView.cameraPosition
        assertEquals("Bearing should match", expected.bearing, actual.bearing, TestConstants.BEARING_DELTA)
        assertEquals("Latitude should match", expected.target!!.latitude, actual.target!!.latitude, TestConstants.LAT_LNG_DELTA)
        assertEquals("Longitude should match", expected.target!!.longitude, actual.target!!.longitude, TestConstants.LAT_LNG_DELTA)
        assertEquals("Tilt should match", expected.tilt, actual.tilt, TestConstants.TILT_DELTA)
        assertEquals("Zoom should match", expected.zoom, actual.zoom, TestConstants.ZOOM_DELTA)
        Assert.assertArrayEquals(expected.padding, actual.padding, DELTA)
    }

    @Test
    @UiThreadTest
    fun testEaseTo() {
        val expected = CameraPosition.Builder()
            .zoom(12.0)
            .tilt(30.0)
            .target(LatLng(12.0, 14.0))
            .bearing(20.0)
            .padding(PADDING_TEST)
            .build()
        // verify that the lazily set padding is ignored when a value is provided with the camera
        nativeMapView.contentPadding = doubleArrayOf(1.0, 2.0, 3.0, 4.0)
        nativeMapView.easeTo(expected.target!!, expected.zoom, expected.bearing, expected.tilt, PADDING_TEST, 0, false)
        val actual = nativeMapView.cameraPosition
        assertEquals("Bearing should match", expected.bearing, actual.bearing, TestConstants.BEARING_DELTA)
        assertEquals("Latitude should match", expected.target!!.latitude, actual.target!!.latitude, TestConstants.LAT_LNG_DELTA)
        assertEquals("Longitude should match", expected.target!!.longitude, actual.target!!.longitude, TestConstants.LAT_LNG_DELTA)
        assertEquals("Tilt should match", expected.tilt, actual.tilt, TestConstants.TILT_DELTA)
        assertEquals("Zoom should match", expected.zoom, actual.zoom, TestConstants.ZOOM_DELTA)
        assertArrayEquals(expected.padding, actual.padding, DELTA)
    }

    @Test
    @UiThreadTest
    fun testResetPosition() {
        val expected = CameraPosition.Builder()
            .zoom(0.0)
            .tilt(0.0)
            .target(LatLng(0.0, 0.0))
            .bearing(0.0)
            .padding(PADDING_TEST)
            .build()
        nativeMapView.jumpTo(LatLng(1.0, 2.0), 12.0, 23.0, 1.0, PADDING_TEST)
        nativeMapView.resetPosition()
        val actual = nativeMapView.cameraPosition
        assertEquals("Bearing should match", expected.bearing, actual.bearing, TestConstants.BEARING_DELTA)
        assertEquals("Latitude should match", expected.target!!.latitude, actual.target!!.latitude, TestConstants.LAT_LNG_DELTA)
        assertEquals("Longitude should match", expected.target!!.longitude, actual.target!!.longitude, TestConstants.LAT_LNG_DELTA)
        assertEquals("Tilt should match", expected.tilt, actual.tilt, TestConstants.TILT_DELTA)
        assertEquals("Zoom should match", expected.zoom, actual.zoom, TestConstants.ZOOM_DELTA)
        assertArrayEquals(expected.padding, actual.padding, DELTA)
    }

    @Test
    @UiThreadTest
    fun testGetCameraForLatLngBounds() {
        val expected = CameraPosition.Builder()
            .zoom(3.5258764777024005)
            .tilt(0.0)
            .target(LatLng(23.182767623652808, 13.999999999994088))
            .bearing(0.0)
            .build()
        val actual = nativeMapView.getCameraForLatLngBounds(
            LatLngBounds.from(30.0, 16.0, 16.0, 12.0),
            intArrayOf(0, 0, 0, 0),
            0.0,
            0.0
        )
        assertEquals("Bearing should match", expected.bearing, actual.bearing, TestConstants.BEARING_DELTA)
        assertEquals("Latitude should match", expected.target!!.latitude, actual.target!!.latitude, TestConstants.LAT_LNG_DELTA)
        assertEquals("Longitude should match", expected.target!!.longitude, actual.target!!.longitude, TestConstants.LAT_LNG_DELTA)
        assertEquals("Tilt should match", expected.tilt, actual.tilt, TestConstants.TILT_DELTA)
        assertEquals("Zoom should match", expected.zoom, actual.zoom, TestConstants.ZOOM_DELTA)
    }

    @Test
    @UiThreadTest
    fun testMoveBy() {
        val expected = CameraPosition.Builder()
            .zoom(0.0)
            .tilt(0.0)
            .target(LatLng(4.21494310024160, -4.218749958739409))
            .bearing(0.0)
            .build()
        nativeMapView.moveBy(12.0, 12.0, 0)
        val actual = nativeMapView.cameraPosition
        assertEquals("Bearing should match", expected.bearing, actual.bearing, TestConstants.BEARING_DELTA)
        assertEquals("Latitude should match", expected.target!!.latitude, actual.target!!.latitude, TestConstants.LAT_LNG_DELTA)
        assertEquals("Longitude should match", expected.target!!.longitude, actual.target!!.longitude, TestConstants.LAT_LNG_DELTA)
        assertEquals("Tilt should match", expected.tilt, actual.tilt, TestConstants.TILT_DELTA)
        assertEquals("Zoom should match", expected.zoom, actual.zoom, TestConstants.ZOOM_DELTA)
    }

    @Test
    @UiThreadTest
    fun testTransitionOptions() {
        val transitionOptions = TransitionOptions(500, 500)
        nativeMapView.transitionOptions = transitionOptions
        assertTrue(transitionOptions.isEnablePlacementTransitions)
        assertEquals(transitionOptions, nativeMapView.transitionOptions)
    }

    @Test
    @UiThreadTest
    fun testTransitionOptions_disablePlacementTransitions() {
        val transitionOptions = TransitionOptions(500, 500, false)
        nativeMapView.transitionOptions = transitionOptions
        assertFalse(transitionOptions.isEnablePlacementTransitions)
        assertEquals(transitionOptions, nativeMapView.transitionOptions)
    }

    class DummyRenderer(context: Context) : MapRenderer(context, null) {

        private var renderingRefreshMode: RenderingRefreshMode = RenderingRefreshMode.WHEN_DIRTY

        override fun requestRender() {
            // no-op
        }

        override fun queueEvent(runnable: Runnable?) {
            // no-op
        }

        override fun waitForEmpty() {
            // no-op
        }

        override fun getView(): View? {
            return null;
        }

        override fun setRenderingRefreshMode(mode : RenderingRefreshMode) {
            renderingRefreshMode = mode
        }

        override fun getRenderingRefreshMode() : RenderingRefreshMode{
            return renderingRefreshMode
        }
    }
}
