// Copyright (C) 2016 The Qt Company Ltd.
// Copyright (C) 2016 Intel Corporation.
// SPDX-License-Identifier: LicenseRef-Qt-Commercial OR GPL-3.0-only

#undef QT_NO_FOREACH // this file contains unported legacy Q_FOREACH uses

#include "tst_bench_qhash.h"

#include <QFile>
#include <QHash>
#include <QString>
#include <QStringList>
#include <QUuid>
#include <QTest>


class tst_QHash : public QObject
{
    Q_OBJECT

private slots:
    void initTestCase();
    void qhash_current_data() { data(); }
    void qhash_current() { qhash_template<QString>(); }
    void qhash_qt50_data() { data(); }
    void qhash_qt50() { qhash_template<Qt50String>(); }
    void qhash_qt4_data() { data(); }
    void qhash_qt4() { qhash_template<Qt4String>(); }
    void qhash_javaString_data() { data(); }
    void qhash_javaString() { qhash_template<JavaString>(); }

    void hashing_current_data() { data(); }
    void hashing_current() { hashing_template<QString>(); }
    void hashing_qt50_data() { data(); }
    void hashing_qt50()  { hashing_template<Qt50String>(); }
    void hashing_qt4_data() { data(); }
    void hashing_qt4() { hashing_template<Qt4String>(); }
    void hashing_javaString_data() { data(); }
    void hashing_javaString() { hashing_template<JavaString>(); }

private:
    void data();
    template <typename String> void qhash_template();
    template <typename String> void hashing_template();

    QStringList smallFilePaths;
    QStringList uuids;
    QStringList dict;
    QStringList numbers;
};

///////////////////// QHash /////////////////////

#include <QDebug>

void tst_QHash::initTestCase()
{
    QHashSeed::setDeterministicGlobalSeed();

    // small list of strings (that happen to look like file paths produced long
    // ago by cd ../.. && find . -print, but that's irrelevant).
    QFile smallPathsData(QFINDTESTDATA("paths_small_data.txt"));
    QVERIFY(smallPathsData.open(QIODevice::ReadOnly));
    smallFilePaths = QString::fromLatin1(smallPathsData.readAll()).split(QLatin1Char('\n'));
    QVERIFY(!smallFilePaths.isEmpty());

    // list of UUIDs
    // guaranteed to be completely random, generated by http://xkcd.com/221/
    QUuid ns = QUuid("{f43d2ef3-2fe9-4563-a6f5-5a0100c2d699}");
    uuids.reserve(smallFilePaths.size());

    foreach (const QString &path, smallFilePaths)
        uuids.append(QUuid::createUuidV5(ns, path).toString());


    // lots of strings with alphabetical characters, vaguely reminiscent of
    // a dictionary.
    //
    // this programmatically generates a series like:
    //  AAAAAA
    //  AAAAAB
    //  AAAAAC
    //  ...
    //  AAAAAZ
    //  AAAABZ
    //  ...
    //  AAAAZZ
    //  AAABZZ
    QByteArray id("AAAAAAA");

    if (dict.isEmpty()) {
        for (int i = id.size() - 1; i > 0;) {
            dict.append(id);
            char c = id.at(i);
            id[i] = ++c;

            if (c == 'Z') {
                // wrap to next digit
                i--;
                id[i] = 'A';
            }
        }
    }

    // string versions of numbers.
    for (int i = 5000000; i < 5005001; ++i)
        numbers.append(QString::number(i));
}

void tst_QHash::data()
{
    QTest::addColumn<QStringList>("items");
    QTest::newRow("paths-small") << smallFilePaths;
    QTest::newRow("uuids-list") << uuids;
    QTest::newRow("dictionary") << dict;
    QTest::newRow("numbers") << numbers;
}

template <typename String> void tst_QHash::qhash_template()
{
    QFETCH(QStringList, items);
    QHash<String, int> hash;

    QList<String> realitems;
    foreach (const QString &s, items)
        realitems.append(s);

    QBENCHMARK {
        for (int i = 0, n = realitems.size(); i != n; ++i) {
            hash[realitems.at(i)] = i;
        }
    }
}

template <typename String> void tst_QHash::hashing_template()
{
    // just the hashing function
    QFETCH(QStringList, items);

    QList<String> realitems;
    realitems.reserve(items.size());
    foreach (const QString &s, items)
        realitems.append(s);

    QBENCHMARK {
        for (int i = 0, n = realitems.size(); i != n; ++i)
            (void)qHash(realitems.at(i));
    }
}

QTEST_MAIN(tst_QHash)

#include "tst_bench_qhash.moc"
