/*
 * faux.c - Generic package support
 *
 * Copyright (C) 2015 Intel Corporation
 *
 * cve-check-tool is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */
#define _GNU_SOURCE
#include <glib.h>
#include <libgen.h>
#include <sys/stat.h>
#include <limits.h>
#include <stdlib.h>

#include "faux.h"
#include "util.h"
#include "cve-check-tool.h"
#include "plugin.h"

void faux_free(struct source_package_t *pkg)
{
        struct FauxData *d = NULL;

        if (pkg && pkg->extra) {
                d = pkg->extra;
                if (d->patched) {
                        g_strfreev(d->patched);
                }
                if (d->ignored) {
                        g_strfreev(d->ignored);
                }
                free(pkg->extra);
                pkg->extra = NULL;
        }
}

static bool faux_patch_check(struct source_package_t *t, char *id, bool ignore)
{
        struct FauxData *d = NULL;
        gchar **list = NULL;

        if (!t->extra) {
                return false;
        }

        d = t->extra;
        list = ignore ? d->ignored : d->patched;
        if (!list) {
                return false;
        }

        for (uint i = 0; i < g_strv_length(list); i++) {
                list[i] = g_strchomp(list[i]);
                if (g_str_equal(list[i], "")) {
                        continue;
                }
                if (g_str_equal(id, list[i])) {
                        return true;
                }
        }

        return false;
}

bool faux_is_patched(struct source_package_t *t, char *id)
{
        return faux_patch_check(t, id, false);
}

bool faux_is_ignored(struct source_package_t *t, char *id)
{
        return faux_patch_check(t, id, true);
}

bool faux_is_package(const char *filename)
{
        return g_str_has_suffix((const gchar*)filename, ".csv");
}

_module_export_ bool cve_plugin_module_init(CvePlugin *self)
{
        self->flags = PLUGIN_TYPE_PACKAGE;
        self->name = "faux";
        self->is_ignored = faux_is_ignored;
        self->is_patched = faux_is_patched;
        self->is_package = faux_is_package;
        self->scan_package = NULL;
        self->free_package = faux_free;
        return true;
}

/*
 * Editor modelines  -  https://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 8
 * tab-width: 8
 * indent-tabs-mode: nil
 * End:
 *
 * vi: set shiftwidth=8 tabstop=8 expandtab:
 * :indentSize=8:tabSize=8:noTabs=true:
 */
