/*
 * This file is part of cve-check-tool
 * Copyright (C) 2015 Intel Corporation
 *
 * cve-check-tool is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#define _GNU_SOURCE
#include <check.h>
#include <stdlib.h>
#include <stdio.h>

#include "util.h"
#include "util.c"
#include "cve-string.c"
#include "core.c"
#include "hashmap.c"

#include "config.h"

START_TEST(cve_database_new)
{
        CveDB *db = NULL;

        db = cve_db_new(":memory:");
        fail_if(!db, "Failed to create CveDB");
        cve_db_free(db);
}
END_TEST

START_TEST(cve_database_load)
{
        CveDB *db = NULL;

        db = cve_db_new(":memory:");
        fail_if(!db, "Failed to create CveDB");

        fail_if(!cve_db_load(db, TOP_DIR "/tests/dummy_data/nvdcve-2.0-2002.xml"),
                "Failed to load database");

        cve_db_free(db);
}
END_TEST

START_TEST(cve_database_fetch)
{
        CveDB *db = NULL;
        GList *ret = NULL;
        __attribute__ ((unused)) gchar *cve = NULL;
        struct cve_entry_t *t = NULL;

        db = cve_db_new(":memory:");
        fail_if(!db, "Failed to create CveDB");

        fail_if(!cve_db_load(db, TOP_DIR "/tests/dummy_data/nvdcve-2.0-2002.xml"),
                "Failed to load database");

        ret = cve_db_get_issues(db, "ssleay", "0.9");
        fail_if(!ret, "Failed to get issues list for ssleay");

        fail_if(g_list_length(ret) != 2, "Incorrect issue count for ssleay");

        cve = ret->data;
        t = cve_db_get_cve(db, cve);
        fail_if(!t, "Failed to retrieve CVE entry");
        fail_if(!g_str_equal(t->id, cve), "Mismatched CVE return");
        cve_free(t);
        t = NULL;

        cve = ret->next->data;
        t = cve_db_get_cve(db, cve);
        fail_if(!t, "Failed to retrieve CVE entry");
        fail_if(!g_str_equal(t->id, cve), "Mismatched CVE return");
        cve_free(t);
        t = NULL;
        

        g_list_free_full(ret, g_free);

        cve_db_free(db);
}
END_TEST

static Suite *core_suite(void)
{
        Suite *s = NULL;
        TCase *tc = NULL;

        s = suite_create("cve_database");
        tc = tcase_create("cve_dtabase_functions");
        tcase_set_timeout(tc, 60);
        tcase_add_test(tc, cve_database_new);
        tcase_add_test(tc, cve_database_load);
        tcase_add_test(tc, cve_database_fetch);
        suite_add_tcase(s, tc);

        return s;
}

int main(void)
{
        Suite *s;
        SRunner *sr;
        int fail;

        s = core_suite();
        sr = srunner_create(s);
        srunner_run_all(sr, CK_VERBOSE);
        fail = srunner_ntests_failed(sr);
        srunner_free(sr);

        xmlCleanupParser();

        if (fail > 0) {
                return EXIT_FAILURE;
        }

        return EXIT_SUCCESS;
}

/*
 * Editor modelines  -  https://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 8
 * tab-width: 8
 * indent-tabs-mode: nil
 * End:
 *
 * vi: set shiftwidth=8 tabstop=8 expandtab:
 * :indentSize=8:tabSize=8:noTabs=true:
 */
