/*
 * update.c - cve-check-tool
 *
 * Copyright (C) 2015 Intel Corporation
 *
 * cve-check-tool is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <string.h>
#include <stdint.h>
#include <glib.h>
#include <gio/gio.h>

#include "cve-check-tool.h"

#include "util.h"
#include "config.h"
#include "cve-string.h"
#include "cve-db-lock.h"
#include "core.h"

#include "update.h"

static void show_version(void)
{
        const gchar *msg = "\
" PACKAGE " " PACKAGE_VERSION "\n\
Copyright (C) 2015 Intel Corporation\n\
" PACKAGE_NAME " is free software; you can redistribute it and/or modify\n\
it under the terms of the GNU General Public License as published by\n\
the Free Software Foundation; either version 2 of the License, or\n\
(at your option) any later version.";
        fprintf(stderr, "%s\n", msg);
}

static gchar *nvds = NULL;
static bool _show_version = false;
static bool _quiet = false;

static GOptionEntry _entries[] = {
        { "nvd-dir", 'd', 0, G_OPTION_ARG_STRING, &nvds, "NVD directory in filesystem", NULL },
        { "version", 'v', 0, G_OPTION_ARG_NONE, &_show_version, "Show version", NULL },
        { "quiet", 'q', 0, G_OPTION_ARG_NONE, &_quiet, "Run silently", NULL },
        { .short_name = 0 }
};

/**
 * Main entry.
 */
int main(int argc, char **argv)
{
        autofree(GError) *error = NULL;
        autofree(GOptionContext) *context = NULL;
        autofree(cve_string) *db_path = NULL;
        int ret = EXIT_FAILURE;
        bool db_locked;

        LIBXML_TEST_VERSION
        context = g_option_context_new(" - cve update");
        g_option_context_add_main_entries(context, _entries, NULL);
        if (!g_option_context_parse(context, &argc, &argv, &error)) {
                g_printerr("Invalid options: %s\n", error->message);
                goto end;
        }

        if (_show_version) {
                show_version();
                ret = EXIT_SUCCESS;
                goto end;
        }

        db_path = get_db_path(nvds);
        if (!db_path) {
                fprintf(stderr, "main(): Can't get db path\n");
                goto end;
        }

        db_locked = cve_db_lock_init(db_path->str);
        if (!db_locked) {
                fputs("Not continuing without a database lock\n", stderr);
                goto end;
        }

        if (update_db(_quiet, db_path->str)) {
                ret = EXIT_SUCCESS;
        } else {
                fprintf(stderr, "Failed to update database\n");
        }

        cve_db_lock_fini();
end:
        xmlCleanupParser();
        return ret;
}

/*
 * Editor modelines  -  https://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 8
 * tab-width: 8
 * indent-tabs-mode: nil
 * End:
 *
 * vi: set shiftwidth=8 tabstop=8 expandtab:
 * :indentSize=8:tabSize=8:noTabs=true:
 */
