/*
 * common.h - cve-check-tool
 *
 * Copyright (C) 2015 Intel Corporation
 *
 * cve-check-tool is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#pragma once

#include <glib.h>
#include <stdbool.h>
#include <stdint.h>

struct source_package_t;

/**
 * Resolution status that should be implemented by the bug plugin
 */
typedef enum {
        REPORT_STATUS_MIN,
        REPORT_STATUS_OPEN,              /**<Bug is open and being tracked */
        REPORT_STATUS_CLOSED,            /**<Bug is now closed */
        REPORT_STATUS_CLOSED_WILLNOTFIX, /**<Bug is now closed but marked as will not fix */
        REPORT_STATUS_UNREPORTED,        /**<Currently unknown to bug tracker */
        REPORT_STATUS_MAX
} ReportStatus;

/**
 * Maps vulnerabilities into a consumable format
 */
struct vulnerability_t {
        gchar *product; /**<CPE product field */
        gchar *vendor;  /**<CPE vendor field */
        gchar *version; /**<Matching CPE version, if any */
        guint hash;     /**<Unique hash for database */
};

/**
 * Simple representation of a CVE
 */
struct cve_entry_t {
        gchar *id;              /**<CVE ID */
        gchar *summary;         /**<Description of CVE */
        GList *uris;            /**<Set of online links, usually patches or ML reports */
        gchar *score;           /**<CVSS score of this CVE */
        gchar *vector;          /**<CVSS access vector of this CVE */
        int64_t modified;       /**<Last modified timestamp */
        ReportStatus status;   /**<Status within the implementations bug tracker */
};

/**
 * Indicates the (supported) package type
 */
typedef enum {
        PACKAGE_TYPE_MIN,
        PACKAGE_TYPE_RPM,       /**<RPM package */
        PACKAGE_TYPE_SRPM,      /**<SRPM package */
        PACKAGE_TYPE_EOPKG,     /**<EOPKG package */
        PACKAGE_TYPE_PKGBUILD,  /**<PKGBUILD (Arch) package */
        PACKAGE_TYPE_FAUX,      /**<Forced CSV input */
        PACKAGE_TYPE_UNKNOWN,   /**<Unknown package type */
        PACKAGE_TYPE_MAX
} PackageType;

/**
 * Representation of a source package
 */
typedef struct source_package_t {
        void *name;             /**<Our name for the package */
        void *version;          /**<Our current version */
        char *path;             /**<Filesystem path of source path */
        int release;            /**<Highest (available) release number */
        GList *issues;          /**<List of applicable (non-patched) CVEs */
        GList *patched;         /**<List of *patched* applicable CVEs */
        bool xml;               /**<Whether this is libxml's memory */
        void* extra;            /**<Extra storage for each implementation */
        PackageType type;      /**<Package type */
} SourcePackage;

/*
 * Editor modelines  -  https://www.wireshark.org/tools/modelines.html
 *
 * Local variables:
 * c-basic-offset: 8
 * tab-width: 8
 * indent-tabs-mode: nil
 * End:
 *
 * vi: set shiftwidth=8 tabstop=8 expandtab:
 * :indentSize=8:tabSize=8:noTabs=true:
 */
