/*
 * Application interface library for the AVIRT driver
 *
 * avirt.h - AVIRT Application interface library header
 *
 * Copyright (C) 2018, 2019 Fiberdyne Systems Pty Ltd
 * Author: Mark Farrugia <mark.farrugia@fiberdyne.com.au>
 *
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef _AVIRT_H_
#define _AVIRT_H_

#define MAX_STREAMS 16
#define MAX_NAME_LEN 80

#include <alsa/asoundlib.h>
#include <stdbool.h>

/**
 * AVIRT Audio Path info
 */
typedef struct snd_avirt_audiopath snd_avirt_audiopath;
struct snd_avirt_audiopath {
  const char *uid; /* Unique identifier */
  const char *name; /* Pretty name */
  unsigned int version[3]; /* Version - Major.Minor.Ext */
};

/*
 * Audio stream configuration
 */
struct snd_avirt_stream {
  const char name[MAX_NAME_LEN]; /* Stream name */
  const char map[MAX_NAME_LEN]; /* Stream Audio Path mapping */
  unsigned int channels; /* Stream channel count */
  unsigned int direction; /* Stream direction */
};

/**
 * Audio routing
 */
struct snd_avirt_route {
  char name[MAX_NAME_LEN]; /* Route name */
  unsigned int channels; /* Route channel count */
  unsigned int direction; /* Stream direction */
  char source_ap[MAX_NAME_LEN];
  char sink_ap[MAX_NAME_LEN];
};

/**
 * snd_avirt_stream_new - Create a stream in AVIRT
 * @name: The name of the stream
 * @channels: The number of channels for the stream
 * @direction: The stream direction
 *             (SND_PCM_STREAM_PLAYBACK or SND_PCM_STREAM_CAPTURE)
 * @map: The audio path to map this stream to
 * @internal: Make the PCM internal.
 * @return: 0 on success, negative ERRNO otherwise
 *
 * Each stream creates a PCM device for the AVIRT sound card.
 * Streams will not appear to the user-space until `snd_avirt_card_seal()`
 * is called.
 */
int snd_avirt_stream_new(const char *name, unsigned int channels, int direction,
                         const char *map, bool internal);

/**
 * snd_avirt_route_new - Create a route between Audio Paths in AVIRT
 * @name: The name of the route.
 * @channels: Number of channels for the route
 * @direction: The route stream direction
 *             (SND_PCM_STREAM_PLAYBACK or SND_PCM_STREAM_CAPTURE)
 * @from_ap: The Audio Path UID to route audio from
 * @to_ap: The Audio Path UID to route audio to
 * @return: 0 on success, negative ERRNO otherwise
 *
 * Allows Audio Paths to be piped together to allow for modular processing
 * and/or networking. The "to" Audio Path must be able to accept a stream
 * input whose channel count equals to "from" Audio Path's output channel
 * count.
 */
int snd_avirt_route_new(const char *name, int channels, int direction,
                        const char *from_ap, const char *to_ap);

int snd_avirt_routes(struct snd_avirt_route **routes, int *count);
int snd_avirt_streams(struct snd_avirt_stream **streams, int *count);
int snd_avirt_audiopaths(struct snd_avirt_audiopath **audiopaths, int *count);

/**
 * snd_avirt_card_configure - Finalize AVIRT stream creation and register sound card
 * @return: 0 on success, negative ERRNO otherwise
 *
 * This should be called once all streams have been created via
 * `snd_avirt_stream_new()`. Calling this function will register the AVIRT
 * sound card to the user-space, and will configure all mapped lower-level
 * Audio Paths for the given stream configuration.
 *
 * NOTE: Once this function is called, no more streams may be added.
 */
int snd_avirt_card_configure();

int snd_avirt_card_unconfigure();

/**
 * snd_avirt_pcm_info - Get PCM info for a given PCM name
 * @pcm_name: The PCM name of which to retrieve the PCM info
 * @pcm_info: The PCM info struct to populate
 * @return: 0 on success, negative ERRNO otherwise
 *
 * Can be used to retrieve a `snd_pcm_info_t` struct for a given PCM device from
 * it's name. Useful for retrieving card index and PCM index for a PCM.
 */
int snd_avirt_pcm_info(const char *pcm_name, snd_pcm_info_t *pcm_info);

/**
 * snd_avirt_card_index_get - Get the ALSA card index for an AVIRT card
 * @avirt_idx: The AVIRT internal index (TD MF: perhaps change to UID?)
 * @return: The ALSA card index on success, negative ERRNO otherwise
 */
int snd_avirt_card_index_get(int avirt_idx);

/**
 * snd_avirt_ctl_set_volume - Set an ALSA mixer volume value
 * @name: The ALSA volume control name to set
 * @volume: The volume to set
 * @return: 0 on success, negative ERRNO otherwise
 */
int snd_avirt_ctl_set_volume(const char *name, long volume);

/**
 * snd_avirt_ctl_set_volume - Get an ALSA mixer volume value
 * @name: The ALSA volume control name to get
 * @volume: Populate this with the gotten value
 * @return: 0 on success, negative ERRNO otherwise
 */
int snd_avirt_ctl_get_volume(const char *name, long *volume);

#endif // _AVIRT_H_
