/*
 * Copyright (c) 2017 TOYOTA MOTOR CORPORATION
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif
#include <memory>
#include <algorithm>
#include "hs-helper.h"
#include "hmi-debug.h"
#include "hs-clientmanager.h"


const char _error[] = "error";
const char _application_id[] = "application_id";
const char _display_message[] = "display_message";
const char _reply_message[] = "reply_message";

static HS_ClientManager* g_client_manager = HS_ClientManager::instance();

/*
********** Method of HomeScreen Service (API) **********
*/

static void pingSample(afb_req_t request)
{
   static int pingcount = 0;
   afb_req_success_f(request, json_object_new_int(pingcount), "Ping count = %d", pingcount);
   HMI_NOTICE("homescreen-service","Verbosity macro at level notice invoked at ping invocation count = %d", pingcount);
   pingcount++;
}

/**
 * tap_shortcut notify for homescreen
 * When Shortcut area is tapped,  notify these applciations
 *
 * #### Parameters
 * Request key
 * - application_id   : application id
 *
 * #### Return
 * None
 *
 */
static void tap_shortcut (afb_req_t request)
{
    HMI_NOTICE("homescreen-service","called.");
    int ret = g_client_manager->tap_shortcut(request);
    if (ret != 0) {
      afb_req_fail_f(request, "failed", "called %s, Unknown parameter", __FUNCTION__);
      return;
    }

    // response to HomeScreen
    struct json_object *res = json_object_new_object();
    hs_add_object_to_json_object_func(res, __FUNCTION__, 2,
      _error,  ret);
    afb_req_success(request, res, "afb_event_push event [tap_shortcut]");
}

/**
 * HomeScreen OnScreen message
 *
 * #### Parameters
 * Request key
 * - display_message   : message for display
 *
 * #### Return
 * None
 *
 */
static void on_screen_message (afb_req_t request)
{
    HMI_NOTICE("homescreen-service","called.");

    int ret = g_client_manager->on_screen_message(request);
    if (ret != 0) {
      afb_req_fail_f(request, "failed", "called %s, Unknown parameter", __FUNCTION__);
      return;
    }

  // response to HomeScreen
    struct json_object *res = json_object_new_object();
    hs_add_object_to_json_object_func(res, __FUNCTION__, 2,
      _error,  ret);
    afb_req_success(request, res, "afb_event_push event [on_screen_message]");
}

/**
 * HomeScreen OnScreen Reply
 *
 * #### Parameters
 * Request key
 * - reply_message   : message for reply
 *
 * #### Return
 * None
 *
 */
static void on_screen_reply (afb_req_t request)
{
    HMI_NOTICE("homescreen-service","called.");

    int ret = g_client_manager->on_screen_reply(request);
    if (ret != 0) {
      afb_req_fail_f(request, "failed", "called %s, Unknown parameter", __FUNCTION__);
      return;
    }

  // response to HomeScreen
    struct json_object *res = json_object_new_object();
    hs_add_object_to_json_object_func(res, __FUNCTION__, 2,
      _error,  ret);
    afb_req_success(request, res, "afb_event_push event [on_screen_reply]");
}

/**
 * Subscribe event
 *
 * #### Parameters
 *  - event  : Event name. Event list is written in libhomescreen.cpp
 *
 * #### Return
 * None
 *
 */
static void subscribe(afb_req_t request)
{
    HMI_NOTICE("homescreen-service","called.");

    int ret = g_client_manager->subscribe(request);
    if(ret) {
      afb_req_fail_f(request, "afb_req_subscribe failed", "called %s.", __FUNCTION__);
      return;
    }

    /*create response json object*/
    struct json_object *res = json_object_new_object();
    hs_add_object_to_json_object_func(res, __FUNCTION__, 2,
        _error, ret);
    afb_req_success_f(request, res, "homescreen binder subscribe.");
}

/**
 * Unsubscribe event
 *
 * #### Parameters
 *  - event  : Event name. Event list is written in libhomescreen.cpp
 *
 * #### Return
 * None
 *
 */
static void unsubscribe(afb_req_t request)
{
    HMI_NOTICE("homescreen-service","called.");

    int ret = g_client_manager->unsubscribe(request);
    if(ret) {
      afb_req_fail_f(request, "afb_req_unsubscribe failed", "called %s.", __FUNCTION__);
      return;
    }

    /*create response json object*/
    struct json_object *res = json_object_new_object();
    hs_add_object_to_json_object_func(res, __FUNCTION__, 2,
        _error, ret);
    afb_req_success_f(request, res, "homescreen binder unsubscribe success.");
}

/**
 * showWindow event
 *
 * #### Parameters
 *  - request : the request
 *
 * #### Return
 * None
 *
 */
static void showWindow(afb_req_t request)
{
    HMI_NOTICE("homescreen-service","called.");

    int ret = g_client_manager->showWindow(request);
    if (ret != 0) {
      afb_req_fail_f(request, "failed", "called %s, Unknown parameter", __FUNCTION__);
      return;
    }

    // response to HomeScreen
    struct json_object *res = json_object_new_object();
    hs_add_object_to_json_object_func(res, __FUNCTION__, 2,
      _error,  ret);
    afb_req_success(request, res, "afb_event_push event [showWindow]");
}

/**
 * hideWindow event
 *
 * #### Parameters
 *  - request : the request
 *
 * #### Return
 * None
 *
 */
static void hideWindow(afb_req_t request)
{
    HMI_NOTICE("homescreen-service","called.");

    int ret = g_client_manager->hideWindow(request);
    if (ret != 0) {
      afb_req_fail_f(request, "failed", "called %s, Unknown parameter", __FUNCTION__);
      return;
    }

    // response to HomeScreen
    struct json_object *res = json_object_new_object();
    hs_add_object_to_json_object_func(res, __FUNCTION__, 2,
      _error,  ret);
    afb_req_success(request, res, "afb_event_push event [hideWindow]");
}

/**
 * replyShowWindow event
 *
 * #### Parameters
 *  - request : the request
 *
 * #### Return
 *  None
 *
 */
static void replyShowWindow(afb_req_t request)
{
    HMI_NOTICE("homescreen-service","called.");

    int ret = g_client_manager->replyShowWindow(request);
    if (ret != 0) {
      afb_req_fail_f(request, "failed", "called %s, Unknown parameter", __FUNCTION__);
      return;
    }

    // response to HomeScreen
    struct json_object *res = json_object_new_object();
    hs_add_object_to_json_object_func(res, __FUNCTION__, 2,
      _error,  ret);
    afb_req_success(request, res, "afb_event_push event [replyShowWindow]");
}

/**
 * showNotification event
 *
 * the contents to homescreen which display at top area.
 *
 * #### Parameters
 *  - request : the request
 *
 * #### Return
 * None
 *
 */
static void showNotification(afb_req_t request)
{
    HMI_NOTICE("homescreen-service","called.");

    int ret = g_client_manager->showNotification(request);
    if (ret != 0) {
      afb_req_fail_f(request, "failed", "called %s, Unknown parameter", __FUNCTION__);
      return;
    }

    // response to Application
    struct json_object *res = json_object_new_object();
    hs_add_object_to_json_object_func(res, __FUNCTION__, 2,
      _error,  ret);
    afb_req_success(request, res, "afb_event_push event [showNotification]");
}

/**
 * showInformation event
 *
 * the contents to homescreen which display at bottom area.
 *
 * #### Parameters
 *  - request : the request
 *
 * #### Return
 * None
 *
 */
static void showInformation(afb_req_t request)
{
    HMI_NOTICE("homescreen-service","called.");

    int ret = g_client_manager->showInformation(request);
    if (ret != 0) {
      afb_req_fail_f(request, "failed", "called %s, Unknown parameter", __FUNCTION__);
      return;
    }

    // response to Application
    struct json_object *res = json_object_new_object();
    hs_add_object_to_json_object_func(res, __FUNCTION__, 2,
      _error,  ret);
    afb_req_success(request, res, "afb_event_push event [showInformation]");
}

/*
 * array of the verbs exported to afb-daemon
 */
static const afb_verb_t verbs[]= {
    /* VERB'S NAME                 FUNCTION TO CALL                  */
    { .verb="ping",              .callback=pingSample             },
    { .verb="tap_shortcut",      .callback=tap_shortcut           },
    { .verb="showWindow",        .callback=showWindow             },
    { .verb="hideWindow",        .callback=hideWindow             },
    { .verb="replyShowWindow",   .callback=replyShowWindow        },
    { .verb="on_screen_message", .callback=on_screen_message      },
    { .verb="on_screen_reply",   .callback=on_screen_reply        },
    { .verb="subscribe",         .callback=subscribe              },
    { .verb="unsubscribe",       .callback=unsubscribe            },
    { .verb="showNotification",  .callback=showNotification       },
    { .verb="showInformation",   .callback=showInformation        },
    {NULL } /* marker for end of the array */
};

/**
 * homescreen binding preinit function
 *
 * #### Parameters
 *  - api : the api serving the request
 *
 * #### Return
 * None
 *
 */
static int preinit(afb_api_t api)
{
   HMI_NOTICE("homescreen-service","binding preinit (was register)");
   return 0;
}

/**
 * homescreen binding init function
 *
 * #### Parameters
 *  - api : the api serving the request
 *
 * #### Return
 * None
 *
 */
static int init(afb_api_t api)
{
    HMI_NOTICE("homescreen-service","binding init");

    g_client_manager->init();

    return 0;
}

/**
 * homescreen binding event function
 *
 * #### Parameters
 *  - api : the api serving the request
 *  - event  : event name
 *  - object : event json object
 *
 * #### Return
 * None
 *
 */
static void onevent(afb_api_t api, const char *event, struct json_object *object)
{
   HMI_NOTICE("homescreen-service","on_event %s", event);
}

const afb_binding_t afbBindingExport = {
    .api = "homescreen",
    .specification = NULL,
    .info = NULL,
    .verbs = verbs,
    .preinit = preinit,
    .init = init,
    .onevent = onevent
};
