/*
 * Copyright (c) 2018 TOYOTA MOTOR CORPORATION
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "hs-client.h"
#include "hs-helper.h"
#include "hmi-debug.h"

static const char _type[] = "type";
static const char _text[] = "text";
static const char _info[] = "info";
static const char _icon[] = "icon";
static const char _parameter[] = "parameter";
static const char _replyto[] = "replyto";

/**
 * HS_Client construction function
 *
 * #### Parameters
 *  - id: app's id
 *
 * #### Return
 * None
 *
 */
HS_Client::HS_Client(afb_req_t request, std::string id) : my_id(id)
{
    HMI_NOTICE("homescreen-service","called.");
    my_event = afb_api_make_event(request->api, id.c_str());
}

/**
 * HS_Client destruction function
 *
 * #### Parameters
 *  - null
 *
 * #### Return
 * None
 *
 */
HS_Client::~HS_Client()
{
    HMI_NOTICE("homescreen-service","called.");
    afb_event_unref(my_event);
}

/**
 * push tap_shortcut event
 *
 * #### Parameters
 *  - appid: app's id.
 *
 * #### Return
 * result
 *
 */
int HS_Client::tap_shortcut(const char* appid)
{
    if(!checkEvent(__FUNCTION__))
        return 0;

    HMI_NOTICE("homescreen-service","%s application_id = %s.", __FUNCTION__, appid);
    struct json_object* push_obj = json_object_new_object();
    hs_add_object_to_json_object_str( push_obj, 4, _application_id, appid,
    _type, __FUNCTION__);
    afb_event_push(my_event, push_obj);
    return 0;
}

/**
 * push on_screen_message event
 *
 * #### Parameters
 *  - message: post message.
 *
 * #### Return
 * result
 *
 */
int HS_Client::on_screen_message(afb_req_t request, const char* message)
{
    if(!checkEvent(__FUNCTION__))
        return 0;

    HMI_NOTICE("homescreen-service","push %s event message [%s].", __FUNCTION__, message);
    struct json_object* push_obj = json_object_new_object();
    hs_add_object_to_json_object_str( push_obj, 4, _display_message, message,
    _type, __FUNCTION__);
    afb_event_push(my_event, push_obj);
    return 0;
}

/**
 * push on_screen_reply event
 *
 * #### Parameters
 *  - message: reply message.
 *
 * #### Return
 * result
 *
 */
int HS_Client::on_screen_reply(afb_req_t request, const char* message)
{
    if(!checkEvent(__FUNCTION__))
        return 0;

    HMI_NOTICE("homescreen-service","push %s event message [%s].", __FUNCTION__, message);
    struct json_object* push_obj = json_object_new_object();
    hs_add_object_to_json_object_str( push_obj, 4, _reply_message, message,
    _type, __FUNCTION__);
    afb_event_push(my_event, push_obj);
    return 0;
}

/**
 * subscribe event
 *
 * #### Parameters
 *  - event: homescreen event, tap_shortcut etc.
 *
 * #### Return
 * result
 *
 */
int HS_Client::subscribe(afb_req_t request, const char* event)
{
    int ret = 0;
    auto ip = event_list.find(std::string(event));
    if(ip == event_list.end()) {
        event_list[std::string(event)] = 0;
    }
    if(!subscription) {
        ret = afb_req_subscribe(request, my_event);
        if(ret == 0) {
            subscription = true;
        }
    }
    return ret;
}

/**
 * unsubscribe event
 *
 * #### Parameters
 *  - event: homescreen event, tap_shortcut etc.
 *
 * #### Return
 * result
 *
 */
int HS_Client::unsubscribe(afb_req_t request, const char* event)
{
    int ret = 0;
    event_list.erase(std::string(event));
    if(event_list.empty()) {
        ret = afb_req_unsubscribe(request, my_event);
    }
    return ret;
}

/**
 * check if client subscribe event
 *
 * #### Parameters
 *  - event: homescreen event, tap_shortcut etc.
 *
 * #### Return
 * true: found
 * false: not found
 *
 */
bool HS_Client::checkEvent(const char* event)
{
    auto ip = event_list.find(std::string(event));
    if(ip == event_list.end())
        return false;
    else
        return true;
}

/**
 * showWindow event
 *
 * input contents : {"application_id":"the appid that want to display", "parameter":{"area": "display area", ...}}
 *
 * #### Parameters
 * - request : the request
 * - appid : the appid that want to display
 *
 * #### Return
 * 0 : success
 * others : fail
 *
 */
int HS_Client::showWindow(afb_req_t request, const char* appid)
{
    if(!checkEvent(__FUNCTION__))
        return 0;

    HMI_NOTICE("homescreen-service","%s application_id = %s.", __FUNCTION__, appid);
    struct json_object* push_obj = json_object_new_object();
    hs_add_object_to_json_object_str( push_obj, 4, _application_id, appid, _type, __FUNCTION__);
    const char* param = afb_req_value(request, _parameter);
    if(param) {
        std::string req_appid = std::move(get_application_id(request));
        if(req_appid.empty()) {
            HMI_NOTICE("homescreen-service","can't get application identifier");
            return AFB_REQ_GETAPPLICATIONID_ERROR;
        }

        struct json_object* param_obj = json_tokener_parse(param);
        json_object_object_add(param_obj, _replyto, json_object_new_string(req_appid.c_str()));
        json_object_object_add(push_obj, _parameter, param_obj);
    }
    else {
        HMI_ERROR("homescreen-service","please input correct parameter.");
        return AFB_EVENT_BAD_REQUEST;
    }
    afb_event_push(my_event, push_obj);
    return 0;
}

/**
 * hideWindow event
 *
 * input contents : {"application_id":"the appid that want to hide"}
 *
 * #### Parameters
 * - request : the request
 *
 * #### Return
 * 0 : success
 * others : fail
 *
 */
int HS_Client::hideWindow(afb_req_t request)
{
    if(!checkEvent(__FUNCTION__))
        return 0;

    HMI_NOTICE("homescreen-service","%s called.", __FUNCTION__);
    std::string req_appid = std::move(get_application_id(request));
    if(req_appid.empty()) {
        HMI_NOTICE("homescreen-service","can't get application identifier");
        return AFB_REQ_GETAPPLICATIONID_ERROR;
    }

    struct json_object* push_obj = json_object_new_object();
    hs_add_object_to_json_object_str( push_obj, 4, _application_id, req_appid.c_str(),
    _type, __FUNCTION__);
    afb_event_push(my_event, push_obj);
    return 0;
}

/**
 * replyShowWindow event
 *
 * input contens : {"application_id": "the appid that want to reply",  "parameter": {...}}
 *
 * #### Parameters
 * - request : the request
 * - appid : the appid that want to reply
 *
 * #### Return
 * 0 : success
 * others : fail
 *
 */
int HS_Client::replyShowWindow(afb_req_t request, const char* appid)
{
    if(!checkEvent(__FUNCTION__))
        return 0;

    HMI_NOTICE("homescreen-service","%s application_id = %s.", __FUNCTION__, appid);
    struct json_object* push_obj = json_object_new_object();
    hs_add_object_to_json_object_str( push_obj, 4, _application_id, appid, _type, __FUNCTION__);
    const char* param = afb_req_value(request, _parameter);
    if(param) {
        json_object_object_add(push_obj, _parameter, json_tokener_parse(param));
    }
    else {
        HMI_ERROR("homescreen-service","please input correct parameter.");
        return AFB_EVENT_BAD_REQUEST;
    }

    afb_event_push(my_event, push_obj);
    return 0;
}

/**
 * showNotification event
 *
 * input contents : {"icon": "icon path", "text": "message contents"}
 *
 * #### Parameters
 *  - request : the request
 *
 * #### Return
 * 0 : success
 * others : fail
 *
 */
int HS_Client::showNotification(afb_req_t request)
{
    int ret = 0;
    const char *value = afb_req_value(request, _text);
    if(value) {
        HMI_NOTICE("homescreen-service","text is %s", value);
        std::string appid =std::move(get_application_id(request));
        if(appid.empty()) {
            HMI_NOTICE("homescreen-service","can't get application identifier");
            return AFB_REQ_GETAPPLICATIONID_ERROR;
        }

        struct json_object* param_obj = json_object_new_object();
        const char *icon = afb_req_value(request, _icon);
        if(icon) {
            json_object_object_add(param_obj, _icon, json_object_new_string(icon));
            json_object_object_add(param_obj, _text, json_object_new_string(value));
            struct json_object* push_obj = json_object_new_object();
            hs_add_object_to_json_object_str( push_obj, 4, _application_id, appid.c_str(), _type, __FUNCTION__);
            json_object_object_add(push_obj, _parameter, param_obj);
            afb_event_push(my_event, push_obj);
        }
        else {
            HMI_NOTICE("homescreen-service","please input icon.");
            ret = AFB_REQ_SHOWNOTIFICATION_ERROR;
        }
    }
    else {
        HMI_NOTICE("homescreen-service","please input text.");
        ret = AFB_REQ_SHOWNOTIFICATION_ERROR;
    }

    return ret;
}

/**
 * showInformation event
 *
 * input contents : {"info": "information contents"}
 *
 * #### Parameters
 *  - request : the request
 *
 * #### Return
 * 0 : success
 * others : fail
 *
 */
int HS_Client::showInformation(afb_req_t request)
{
    int ret = 0;
    const char *value = afb_req_value(request, _info);
    if(value) {
        HMI_NOTICE("homescreen-service","info is %s", value);
        std::string appid = std::move(get_application_id(request));
        if(appid.empty()) {
            HMI_NOTICE("homescreen-service","can't get application identifier");
            return AFB_REQ_GETAPPLICATIONID_ERROR;
        }

        struct json_object* param_obj = json_object_new_object();
        json_object_object_add(param_obj, _info, json_object_new_string(value));
        struct json_object* push_obj = json_object_new_object();
        hs_add_object_to_json_object_str( push_obj, 4, _application_id, appid.c_str(), _type, __FUNCTION__);
        json_object_object_add(push_obj, _parameter, param_obj);
        afb_event_push(my_event, push_obj);
    }
    else {
        HMI_NOTICE("homescreen-service","please input information.");
        ret = AFB_REQ_SHOWINFORMATION_ERROR;
    }

    return ret;
}
