// SPDX-License-Identifier: GPL-2.0
/*
 * AVIRT - ALSA Virtual Soundcard
 *
 * Copyright (c) 2010-2018 Fiberdyne Systems Pty Ltd
 * 
 * core.h - AVIRT internal header
 */

#ifndef __SOUND_AVIRT_CORE_H
#define __SOUND_AVIRT_CORE_H

#include <sound/avirt.h>

#include "utils.h"

extern struct snd_pcm_ops pcm_ops_avirt;

struct snd_avirt_core {
	int version[3];
	struct snd_card *card;
	struct device *dev;
	struct class *class;
	struct platform_device *plat_dev;
	struct config_group *stream_group;
	struct config_group *route_group;
	unsigned int stream_count;
	bool streams_configured;
};

struct snd_avirt_audiopath_obj {
	struct kobject kobj;
	struct list_head list;
	struct snd_avirt_audiopath *path;
};

/**
 * snd_avirt_configfs_init - Initialise the configfs system
 * @core: The snd_avirt_core pointer
 * @return: 0 on success, negative ERRNO on failure
 */
int __init snd_avirt_configfs_init(struct snd_avirt_core *core);

/**
 * snd_avirt_configfs_exit - Clean up and exit the configfs system
 * @core: The snd_avirt_core pointer
 */
void __exit snd_avirt_configfs_exit(struct snd_avirt_core *core);

/**
 * snd_avirt_sysfs_init - Initialise the sysfs system
 * @core: The snd_avirt_core pointer
 * @return: 0 on success, negative ERRNO on failure
 */
int __init snd_avirt_sysfs_init(struct snd_avirt_core *core);

/**
 * snd_avirt_sysfs_exit - Clean up and exit the sysfs system
 * @core: The snd_avirt_core pointer
 */
void __exit snd_avirt_sysfs_exit(struct snd_avirt_core *core);

/**
 * snd_avirt_audiopath_obj - creates an Audio Path object, assigns the kset
 * and registers it with sysfs.
 * @uid: Unique ID of the Audio Path
 * @return: Pointer to the Audio Path object or NULL if it failed.
 */
struct snd_avirt_audiopath_obj *snd_avirt_audiopath_create_obj(const char *uid);

/**
 * snd_avirt_audiopath_destroy_obj - destroys an Audio Path object, deregisters
 * it with sysfs
 * @name: the Audio Path object
 */
void snd_avirt_audiopath_destroy_obj(struct snd_avirt_audiopath_obj *p);

/**
 * snd_avirt_streams_configure - Register the sound card to user space
 * @return: 0 on success, negative ERRNO on failure
 */
int snd_avirt_streams_configure(void);

int snd_avirt_streams_unconfigure(void);

/**
 * snd_avirt_streams_configured - Check if streams have been configured or not
 * @return: true if configured, false otherwise
 */
bool snd_avirt_streams_configured(void);

/**
 * snd_avirt_stream_find_by_device - Get audio stream from device number
 * @device: The PCM device number corresponding to the desired stream
 * @return: The audio stream if found, or an error pointer otherwise
 */
struct snd_avirt_stream *snd_avirt_stream_find_by_device(unsigned int device);

/**
 * snd_avirt_stream_create - Create audio stream, including it's ALSA PCM device
 * @name: The name designated to the audio stream
 * @direction: The PCM direction (SNDRV_PCM_STREAM_PLAYBACK or
 *             SNDRV_PCM_STREAM_CAPTURE)
 * @internal: Whether the PCM should be internal or not
 * @return: The newly created audio stream if successful, or an error pointer
 */
struct snd_avirt_stream *snd_avirt_stream_create(const char *name,
						 int direction, bool internal);

/**
 * int snd_avirt_route_try_complete - Set up remaining parameters for a route.
 *                                    Channels, sink, and source Audio Paths
 *                                    should be set when calling this function.
 * @stream: The route to attempt to finalize parameters for.
 * @return: 0 on success, negative ERRNO on failure
 */
int snd_avirt_route_try_complete(struct snd_avirt_route *route);

/**
 * int snd_avirt_stream_try_complete - Set up remaining parameters for a stream.
 *                                     Channels and map should be set when
 *                                     calling this function.
 * @stream: The stream to attempt to finalize parameters for.
 * @return: 0 on success, negative ERRNO on failure
 */
int snd_avirt_stream_try_complete(struct snd_avirt_stream *stream);
void snd_avirt_stream_try_destroy(struct snd_avirt_stream *stream);

/**
 * snd_avirt_stream_set_map - Set Audio Path mapping for a given stream
 * @stream: The stream to assign the mapping to.
 * @map: The Audio Path UID to map the stream to.
 * @return: 0 on success, negative ERRNO on failure
 */
int snd_avirt_stream_set_map(struct snd_avirt_stream *stream, const char *map);

/**
 * snd_avirt_audiopath_get - retrieves the Audio Path by it's UID
 * @uid: Unique ID for the Audio Path
 * @return: Corresponding Audio Path
 */
struct snd_avirt_audiopath *snd_avirt_audiopath_get(const char *uid);

/**
 * snd_avirt_route_create - Create audio route
 * @name: The name designated to the audio route
 * @direction: The PCM direction (SNDRV_PCM_STREAM_PLAYBACK or
 *             SNDRV_PCM_STREAM_CAPTURE)
 * @return: The newly created audio route if successful, or an error pointer
 */
struct snd_avirt_route *snd_avirt_route_create(const char *name, int direction);

/**
 * snd_avirt_route_endpoint_pos - get route endpoint position for Audio Path
 * @pcm: The PCM whose route to inspect.
 * @ap_uid: The Audio Path UID to get
 * @endpoint: The route position (SND_AVIRT_ROUTE_SOURCE or SND_AVIRT_ROUTE_SINK)
 *            of the Audio Path in it's route (if any).
 * @return: 0 if an Audio Path is found in the route, -1 if there is no route,
 *          or -2 otherwise.
 */
int snd_avirt_route_endpoint_pos(struct snd_pcm *pcm, const char *ap_uid,
				 snd_avirt_route_endpoint *endpoint);

/**
 * snd_avirt_stream_from_config_item - Convert config_item to a snd_avirt_stream
 * @item: The config_item to convert from
 * @return: The item's snd_avirt_stream if successful, NULL otherwise
 */
static inline struct snd_avirt_stream *
snd_avirt_stream_from_config_item(struct config_item *item)
{
	return item ? container_of(item, struct snd_avirt_stream, item) : NULL;
}

/**
 * snd_avirt_route_from_config_item - Convert config_item to a snd_avirt_route
 * @item: The config_item to convert from
 * @return: The item's snd_avirt_stream if successful, NULL otherwise
 */
static inline struct snd_avirt_route *
snd_avirt_route_from_config_item(struct config_item *item)
{
	return item ? container_of(item, struct snd_avirt_route, item) : NULL;
}

#endif /* __SOUND_AVIRT_CORE_H */
