/*
 * Copyright (C) 2018 "IoT.bzh"
 * Author Fulup Ar Foll <fulup@iot.bzh>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#define _GNU_SOURCE  // needed for vasprintf

#include "alsa-softmixer.h"
#include <string.h>

PUBLIC AlsaSndCtlT *ApiSourceFindSubdev(SoftMixerT *mixer, const char *target) {

	// search for subdev into every registered source in the mixer

	AlsaSndPcmT * source;
	cds_list_for_each_entry(source, &mixer->sources.list, list) {
		if (source->uid && !strcmp(source->uid, target))
			return source->sndcard;
	}

    return NULL;
}

static void sourceDestroy(SoftMixerT* mixer, void * arg) {
	AlsaSndPcmT * source = (AlsaSndPcmT*) arg;
	AFB_ApiDebug(mixer->api, "%s... %s", __func__, source->uid);

	cds_list_del(&source->list);
	mixer->nbSources--;
	ApiPcmDelete(mixer, source);

	AFB_ApiDebug(mixer->api, "%s... DONE", __func__);
}

static AlsaSndPcmT * sourceCreate(SoftMixerT* mixer, const char * uid, json_object * argsJ) {
	AlsaSndPcmT * newSource = NULL;
	newSource = ApiPcmAttachOne(mixer, uid, SND_PCM_STREAM_CAPTURE, argsJ);
	if (!newSource) {
		goto fail;
	}

	mixer->nbSources++;
	cds_list_add(&newSource->list, &mixer->sources.list);
	AlsaMixerTransactionObjectAdd(mixer->transaction, newSource, sourceDestroy);

fail:
	return newSource;
}

PUBLIC int ApiSourceAttach(SoftMixerT *mixer, AFB_ReqT request, const char * uid, json_object * argsJ) {

	if (mixer->nbSources >=  mixer->max.sources) {
        AFB_ReqFailF(request, "too-small", "mixer=%s max source=%d", mixer->uid, mixer->max.sources);
		goto fail;
    }

	AlsaSndPcmT * newSource = NULL;

    switch (json_object_get_type(argsJ)) {
        long count;
        
        case json_type_object:
			newSource = sourceCreate(mixer, uid, argsJ);
			if (!newSource) {
                AFB_ReqFailF(request, "bad-pcm", "mixer=%s invalid source= %s", mixer->uid, json_object_get_string(argsJ));
				goto fail;
            }

            break;

        case json_type_array:
            count = json_object_array_length(argsJ);
			if (count > (mixer->max.sources - mixer->nbSources)) {
                AFB_ReqFailF(request, "too-small", "mixer=%s max source=%d", mixer->uid, mixer->max.sources);
				goto fail;
            }

            for (int idx = 0; idx < count; idx++) {
                json_object *sourceJ = json_object_array_get_idx(argsJ, idx);
				newSource = sourceCreate(mixer, uid, sourceJ);
				if (!newSource) {
                    AFB_ReqFailF(request, "bad-pcm", "mixer=%s invalid source= %s", mixer->uid, json_object_get_string(sourceJ));
					goto fail;
                }
            }

            break;
        default:
            AFB_ReqFailF(request, "invalid-syntax", "mixer=%s sources invalid argsJ= %s", mixer->uid, json_object_get_string(argsJ));
			goto fail;
    }

    return 0;

fail:
    return -1;
}
