/*
 * Copyright (C) 2018 "IoT.bzh"
 * Author Loïc Collignon <loic.collignon@iot.bzh>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "ahl-binding.hpp"

/**
 * @brief Callback invoked on new api creation.
 * @param[in] handle Handle to the new api.
 * @return Status code, zero if success.
 */
int ahl_api_create(void*, afb_api_t handle)
{
	return ahl_binding_t::instance().preinit(handle);
}

/**
 * @brief Entry point for API.
 * @param[in] handle Handle to start with for API creation.
 * @return Status code, zero if success.
 */
int afbBindingEntry(afb_api_t handle)
{
	using namespace std::placeholders;
	assert(handle != nullptr);

	afb_api_new_api(
		handle,
		HL_API_NAME,
		HL_API_INFO,
		1,
		ahl_api_create,
		nullptr
	);

	return 0;
}

/**
 * @brief Callback invoked when API enter the init phase.
 * @return Status code, zero if success.
 */
int ahl_api_init(afb_api_t)
{
	return ahl_binding_t::instance().init();
}

/**
 * @brief Callback invoked when an event is received.
 * @param[in] e Event's name.
 * @param[in] o Event's args.
 */
void ahl_api_on_event(afb_api_t, const char* e, struct json_object* o)
{
	ahl_binding_t::instance().event(e, o);
}

/**
 * @brief Callback invoked when a 'roles' section is found in config file.
 * @param[in] o Config section to handle.
 * @return Status code, zero if success.
 */
int ahl_api_config_roles(afb_api_t, CtlSectionT*, json_object* o)
{
	return ahl_binding_t::instance().parse_roles_config(o);
}

/**
 * @brief Callback invoked when clients call the verb 'get_roles'.
 * @param[in] req Request to handle.
 */
void ahl_api_get_roles(afb_req_t req)
{
	ahl_binding_t::instance().get_roles(req);
}

/**
 * @brief Callback invoked when clients call the verb 'activerole'.
 * @param[in] req Request to handle.
 */
void ahl_api_activerole(afb_req_t req)
{
	ahl_binding_t::instance().activerole(req);
}

/**
 * @brief Callback invoked when clients call the verb 'subscribe'.
 * @param[in] req Request to handle.
 */
void ahl_api_subscribe(afb_req_t req)
{
	ahl_binding_t::instance().subscribe(req);
}

/**
 * @brief Callback invoked when clients call the verb 'unsubscribe'.
 * @param[in] req Request to handle.
 */
void ahl_api_unsubscribe(afb_req_t req)
{
	ahl_binding_t::instance().unsubscribe(req);
}

/**
 * @brief Callback invoked when clients call a 'role' verb.
 * @param[in] req Request to handle.
 *
 * Handle dynamic verbs based on role name ('multimedia', 'navigation', ...)
 */
void ahl_api_role(afb_req_t req)
{
	role_t* role = (role_t*)req->vcbdata;
	assert(role != nullptr);

	role->invoke(req);
}
