/*
 * Copyright (C) 2018 "IoT.bzh"
 * Author Fulup Ar Foll <fulup@iot.bzh>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#define _GNU_SOURCE  // needed for vasprintf

#include "alsa-softmixer.h"


ALSA_PLUG_PROTO(rate);

static void rateConfigClean(SoftMixerT *mixer, void * arg) {
	snd_config_t * rateConfig = arg;
	AFB_API_DEBUG(mixer->api, "%s... rate config", __func__);
	snd_config_delete(rateConfig);
	snd_config_update();
}

PUBLIC AlsaPcmCtlT* AlsaCreateRate(SoftMixerT *mixer, AlsaStreamAudioT * stream, const char* pcmName, AlsaPcmCtlT *pcmSlave, AlsaPcmHwInfoT *params, int open) {

    snd_config_t *rateConfig, *slaveConfig, *elemConfig, *pcmConfig;

	AFB_ApiDebug(mixer->api,"%s : %s", __func__, pcmName);

	AlsaPcmCtlT *pcmPlug = AlsaPcmCtlNew(mixer, pcmName);
	if (!pcmPlug) {
		SOFTMIXER_NOMEM(mixer->api);
		goto OnErrorExit;
	}

    pcmPlug->cid.cardid = pcmName;

    int error = 0;

    // refresh global alsalib config and create PCM top config
    snd_config_update();
    error += snd_config_top(&rateConfig);
    error += snd_config_set_id(rateConfig, pcmPlug->cid.cardid);
    error += snd_config_imake_string(&elemConfig, "type", "rate");
    error += snd_config_add(rateConfig, elemConfig);
    if (error) goto OnErrorExit;

    error += snd_config_make_compound(&slaveConfig, "slave", 0);
    error += snd_config_imake_string(&elemConfig, "pcm", pcmSlave->cid.cardid);
    error += snd_config_add(slaveConfig, elemConfig);
    if (params->rate) {
        error += snd_config_imake_integer(&elemConfig, "rate", params->rate);
        error += snd_config_add(slaveConfig, elemConfig);
    }
    if (params->format) {
        error += snd_config_imake_string(&elemConfig, "format", params->formatString);
        error += snd_config_add(slaveConfig, elemConfig);
    }
    if (error) goto OnErrorExit;

    // add leaf into config
    error += snd_config_add(rateConfig, slaveConfig);
    if (error) goto OnErrorExit;

    error += snd_config_search(snd_config, "pcm", &pcmConfig);
    error += snd_config_add(pcmConfig, rateConfig);
    if (error) {
		AFB_ApiError(mixer->api, "%s: fail to add config RATE=%s", __func__, pcmPlug->cid.cardid);
        goto OnErrorExit;
    }

    if (open) error = _snd_pcm_rate_open(&pcmPlug->handle, pcmPlug->cid.cardid, snd_config, rateConfig, SND_PCM_STREAM_PLAYBACK, SND_PCM_NONBLOCK);
    if (error) {
		AFB_ApiError(mixer->api,
					 "%s: fail to create Rate=%s Slave=%s Error=%s",
					 __func__, pcmPlug->cid.cardid, pcmSlave->cid.cardid, snd_strerror(error));
        goto OnErrorExit;
    }

    pcmPlug->private_data = rateConfig;
    pcmPlug->private_data_clean = rateConfigClean;

    // Debug config & pcm
    //AlsaDumpCtlConfig(mixer, "plug-rate", pcmConfig, 1);
	AlsaDumpCtlConfig (mixer, "plug-rate", rateConfig, 1);
	AFB_ApiNotice(mixer->api, "%s: %s done", __func__, pcmPlug->cid.cardid);
    return pcmPlug;

OnErrorExit:
    AlsaDumpCtlConfig(mixer, "plug-rate", rateConfig, 1);
	AFB_ApiNotice(mixer->api, "%s: OnErrorExit", __func__);
    return NULL;
}
