/*
 * Copyright (C) 2018 "IoT.bzh"
 * Author Fulup Ar Foll <fulup@iot.bzh>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

#define _GNU_SOURCE  // needed for vasprintf

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#include "alsa-softmixer.h"

// extracted IOCTLs from <alsa/asoundlib.h>
#define _IOR_HACKED(type,nr,size)       _IOC(_IOC_READ,(type),(nr),size)
#define SNDRV_CTL_IOCTL_CARD_INFO(size) _IOR_HACKED('U', 0x01, size)


// Clone of AlsaLib snd_card_load2 static function

PUBLIC snd_ctl_card_info_t *AlsaByPathInfo(SoftMixerT *mixer, const char *devpath) {
    int open_dev;
	snd_ctl_card_info_t *cardInfo = (snd_ctl_card_info_t *) malloc(snd_ctl_card_info_sizeof());
	if (!devpath) {
		SOFTMIXER_NOMEM(mixer->api);
		goto fail;
	}

    open_dev = open(devpath, O_RDONLY);
	if (open_dev < 0) {
		AFB_ApiError(mixer->api, "%s: Unable to open %s", __func__, devpath);
		goto fail_cardinfo;
	}

    int rc = ioctl(open_dev, SNDRV_CTL_IOCTL_CARD_INFO(snd_ctl_card_info_sizeof()), cardInfo);
    if (rc < 0) {
		AFB_ApiError(mixer->api, "%s: ioctl on %s failed", __func__, devpath);
		goto fail_dev;
    }

    close(open_dev);
    return cardInfo;


fail_dev:
	close(open_dev);
fail_cardinfo:
	free(cardInfo);
fail:
    return NULL;
}

PUBLIC AlsaPcmCtlT * AlsaPcmCtlNew(SoftMixerT* mixer, const char * name) {

	AFB_ApiDebug(mixer->api, "%s: NEW %s",__func__, name);

	AlsaPcmCtlT *pcmCtl = calloc(1, sizeof (AlsaPcmCtlT));
	if (pcmCtl == NULL) {
		SOFTMIXER_NOMEM(mixer->api);
		goto fail;
	}

	pcmCtl->name = name;
	pcmCtl->closeAtDeletion = false;

	AlsaMixerTransactionObjectAdd(mixer->transaction, pcmCtl, AlsaPcmCtlDelete);

fail:
	return pcmCtl;
}

PUBLIC void AlsaPcmCtlDelete(SoftMixerT* mixer, void * arg) {
	int error = 0;

	AlsaPcmCtlT * pcmCtl = (AlsaPcmCtlT *) arg;

	AFB_ApiDebug(mixer->api, "%s of %s (plug: %d)", __func__, pcmCtl->name, pcmCtl->isPcmPlug);

	if (pcmCtl->private_data_clean)
		pcmCtl->private_data_clean(mixer, pcmCtl->private_data);

	if (!pcmCtl->closeAtDeletion)
		goto done;

	AFB_ApiDebug(mixer->api, "%s: Closing %s", __func__, pcmCtl->cid.cardid);
	error = snd_pcm_close(pcmCtl->handle);
	if (error) {
		// It's safe to ignore this error. In case of ioplug PCM, the device may have already disappeared
		AFB_ApiDebug(mixer->api,
				"%s: failed to close capture PCM %s: err=%s", __func__, pcmCtl->cid.cardid, snd_strerror(error));
	}

done:

	ApiPcmDelParams(mixer, pcmCtl->params);

	free((char*) pcmCtl->cid.cardid);
	free(pcmCtl);

	AFB_ApiDebug(mixer->api, "%s DONE", __func__);

}

PUBLIC AlsaPcmCtlT *AlsaByPathOpenPcmCtl(SoftMixerT *mixer, AlsaDevInfoT *pcmDev, snd_pcm_stream_t direction) {
    int error;
    char *cardid = NULL;

	AlsaPcmCtlT *pcmCtl = NULL;

    if (pcmDev->pcmplug_params) {
		pcmDev->cardid = strdup(pcmDev->pcmplug_params);
		if (pcmDev->cardid == NULL) {
			SOFTMIXER_NOMEM(mixer->api);
			goto fail;
		}
    }

    if (!pcmDev->cardid) {
        if (pcmDev->subdev) {
			if (asprintf(&cardid, "hw:%i,%i,%i", pcmDev->cardidx, pcmDev->device, pcmDev->subdev) == -1) {
				SOFTMIXER_NOMEM(mixer->api);
				goto fail;
			}
        }
        else if (pcmDev->device) {
			if (asprintf(&cardid, "hw:%i,%i", pcmDev->cardidx, pcmDev->device) == -1) {
				SOFTMIXER_NOMEM(mixer->api);
				goto fail;
			}
        }
        else {
			if (asprintf(&cardid, "hw:%i", pcmDev->cardidx) == -1) {
				SOFTMIXER_NOMEM(mixer->api);
				goto fail;
			}
        }

        pcmDev->cardid = (const char*)cardid;
    }

	pcmCtl = AlsaPcmCtlNew(mixer, pcmDev->cardid);
	if (pcmCtl == NULL) {
		SOFTMIXER_NOMEM(mixer->api);
		goto fail_cardid;
	}

    // inherit CID from pcmDev
    pcmCtl->cid.cardid = pcmDev->cardid;
    pcmCtl->cid.cardidx = pcmDev->cardidx;
    pcmCtl->cid.device = pcmDev->device;
    pcmCtl->cid.subdev = pcmDev->subdev;
    pcmCtl->cid.name=NULL;
    pcmCtl->cid.longname=NULL;

	if (pcmDev->pcmplug_params)
		pcmCtl->isPcmPlug = true;

	AFB_ApiDebug(mixer->api,
                "%s OPEN PCM '%s', direction %s",
                __func__, pcmDev->cardid, direction==SND_PCM_STREAM_PLAYBACK?"playback":"capture");

    error = snd_pcm_open(&pcmCtl->handle, pcmCtl->cid.cardid, direction, SND_PCM_NONBLOCK);
    if (error < 0) {
        AFB_ApiError(mixer->api, "%s: fail openpcm cardid=%s error=%s", __func__, pcmCtl->cid.cardid, snd_strerror(error));
		goto fail_pcmctl;
    }

    return pcmCtl;

fail_pcmctl:
    free(pcmCtl);
fail_cardid:
	free((char*)pcmDev->cardid);
fail:
    return NULL;
}

PUBLIC snd_ctl_t *AlsaByPathOpenCtl(SoftMixerT *mixer, const char *uid, AlsaSndCtlT *dev) {
    int err;
    snd_ctl_t *handle;

    // get card info from /dev/snd/xxx if not use hw:x,x,x
    snd_ctl_card_info_t *cardInfo = NULL;

	AFB_ApiDebug(mixer->api, "%s: devpath %s, cardid %s, plug %s", __func__, dev->cid.devpath, dev->cid.cardid, dev->cid.pcmplug_params);

    if (dev->cid.devpath)
        cardInfo = AlsaByPathInfo(mixer, dev->cid.devpath);
    else if (dev->cid.cardid)
        cardInfo = AlsaCtlGetCardInfo(mixer, dev->cid.cardid);
	else if (dev->cid.pcmplug_params) {
		AFB_ApiDebug(mixer->api, "Get card info from plug params %s", dev->cid.pcmplug_params);
    	cardInfo = AlsaCtlGetCardInfo(mixer, dev->cid.pcmplug_params);
	}

    if (!cardInfo) {
        AFB_ApiError(mixer->api,
                     "%s: uid=%s fail to find sndcard by path=%s id=%s",
                     __func__, uid, dev->cid.devpath, dev->cid.cardid);
		goto fail;
    }

    // extract useful info from cardInfo handle
    dev->cid.devpath = NULL;
    dev->cid.cardidx = snd_ctl_card_info_get_card(cardInfo);
    dev->cid.name = strdup(snd_ctl_card_info_get_name(cardInfo));
	if (dev->cid.name == NULL) {
		SOFTMIXER_NOMEM(mixer->api);
		goto fail;
	}
    dev->cid.longname = strdup(snd_ctl_card_info_get_longname(cardInfo));
	if (dev->cid.longname == NULL) {
		SOFTMIXER_NOMEM(mixer->api);
		goto fail_cid_name;
	}

    // build a valid name and open sndcard
	if (asprintf((char**) &dev->cid.cardid, "hw:%i", dev->cid.cardidx) == -1) {
		SOFTMIXER_NOMEM(mixer->api);
		goto fail_cid_longname;
	}

    if ((err = snd_ctl_open(&handle, dev->cid.cardid, 0)) < 0) {
        AFB_ApiError(mixer->api,
                     "%s uid=%s sndcard open fail cardid=%s longname=%s error=%s",
                     __func__, uid, dev->cid.cardid, dev->cid.longname, snd_strerror(err));
		goto fail_cardid;
    }

    AFB_ApiNotice(mixer->api,
				  "%s: uid=%s cardid=%s cardname=%s pcmplug %s",
				  __func__, uid, dev->cid.cardid, dev->cid.longname, dev->cid.pcmplug_params);
    free(cardInfo);
    return handle;

fail_cid_longname:
	free((char*)dev->cid.longname);
fail_cid_name:
	free((char*)dev->cid.name);
fail_cardid:
	free((char*)dev->cid.cardid);
fail:
    return NULL;
}
