/*
 * R-Car SYSC Power management support
 *
 * Copyright (C) 2014  Magnus Damm
 * Copyright (C) 2015-2017 Glider bvba
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file "COPYING" in the main directory of this archive
 * for more details.
 */

#include <dt-bindings/power/r8a7795-sysc.h>
#include <dt-bindings/power/r8a7796-sysc.h>
#include <dt-bindings/power/r8a77965-sysc.h>
#include <linux/clk/renesas.h>
#include <linux/delay.h>
#include <linux/err.h>
#include <linux/mm.h>
#include <linux/of_address.h>
#include <linux/pm_domain.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/io.h>
#include <linux/soc/renesas/rcar-sysc.h>
#include <linux/sys_soc.h>
#include <linux/syscore_ops.h>

#include "rcar-sysc.h"

/* SYSC Common */
#define SYSCSR			0x00	/* SYSC Status Register */
#define SYSCISR			0x04	/* Interrupt Status Register */
#define SYSCISCR		0x08	/* Interrupt Status Clear Register */
#define SYSCIER			0x0c	/* Interrupt Enable Register */
#define SYSCIMR			0x10	/* Interrupt Mask Register */

/* SYSC Status Register */
#define SYSCSR_PONENB		1	/* Ready for power resume requests */
#define SYSCSR_POFFENB		0	/* Ready for power shutoff requests */

/*
 * Power Control Register Offsets inside the register block for each domain
 * Note: The "CR" registers for ARM cores exist on H1 only
 *	 Use WFI to power off, CPG/APMU to resume ARM cores on R-Car Gen2
 *	 Use PSCI on R-Car Gen3
 */
#define PWRSR_OFFS		0x00	/* Power Status Register */
#define PWROFFCR_OFFS		0x04	/* Power Shutoff Control Register */
#define PWROFFSR_OFFS		0x08	/* Power Shutoff Status Register */
#define PWRONCR_OFFS		0x0c	/* Power Resume Control Register */
#define PWRONSR_OFFS		0x10	/* Power Resume Status Register */
#define PWRER_OFFS		0x14	/* Power Shutoff/Resume Error */


#define SYSCSR_RETRIES		1000
#define SYSCSR_DELAY_US		10

#define PWRER_RETRIES		1000
#define PWRER_DELAY_US		10

#define SYSCISR_RETRIES		1000
#define SYSCISR_DELAY_US	10

#define RCAR_PD_ALWAYS_ON	32	/* Always-on power area */

static
const struct soc_device_attribute rcar_sysc_quirks_match[] __initconst = {
	{
		.soc_id = "r8a7795", .revision = "ES2.0",
		.data = (void *)(BIT(R8A7795_PD_A3VP) | BIT(R8A7795_PD_CR7)
			| BIT(R8A7795_PD_A3VC) | BIT(R8A7795_PD_A2VC0)
			| BIT(R8A7795_PD_A2VC1) | BIT(R8A7795_PD_A3IR)
			| BIT(R8A7795_PD_3DG_A) | BIT(R8A7795_PD_3DG_B)
			| BIT(R8A7795_PD_3DG_C) | BIT(R8A7795_PD_3DG_D)
			| BIT(R8A7795_PD_3DG_E)),
	},
	{
		.soc_id = "r8a7795", .revision = "ES1.*",
		.data = (void *)(BIT(R8A7795_PD_A3VP) | BIT(R8A7795_PD_CR7)
			| BIT(R8A7795_PD_A3VC) | BIT(R8A7795_PD_A2VC0)
			| BIT(R8A7795_PD_A2VC1) | BIT(R8A7795_PD_A3IR)
			| BIT(R8A7795_PD_3DG_A) | BIT(R8A7795_PD_3DG_B)
			| BIT(R8A7795_PD_3DG_C) | BIT(R8A7795_PD_3DG_D)
			| BIT(R8A7795_PD_3DG_E)),

	},
	{
		.soc_id = "r8a7796", .revision = "ES1.*",
		.data = (void *)(BIT(R8A7796_PD_CR7) | BIT(R8A7796_PD_A3VC)
			| BIT(R8A7796_PD_A2VC0) | BIT(R8A7796_PD_A2VC1)
			| BIT(R8A7796_PD_A3IR) | BIT(R8A7796_PD_3DG_A)
			| BIT(R8A7796_PD_3DG_B)),
	},
	{
		.soc_id = "r8a77965", .revision = "ES1.0",
		.data = (void *)(BIT(R8A77965_PD_A3VP) | BIT(R8A77965_PD_CR7)
			| BIT(R8A77965_PD_A3VC) | BIT(R8A77965_PD_A2VC1)
			| BIT(R8A77965_PD_A3IR)),
	},
	{ /* sentinel */ }
};

static u32 rcar_sysc_quirks;

static void __iomem *rcar_sysc_base;
static DEFINE_SPINLOCK(rcar_sysc_lock); /* SMP CPUs + I/O devices */
static struct rcar_sysc_extra_regs *extra_regs;

static const char *to_pd_name(const struct rcar_sysc_ch *sysc_ch);

static void rcar_sysc_extmask_ctrl(bool on)
{
	u32 extmask_msks = 0;

	if (!extra_regs || !extra_regs->sysc_extmask_msks)
		return;

	if (on)
		extmask_msks = extra_regs->sysc_extmask_msks;

	iowrite32(extmask_msks,
		  rcar_sysc_base + extra_regs->sysc_extmask_offs);
}

static int rcar_sysc_pwr_on_off(const struct rcar_sysc_ch *sysc_ch, bool on)
{
	unsigned int sr_bit, reg_offs;
	int k;

	if (on) {
		sr_bit = SYSCSR_PONENB;
		reg_offs = PWRONCR_OFFS;
	} else {
		sr_bit = SYSCSR_POFFENB;
		reg_offs = PWROFFCR_OFFS;
	}

	/* Wait until SYSC is ready to accept a power request */
	for (k = 0; k < SYSCSR_RETRIES; k++) {
		if (ioread32(rcar_sysc_base + SYSCSR) & BIT(sr_bit))
			break;
		udelay(SYSCSR_DELAY_US);
	}

	if (k == SYSCSR_RETRIES)
		return -EAGAIN;

	/* Start W/A for A3VP, A3VC, and A3IR domains */
	if (!on && (!strcmp("a3vp", to_pd_name(sysc_ch))
		|| !strcmp("a3ir", to_pd_name(sysc_ch))
		|| !strcmp("a3vc", to_pd_name(sysc_ch))))
		udelay(1);

	/* Submit power shutoff or power resume request */
	iowrite32(BIT(sysc_ch->chan_bit),
		  rcar_sysc_base + sysc_ch->chan_offs + reg_offs);

	return 0;
}

static int rcar_sysc_power(const struct rcar_sysc_ch *sysc_ch, bool on)
{
	unsigned int isr_mask = BIT(sysc_ch->isr_bit);
	unsigned int chan_mask = BIT(sysc_ch->chan_bit);
	unsigned int status;
	unsigned long flags;
	int ret = 0;
	int k;

	spin_lock_irqsave(&rcar_sysc_lock, flags);

	rcar_sysc_extmask_ctrl(1); /* set EXTMSK0 */
	iowrite32(isr_mask, rcar_sysc_base + SYSCISCR);

	/* Submit power shutoff or resume request until it was accepted */
	for (k = 0; k < PWRER_RETRIES; k++) {
		ret = rcar_sysc_pwr_on_off(sysc_ch, on);
		if (ret)
			goto out;

		status = ioread32(rcar_sysc_base +
				  sysc_ch->chan_offs + PWRER_OFFS);
		if (!(status & chan_mask))
			break;

		udelay(PWRER_DELAY_US);
	}

	if (k == PWRER_RETRIES) {
		ret = -EIO;
		goto out;
	}

	/* Wait until the power shutoff or resume request has completed * */
	for (k = 0; k < SYSCISR_RETRIES; k++) {
		if (ioread32(rcar_sysc_base + SYSCISR) & isr_mask)
			break;
		udelay(SYSCISR_DELAY_US);
	}

	if (k == SYSCISR_RETRIES)
		ret = -EIO;

	iowrite32(isr_mask, rcar_sysc_base + SYSCISCR);

 out:
	rcar_sysc_extmask_ctrl(0); /* clear EXTMSK0 */
	spin_unlock_irqrestore(&rcar_sysc_lock, flags);

	pr_debug("sysc power %s domain %d: %08x -> %d\n", on ? "on" : "off",
		 sysc_ch->isr_bit, ioread32(rcar_sysc_base + SYSCISR), ret);
	return ret;
}

int rcar_sysc_power_down(const struct rcar_sysc_ch *sysc_ch)
{
	return rcar_sysc_power(sysc_ch, false);
}

int rcar_sysc_power_up(const struct rcar_sysc_ch *sysc_ch)
{
	return rcar_sysc_power(sysc_ch, true);
}

static bool rcar_sysc_power_is_off(const struct rcar_sysc_ch *sysc_ch)
{
	unsigned int st;

	st = ioread32(rcar_sysc_base + sysc_ch->chan_offs + PWRSR_OFFS);
	if (st & BIT(sysc_ch->chan_bit))
		return true;

	return false;
}

struct rcar_sysc_pd {
	struct generic_pm_domain genpd;
	struct rcar_sysc_ch ch;
	unsigned int flags;
	char name[0];
};

static inline struct rcar_sysc_pd *to_rcar_pd(struct generic_pm_domain *d)
{
	return container_of(d, struct rcar_sysc_pd, genpd);
}

static inline const char *to_pd_name(const struct rcar_sysc_ch *sysc_ch)
{
	return container_of(sysc_ch, struct rcar_sysc_pd, ch)->genpd.name;
}

static int rcar_sysc_pd_power_off(struct generic_pm_domain *genpd)
{
	struct rcar_sysc_pd *pd = to_rcar_pd(genpd);

	pr_debug("%s: %s\n", __func__, genpd->name);
	return rcar_sysc_power_down(&pd->ch);
}

static int rcar_sysc_pd_power_on(struct generic_pm_domain *genpd)
{
	struct rcar_sysc_pd *pd = to_rcar_pd(genpd);

	pr_debug("%s: %s\n", __func__, genpd->name);
	return rcar_sysc_power_up(&pd->ch);
}

static bool has_cpg_mstp;

static int __init rcar_sysc_pd_setup(struct rcar_sysc_pd *pd)
{
	struct generic_pm_domain *genpd = &pd->genpd;
	const char *name = pd->genpd.name;
	struct dev_power_governor *gov = &simple_qos_governor;
	int error;

	if (pd->flags & PD_CPU) {
		/*
		 * This domain contains a CPU core and therefore it should
		 * only be turned off if the CPU is not in use.
		 */
		pr_debug("PM domain %s contains %s\n", name, "CPU");
		genpd->flags |= GENPD_FLAG_ALWAYS_ON;
	} else if (pd->flags & PD_SCU) {
		/*
		 * This domain contains an SCU and cache-controller, and
		 * therefore it should only be turned off if the CPU cores are
		 * not in use.
		 */
		pr_debug("PM domain %s contains %s\n", name, "SCU");
		genpd->flags |= GENPD_FLAG_ALWAYS_ON;
	} else if (pd->flags & PD_NO_CR) {
		/*
		 * This domain cannot be turned off.
		 */
		genpd->flags |= GENPD_FLAG_ALWAYS_ON;
	}

	if (!(pd->flags & (PD_CPU | PD_SCU))) {
		/* Enable Clock Domain for I/O devices */
		genpd->flags |= GENPD_FLAG_PM_CLK | GENPD_FLAG_ACTIVE_WAKEUP;
		if (has_cpg_mstp) {
			genpd->attach_dev = cpg_mstp_attach_dev;
			genpd->detach_dev = cpg_mstp_detach_dev;
		} else {
			genpd->attach_dev = cpg_mssr_attach_dev;
			genpd->detach_dev = cpg_mssr_detach_dev;
		}
	}

	genpd->power_off = rcar_sysc_pd_power_off;
	genpd->power_on = rcar_sysc_pd_power_on;

	if (pd->flags & (PD_CPU | PD_NO_CR)) {
		/* Skip CPUs (handled by SMP code) and areas without control */
		pr_debug("%s: Not touching %s\n", __func__, genpd->name);
		goto finalize;
	}

	if (!rcar_sysc_power_is_off(&pd->ch)) {
		pr_debug("%s: %s is already powered\n", __func__, genpd->name);
		goto finalize;
	}

	rcar_sysc_power_up(&pd->ch);

finalize:
	error = pm_genpd_init(genpd, gov, false);
	if (error)
		pr_err("Failed to init PM domain %s: %d\n", name, error);

	return error;
}

struct rcar_sysc_pd *rcar_domains[RCAR_PD_ALWAYS_ON + 1];

static void rcar_power_on_force(void)
{
	int i;

	for (i = 0; i < RCAR_PD_ALWAYS_ON; i++) {
		struct rcar_sysc_pd *pd = rcar_domains[i];

		if (!pd)
			continue;

		if (rcar_sysc_quirks & BIT(pd->ch.isr_bit)) {
			if (!rcar_sysc_power_is_off(&pd->ch))
				continue;

			rcar_sysc_power_up(&pd->ch);
		}
	}
}

static u32 syscier_val, syscimr_val;
#ifdef CONFIG_PM_SLEEP
static void rcar_sysc_resume(void)
{
	pr_debug("%s\n", __func__);

	/* Re-enable interrupts as init */
	iowrite32(syscimr_val, rcar_sysc_base + SYSCIMR);
	iowrite32(syscier_val, rcar_sysc_base + SYSCIER);

	rcar_power_on_force();
}

static struct syscore_ops rcar_sysc_syscore_ops = {
	.resume = rcar_sysc_resume,
};
#endif

static const struct of_device_id rcar_sysc_matches[] __initconst = {
#ifdef CONFIG_SYSC_R8A7743
	{ .compatible = "renesas,r8a7743-sysc", .data = &r8a7743_sysc_info },
#endif
#ifdef CONFIG_SYSC_R8A7745
	{ .compatible = "renesas,r8a7745-sysc", .data = &r8a7745_sysc_info },
#endif
#ifdef CONFIG_SYSC_R8A77470
	{ .compatible = "renesas,r8a77470-sysc", .data = &r8a77470_sysc_info },
#endif
#ifdef CONFIG_SYSC_R8A7779
	{ .compatible = "renesas,r8a7779-sysc", .data = &r8a7779_sysc_info },
#endif
#ifdef CONFIG_SYSC_R8A7790
	{ .compatible = "renesas,r8a7790-sysc", .data = &r8a7790_sysc_info },
#endif
#ifdef CONFIG_SYSC_R8A7791
	{ .compatible = "renesas,r8a7791-sysc", .data = &r8a7791_sysc_info },
	/* R-Car M2-N is identical to R-Car M2-W w.r.t. power domains. */
	{ .compatible = "renesas,r8a7793-sysc", .data = &r8a7791_sysc_info },
#endif
#ifdef CONFIG_SYSC_R8A7792
	{ .compatible = "renesas,r8a7792-sysc", .data = &r8a7792_sysc_info },
#endif
#ifdef CONFIG_SYSC_R8A7794
	{ .compatible = "renesas,r8a7794-sysc", .data = &r8a7794_sysc_info },
#endif
#ifdef CONFIG_SYSC_R8A7795
	{ .compatible = "renesas,r8a7795-sysc", .data = &r8a7795_sysc_info },
#endif
#ifdef CONFIG_SYSC_R8A7796
	{ .compatible = "renesas,r8a7796-sysc", .data = &r8a7796_sysc_info },
#endif
#ifdef CONFIG_SYSC_R8A77965
	{ .compatible = "renesas,r8a77965-sysc", .data = &r8a77965_sysc_info },
#endif
#ifdef CONFIG_SYSC_R8A77970
	{ .compatible = "renesas,r8a77970-sysc", .data = &r8a77970_sysc_info },
#endif
#ifdef CONFIG_SYSC_R8A77980
	{ .compatible = "renesas,r8a77980-sysc", .data = &r8a77980_sysc_info },
#endif
#ifdef CONFIG_SYSC_R8A77990
	{ .compatible = "renesas,r8a77990-sysc", .data = &r8a77990_sysc_info },
#endif
#ifdef CONFIG_SYSC_R8A77995
	{ .compatible = "renesas,r8a77995-sysc", .data = &r8a77995_sysc_info },
#endif
	{ /* sentinel */ }
};

struct rcar_pm_domains {
	struct genpd_onecell_data onecell_data;
	struct generic_pm_domain *domains[RCAR_PD_ALWAYS_ON + 1];
};

static int __init rcar_sysc_pd_init(void)
{
	const struct rcar_sysc_info *info;
	const struct of_device_id *match;
	struct rcar_pm_domains *domains;
	struct device_node *np;
	u32 syscier, syscimr;
	void __iomem *base;
	unsigned int i;
	int error;
	const struct soc_device_attribute *attr;

	if (rcar_sysc_base)
		return 0;

	np = of_find_matching_node_and_match(NULL, rcar_sysc_matches, &match);
	if (!np)
		return -ENODEV;

	info = match->data;

	if (info->init) {
		error = info->init();
		if (error)
			return error;
	}

	has_cpg_mstp = of_find_compatible_node(NULL, NULL,
					       "renesas,cpg-mstp-clocks");

	attr = soc_device_match(rcar_sysc_quirks_match);
	if (attr)
		rcar_sysc_quirks = (uintptr_t)attr->data;

	base = of_iomap(np, 0);
	if (!base) {
		pr_warn("%pOF: Cannot map regs\n", np);
		error = -ENOMEM;
		goto out_put;
	}

	rcar_sysc_base = base;

	/* Check and get extra registers */
	if (info->extra_regs)
		extra_regs = info->extra_regs;

	domains = kzalloc(sizeof(*domains), GFP_KERNEL);
	if (!domains) {
		error = -ENOMEM;
		goto out_put;
	}

	domains->onecell_data.domains = domains->domains;
	domains->onecell_data.num_domains = ARRAY_SIZE(domains->domains);

	for (i = 0, syscier = 0; i < info->num_areas; i++)
		syscier |= BIT(info->areas[i].isr_bit);

	/*
	 * Mask all interrupt sources to prevent the CPU from receiving them.
	 * Make sure not to clear reserved bits that were set before.
	 */
	syscimr = ioread32(base + SYSCIMR);
	syscimr |= syscier;
	pr_debug("%pOF: syscimr = 0x%08x\n", np, syscimr);
	iowrite32(syscimr, base + SYSCIMR);

	/*
	 * SYSC needs all interrupt sources enabled to control power.
	 */
	pr_debug("%pOF: syscier = 0x%08x\n", np, syscier);
	iowrite32(syscier, base + SYSCIER);

	syscier_val = syscier;
	syscimr_val = syscimr;

	/*
	 * First, create all PM domains
	 */
	for (i = 0; i < info->num_areas; i++) {
		const struct rcar_sysc_area *area = &info->areas[i];
		struct rcar_sysc_pd *pd;

		if (!area->name) {
			/* Skip NULLified area */
			continue;
		}

		pd = kzalloc(sizeof(*pd) + strlen(area->name) + 1, GFP_KERNEL);
		if (!pd) {
			error = -ENOMEM;
			goto out_put;
		}

		strcpy(pd->name, area->name);
		pd->genpd.name = pd->name;
		pd->ch.chan_offs = area->chan_offs;
		pd->ch.chan_bit = area->chan_bit;
		pd->ch.isr_bit = area->isr_bit;
		pd->flags = area->flags;

		if (rcar_sysc_quirks & BIT(pd->ch.isr_bit))
			pd->flags |= PD_NO_CR;

		error = rcar_sysc_pd_setup(pd);
		if (error)
			goto out_put;

		domains->domains[area->isr_bit] = &pd->genpd;
		rcar_domains[i] = pd;
	}

	/*
	 * Second, link all PM domains to their parents
	 */
	for (i = 0; i < info->num_areas; i++) {
		const struct rcar_sysc_area *area = &info->areas[i];

		if (!area->name || area->parent < 0)
			continue;

		error = pm_genpd_add_subdomain(domains->domains[area->parent],
					       domains->domains[area->isr_bit]);
		if (error)
			pr_warn("Failed to add PM subdomain %s to parent %u\n",
				area->name, area->parent);
	}

	rcar_power_on_force();

	error = of_genpd_add_provider_onecell(np, &domains->onecell_data);

#ifdef CONFIG_PM_SLEEP
	if (!error)
		register_syscore_ops(&rcar_sysc_syscore_ops);
#endif

out_put:
	of_node_put(np);
	return error;
}
early_initcall(rcar_sysc_pd_init);

void __init rcar_sysc_nullify(struct rcar_sysc_area *areas,
			      unsigned int num_areas, u8 id)
{
	unsigned int i;

	for (i = 0; i < num_areas; i++)
		if (areas[i].isr_bit == id) {
			areas[i].name = NULL;
			return;
		}
}

void __init rcar_sysc_init(phys_addr_t base, u32 syscier)
{
	u32 syscimr;

	if (!rcar_sysc_pd_init())
		return;

	rcar_sysc_base = ioremap_nocache(base, PAGE_SIZE);

	/*
	 * Mask all interrupt sources to prevent the CPU from receiving them.
	 * Make sure not to clear reserved bits that were set before.
	 */
	syscimr = ioread32(rcar_sysc_base + SYSCIMR);
	syscimr |= syscier;
	pr_debug("%s: syscimr = 0x%08x\n", __func__, syscimr);
	iowrite32(syscimr, rcar_sysc_base + SYSCIMR);

	/*
	 * SYSC needs all interrupt sources enabled to control power.
	 */
	pr_debug("%s: syscier = 0x%08x\n", __func__, syscier);
	iowrite32(syscier, rcar_sysc_base + SYSCIER);
}
