// Copyright 2008 John Maddock
//
// Use, modification and distribution are subject to the
// Boost Software License, Version 1.0.
// (See accompanying file LICENSE_1_0.txt
// or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef BOOST_MATH_SF_PRIME_HPP
#define BOOST_MATH_SF_PRIME_HPP

#include <boost/cstdint.hpp>
#include <boost/math/policies/error_handling.hpp>
#include <boost/math/special_functions/math_fwd.hpp>
#ifdef BOOST_MATH_HAVE_CONSTEXPR_TABLES
#include <array>
#else
#include <boost/array.hpp>
#endif

namespace boost{ namespace math{

   template <class Policy>
   BOOST_MATH_CONSTEXPR_TABLE_FUNCTION boost::uint32_t prime(unsigned n, const Policy& pol)
   {
      //
      // This is basically three big tables which together
      // occupy 19946 bytes, we use the smallest type which
      // will handle each value, and store the final set of 
      // values in a uint16_t with the values offset by 0xffff.
      // That gives us the first 10000 primes with the largest
      // being 104729:
      //
#ifdef BOOST_MATH_HAVE_CONSTEXPR_TABLES
      constexpr unsigned b1 = 53;
      constexpr unsigned b2 = 6541;
      constexpr unsigned b3 = 10000;
      constexpr std::array<unsigned char, 54> a1 = {{
#else
      static const unsigned b1 = 53;
      static const unsigned b2 = 6541;
      static const unsigned b3 = 10000;
      static const boost::array<unsigned char, 54> a1 = {{
#endif
         2u, 3u, 5u, 7u, 11u, 13u, 17u, 19u, 23u, 29u, 31u, 
         37u, 41u, 43u, 47u, 53u, 59u, 61u, 67u, 71u, 73u, 
         79u, 83u, 89u, 97u, 101u, 103u, 107u, 109u, 113u, 
         127u, 131u, 137u, 139u, 149u, 151u, 157u, 163u, 
         167u, 173u, 179u, 181u, 191u, 193u, 197u, 199u, 
         211u, 223u, 227u, 229u, 233u, 239u, 241u, 251u
      }};
#ifdef BOOST_MATH_HAVE_CONSTEXPR_TABLES
      constexpr std::array<boost::uint16_t, 6488> a2 = {{
#else      
      static const boost::array<boost::uint16_t, 6488> a2 = {{
#endif
         257u, 263u, 269u, 271u, 277u, 281u, 283u, 293u, 
         307u, 311u, 313u, 317u, 331u, 337u, 347u, 349u, 353u, 
         359u, 367u, 373u, 379u, 383u, 389u, 397u, 401u, 409u, 
         419u, 421u, 431u, 433u, 439u, 443u, 449u, 457u, 461u, 
         463u, 467u, 479u, 487u, 491u, 499u, 503u, 509u, 521u, 
         523u, 541u, 547u, 557u, 563u, 569u, 571u, 577u, 587u, 
         593u, 599u, 601u, 607u, 613u, 617u, 619u, 631u, 641u, 
         643u, 647u, 653u, 659u, 661u, 673u, 677u, 683u, 691u, 
         701u, 709u, 719u, 727u, 733u, 739u, 743u, 751u, 757u, 
         761u, 769u, 773u, 787u, 797u, 809u, 811u, 821u, 823u, 
         827u, 829u, 839u, 853u, 857u, 859u, 863u, 877u, 881u, 
         883u, 887u, 907u, 911u, 919u, 929u, 937u, 941u, 947u, 
         953u, 967u, 971u, 977u, 983u, 991u, 997u, 1009u, 1013u, 
         1019u, 1021u, 1031u, 1033u, 1039u, 1049u, 1051u, 1061u, 1063u, 
         1069u, 1087u, 1091u, 1093u, 1097u, 1103u, 1109u, 1117u, 1123u, 
         1129u, 1151u, 1153u, 1163u, 1171u, 1181u, 1187u, 1193u, 1201u, 
         1213u, 1217u, 1223u, 1229u, 1231u, 1237u, 1249u, 1259u, 1277u, 
         1279u, 1283u, 1289u, 1291u, 1297u, 1301u, 1303u, 1307u, 1319u, 
         1321u, 1327u, 1361u, 1367u, 1373u, 1381u, 1399u, 1409u, 1423u, 
         1427u, 1429u, 1433u, 1439u, 1447u, 1451u, 1453u, 1459u, 1471u, 
         1481u, 1483u, 1487u, 1489u, 1493u, 1499u, 1511u, 1523u, 1531u, 
         1543u, 1549u, 1553u, 1559u, 1567u, 1571u, 1579u, 1583u, 1597u, 
         1601u, 1607u, 1609u, 1613u, 1619u, 1621u, 1627u, 1637u, 1657u, 
         1663u, 1667u, 1669u, 1693u, 1697u, 1699u, 1709u, 1721u, 1723u, 
         1733u, 1741u, 1747u, 1753u, 1759u, 1777u, 1783u, 1787u, 1789u, 
         1801u, 1811u, 1823u, 1831u, 1847u, 1861u, 1867u, 1871u, 1873u, 
         1877u, 1879u, 1889u, 1901u, 1907u, 1913u, 1931u, 1933u, 1949u, 
         1951u, 1973u, 1979u, 1987u, 1993u, 1997u, 1999u, 2003u, 2011u, 
         2017u, 2027u, 2029u, 2039u, 2053u, 2063u, 2069u, 2081u, 2083u, 
         2087u, 2089u, 2099u, 2111u, 2113u, 2129u, 2131u, 2137u, 2141u, 
         2143u, 2153u, 2161u, 2179u, 2203u, 2207u, 2213u, 2221u, 2237u, 
         2239u, 2243u, 2251u, 2267u, 2269u, 2273u, 2281u, 2287u, 2293u, 
         2297u, 2309u, 2311u, 2333u, 2339u, 2341u, 2347u, 2351u, 2357u, 
         2371u, 2377u, 2381u, 2383u, 2389u, 2393u, 2399u, 2411u, 2417u, 
         2423u, 2437u, 2441u, 2447u, 2459u, 2467u, 2473u, 2477u, 2503u, 
         2521u, 2531u, 2539u, 2543u, 2549u, 2551u, 2557u, 2579u, 2591u, 
         2593u, 2609u, 2617u, 2621u, 2633u, 2647u, 2657u, 2659u, 2663u, 
         2671u, 2677u, 2683u, 2687u, 2689u, 2693u, 2699u, 2707u, 2711u, 
         2713u, 2719u, 2729u, 2731u, 2741u, 2749u, 2753u, 2767u, 2777u, 
         2789u, 2791u, 2797u, 2801u, 2803u, 2819u, 2833u, 2837u, 2843u, 
         2851u, 2857u, 2861u, 2879u, 2887u, 2897u, 2903u, 2909u, 2917u, 
         2927u, 2939u, 2953u, 2957u, 2963u, 2969u, 2971u, 2999u, 3001u, 
         3011u, 3019u, 3023u, 3037u, 3041u, 3049u, 3061u, 3067u, 3079u, 
         3083u, 3089u, 3109u, 3119u, 3121u, 3137u, 3163u, 3167u, 3169u, 
         3181u, 3187u, 3191u, 3203u, 3209u, 3217u, 3221u, 3229u, 3251u, 
         3253u, 3257u, 3259u, 3271u, 3299u, 3301u, 3307u, 3313u, 3319u, 
         3323u, 3329u, 3331u, 3343u, 3347u, 3359u, 3361u, 3371u, 3373u, 
         3389u, 3391u, 3407u, 3413u, 3433u, 3449u, 3457u, 3461u, 3463u, 
         3467u, 3469u, 3491u, 3499u, 3511u, 3517u, 3527u, 3529u, 3533u, 
         3539u, 3541u, 3547u, 3557u, 3559u, 3571u, 3581u, 3583u, 3593u, 
         3607u, 3613u, 3617u, 3623u, 3631u, 3637u, 3643u, 3659u, 3671u, 
         3673u, 3677u, 3691u, 3697u, 3701u, 3709u, 3719u, 3727u, 3733u, 
         3739u, 3761u, 3767u, 3769u, 3779u, 3793u, 3797u, 3803u, 3821u, 
         3823u, 3833u, 3847u, 3851u, 3853u, 3863u, 3877u, 3881u, 3889u, 
         3907u, 3911u, 3917u, 3919u, 3923u, 3929u, 3931u, 3943u, 3947u, 
         3967u, 3989u, 4001u, 4003u, 4007u, 4013u, 4019u, 4021u, 4027u, 
         4049u, 4051u, 4057u, 4073u, 4079u, 4091u, 4093u, 4099u, 4111u, 
         4127u, 4129u, 4133u, 4139u, 4153u, 4157u, 4159u, 4177u, 4201u, 
         4211u, 4217u, 4219u, 4229u, 4231u, 4241u, 4243u, 4253u, 4259u, 
         4261u, 4271u, 4273u, 4283u, 4289u, 4297u, 4327u, 4337u, 4339u, 
         4349u, 4357u, 4363u, 4373u, 4391u, 4397u, 4409u, 4421u, 4423u, 
         4441u, 4447u, 4451u, 4457u, 4463u, 4481u, 4483u, 4493u, 4507u, 
         4513u, 4517u, 4519u, 4523u, 4547u, 4549u, 4561u, 4567u, 4583u, 
         4591u, 4597u, 4603u, 4621u, 4637u, 4639u, 4643u, 4649u, 4651u, 
         4657u, 4663u, 4673u, 4679u, 4691u, 4703u, 4721u, 4723u, 4729u, 
         4733u, 4751u, 4759u, 4783u, 4787u, 4789u, 4793u, 4799u, 4801u, 
         4813u, 4817u, 4831u, 4861u, 4871u, 4877u, 4889u, 4903u, 4909u, 
         4919u, 4931u, 4933u, 4937u, 4943u, 4951u, 4957u, 4967u, 4969u, 
         4973u, 4987u, 4993u, 4999u, 5003u, 5009u, 5011u, 5021u, 5023u, 
         5039u, 5051u, 5059u, 5077u, 5081u, 5087u, 5099u, 5101u, 5107u, 
         5113u, 5119u, 5147u, 5153u, 5167u, 5171u, 5179u, 5189u, 5197u, 
         5209u, 5227u, 5231u, 5233u, 5237u, 5261u, 5273u, 5279u, 5281u, 
         5297u, 5303u, 5309u, 5323u, 5333u, 5347u, 5351u, 5381u, 5387u, 
         5393u, 5399u, 5407u, 5413u, 5417u, 5419u, 5431u, 5437u, 5441u, 
         5443u, 5449u, 5471u, 5477u, 5479u, 5483u, 5501u, 5503u, 5507u, 
         5519u, 5521u, 5527u, 5531u, 5557u, 5563u, 5569u, 5573u, 5581u, 
         5591u, 5623u, 5639u, 5641u, 5647u, 5651u, 5653u, 5657u, 5659u, 
         5669u, 5683u, 5689u, 5693u, 5701u, 5711u, 5717u, 5737u, 5741u, 
         5743u, 5749u, 5779u, 5783u, 5791u, 5801u, 5807u, 5813u, 5821u, 
         5827u, 5839u, 5843u, 5849u, 5851u, 5857u, 5861u, 5867u, 5869u, 
         5879u, 5881u, 5897u, 5903u, 5923u, 5927u, 5939u, 5953u, 5981u, 
         5987u, 6007u, 6011u, 6029u, 6037u, 6043u, 6047u, 6053u, 6067u, 
         6073u, 6079u, 6089u, 6091u, 6101u, 6113u, 6121u, 6131u, 6133u, 
         6143u, 6151u, 6163u, 6173u, 6197u, 6199u, 6203u, 6211u, 6217u, 
         6221u, 6229u, 6247u, 6257u, 6263u, 6269u, 6271u, 6277u, 6287u, 
         6299u, 6301u, 6311u, 6317u, 6323u, 6329u, 6337u, 6343u, 6353u, 
         6359u, 6361u, 6367u, 6373u, 6379u, 6389u, 6397u, 6421u, 6427u, 
         6449u, 6451u, 6469u, 6473u, 6481u, 6491u, 6521u, 6529u, 6547u, 
         6551u, 6553u, 6563u, 6569u, 6571u, 6577u, 6581u, 6599u, 6607u, 
         6619u, 6637u, 6653u, 6659u, 6661u, 6673u, 6679u, 6689u, 6691u, 
         6701u, 6703u, 6709u, 6719u, 6733u, 6737u, 6761u, 6763u, 6779u, 
         6781u, 6791u, 6793u, 6803u, 6823u, 6827u, 6829u, 6833u, 6841u, 
         6857u, 6863u, 6869u, 6871u, 6883u, 6899u, 6907u, 6911u, 6917u, 
         6947u, 6949u, 6959u, 6961u, 6967u, 6971u, 6977u, 6983u, 6991u, 
         6997u, 7001u, 7013u, 7019u, 7027u, 7039u, 7043u, 7057u, 7069u, 
         7079u, 7103u, 7109u, 7121u, 7127u, 7129u, 7151u, 7159u, 7177u, 
         7187u, 7193u, 7207u, 7211u, 7213u, 7219u, 7229u, 7237u, 7243u, 
         7247u, 7253u, 7283u, 7297u, 7307u, 7309u, 7321u, 7331u, 7333u, 
         7349u, 7351u, 7369u, 7393u, 7411u, 7417u, 7433u, 7451u, 7457u, 
         7459u, 7477u, 7481u, 7487u, 7489u, 7499u, 7507u, 7517u, 7523u, 
         7529u, 7537u, 7541u, 7547u, 7549u, 7559u, 7561u, 7573u, 7577u, 
         7583u, 7589u, 7591u, 7603u, 7607u, 7621u, 7639u, 7643u, 7649u, 
         7669u, 7673u, 7681u, 7687u, 7691u, 7699u, 7703u, 7717u, 7723u, 
         7727u, 7741u, 7753u, 7757u, 7759u, 7789u, 7793u, 7817u, 7823u, 
         7829u, 7841u, 7853u, 7867u, 7873u, 7877u, 7879u, 7883u, 7901u, 
         7907u, 7919u, 7927u, 7933u, 7937u, 7949u, 7951u, 7963u, 7993u, 
         8009u, 8011u, 8017u, 8039u, 8053u, 8059u, 8069u, 8081u, 8087u, 
         8089u, 8093u, 8101u, 8111u, 8117u, 8123u, 8147u, 8161u, 8167u, 
         8171u, 8179u, 8191u, 8209u, 8219u, 8221u, 8231u, 8233u, 8237u, 
         8243u, 8263u, 8269u, 8273u, 8287u, 8291u, 8293u, 8297u, 8311u, 
         8317u, 8329u, 8353u, 8363u, 8369u, 8377u, 8387u, 8389u, 8419u, 
         8423u, 8429u, 8431u, 8443u, 8447u, 8461u, 8467u, 8501u, 8513u, 
         8521u, 8527u, 8537u, 8539u, 8543u, 8563u, 8573u, 8581u, 8597u, 
         8599u, 8609u, 8623u, 8627u, 8629u, 8641u, 8647u, 8663u, 8669u, 
         8677u, 8681u, 8689u, 8693u, 8699u, 8707u, 8713u, 8719u, 8731u, 
         8737u, 8741u, 8747u, 8753u, 8761u, 8779u, 8783u, 8803u, 8807u, 
         8819u, 8821u, 8831u, 8837u, 8839u, 8849u, 8861u, 8863u, 8867u, 
         8887u, 8893u, 8923u, 8929u, 8933u, 8941u, 8951u, 8963u, 8969u, 
         8971u, 8999u, 9001u, 9007u, 9011u, 9013u, 9029u, 9041u, 9043u, 
         9049u, 9059u, 9067u, 9091u, 9103u, 9109u, 9127u, 9133u, 9137u, 
         9151u, 9157u, 9161u, 9173u, 9181u, 9187u, 9199u, 9203u, 9209u, 
         9221u, 9227u, 9239u, 9241u, 9257u, 9277u, 9281u, 9283u, 9293u, 
         9311u, 9319u, 9323u, 9337u, 9341u, 9343u, 9349u, 9371u, 9377u, 
         9391u, 9397u, 9403u, 9413u, 9419u, 9421u, 9431u, 9433u, 9437u, 
         9439u, 9461u, 9463u, 9467u, 9473u, 9479u, 9491u, 9497u, 9511u, 
         9521u, 9533u, 9539u, 9547u, 9551u, 9587u, 9601u, 9613u, 9619u, 
         9623u, 9629u, 9631u, 9643u, 9649u, 9661u, 9677u, 9679u, 9689u, 
         9697u, 9719u, 9721u, 9733u, 9739u, 9743u, 9749u, 9767u, 9769u, 
         9781u, 9787u, 9791u, 9803u, 9811u, 9817u, 9829u, 9833u, 9839u, 
         9851u, 9857u, 9859u, 9871u, 9883u, 9887u, 9901u, 9907u, 9923u, 
         9929u, 9931u, 9941u, 9949u, 9967u, 9973u, 10007u, 10009u, 10037u, 
         10039u, 10061u, 10067u, 10069u, 10079u, 10091u, 10093u, 10099u, 10103u, 
         10111u, 10133u, 10139u, 10141u, 10151u, 10159u, 10163u, 10169u, 10177u, 
         10181u, 10193u, 10211u, 10223u, 10243u, 10247u, 10253u, 10259u, 10267u, 
         10271u, 10273u, 10289u, 10301u, 10303u, 10313u, 10321u, 10331u, 10333u, 
         10337u, 10343u, 10357u, 10369u, 10391u, 10399u, 10427u, 10429u, 10433u, 
         10453u, 10457u, 10459u, 10463u, 10477u, 10487u, 10499u, 10501u, 10513u, 
         10529u, 10531u, 10559u, 10567u, 10589u, 10597u, 10601u, 10607u, 10613u, 
         10627u, 10631u, 10639u, 10651u, 10657u, 10663u, 10667u, 10687u, 10691u, 
         10709u, 10711u, 10723u, 10729u, 10733u, 10739u, 10753u, 10771u, 10781u, 
         10789u, 10799u, 10831u, 10837u, 10847u, 10853u, 10859u, 10861u, 10867u, 
         10883u, 10889u, 10891u, 10903u, 10909u, 10937u, 10939u, 10949u, 10957u, 
         10973u, 10979u, 10987u, 10993u, 11003u, 11027u, 11047u, 11057u, 11059u, 
         11069u, 11071u, 11083u, 11087u, 11093u, 11113u, 11117u, 11119u, 11131u, 
         11149u, 11159u, 11161u, 11171u, 11173u, 11177u, 11197u, 11213u, 11239u, 
         11243u, 11251u, 11257u, 11261u, 11273u, 11279u, 11287u, 11299u, 11311u, 
         11317u, 11321u, 11329u, 11351u, 11353u, 11369u, 11383u, 11393u, 11399u, 
         11411u, 11423u, 11437u, 11443u, 11447u, 11467u, 11471u, 11483u, 11489u, 
         11491u, 11497u, 11503u, 11519u, 11527u, 11549u, 11551u, 11579u, 11587u, 
         11593u, 11597u, 11617u, 11621u, 11633u, 11657u, 11677u, 11681u, 11689u, 
         11699u, 11701u, 11717u, 11719u, 11731u, 11743u, 11777u, 11779u, 11783u, 
         11789u, 11801u, 11807u, 11813u, 11821u, 11827u, 11831u, 11833u, 11839u, 
         11863u, 11867u, 11887u, 11897u, 11903u, 11909u, 11923u, 11927u, 11933u, 
         11939u, 11941u, 11953u, 11959u, 11969u, 11971u, 11981u, 11987u, 12007u, 
         12011u, 12037u, 12041u, 12043u, 12049u, 12071u, 12073u, 12097u, 12101u, 
         12107u, 12109u, 12113u, 12119u, 12143u, 12149u, 12157u, 12161u, 12163u, 
         12197u, 12203u, 12211u, 12227u, 12239u, 12241u, 12251u, 12253u, 12263u, 
         12269u, 12277u, 12281u, 12289u, 12301u, 12323u, 12329u, 12343u, 12347u, 
         12373u, 12377u, 12379u, 12391u, 12401u, 12409u, 12413u, 12421u, 12433u, 
         12437u, 12451u, 12457u, 12473u, 12479u, 12487u, 12491u, 12497u, 12503u, 
         12511u, 12517u, 12527u, 12539u, 12541u, 12547u, 12553u, 12569u, 12577u, 
         12583u, 12589u, 12601u, 12611u, 12613u, 12619u, 12637u, 12641u, 12647u, 
         12653u, 12659u, 12671u, 12689u, 12697u, 12703u, 12713u, 12721u, 12739u, 
         12743u, 12757u, 12763u, 12781u, 12791u, 12799u, 12809u, 12821u, 12823u, 
         12829u, 12841u, 12853u, 12889u, 12893u, 12899u, 12907u, 12911u, 12917u, 
         12919u, 12923u, 12941u, 12953u, 12959u, 12967u, 12973u, 12979u, 12983u, 
         13001u, 13003u, 13007u, 13009u, 13033u, 13037u, 13043u, 13049u, 13063u, 
         13093u, 13099u, 13103u, 13109u, 13121u, 13127u, 13147u, 13151u, 13159u, 
         13163u, 13171u, 13177u, 13183u, 13187u, 13217u, 13219u, 13229u, 13241u, 
         13249u, 13259u, 13267u, 13291u, 13297u, 13309u, 13313u, 13327u, 13331u, 
         13337u, 13339u, 13367u, 13381u, 13397u, 13399u, 13411u, 13417u, 13421u, 
         13441u, 13451u, 13457u, 13463u, 13469u, 13477u, 13487u, 13499u, 13513u, 
         13523u, 13537u, 13553u, 13567u, 13577u, 13591u, 13597u, 13613u, 13619u, 
         13627u, 13633u, 13649u, 13669u, 13679u, 13681u, 13687u, 13691u, 13693u, 
         13697u, 13709u, 13711u, 13721u, 13723u, 13729u, 13751u, 13757u, 13759u, 
         13763u, 13781u, 13789u, 13799u, 13807u, 13829u, 13831u, 13841u, 13859u, 
         13873u, 13877u, 13879u, 13883u, 13901u, 13903u, 13907u, 13913u, 13921u, 
         13931u, 13933u, 13963u, 13967u, 13997u, 13999u, 14009u, 14011u, 14029u, 
         14033u, 14051u, 14057u, 14071u, 14081u, 14083u, 14087u, 14107u, 14143u, 
         14149u, 14153u, 14159u, 14173u, 14177u, 14197u, 14207u, 14221u, 14243u, 
         14249u, 14251u, 14281u, 14293u, 14303u, 14321u, 14323u, 14327u, 14341u, 
         14347u, 14369u, 14387u, 14389u, 14401u, 14407u, 14411u, 14419u, 14423u, 
         14431u, 14437u, 14447u, 14449u, 14461u, 14479u, 14489u, 14503u, 14519u, 
         14533u, 14537u, 14543u, 14549u, 14551u, 14557u, 14561u, 14563u, 14591u, 
         14593u, 14621u, 14627u, 14629u, 14633u, 14639u, 14653u, 14657u, 14669u, 
         14683u, 14699u, 14713u, 14717u, 14723u, 14731u, 14737u, 14741u, 14747u, 
         14753u, 14759u, 14767u, 14771u, 14779u, 14783u, 14797u, 14813u, 14821u, 
         14827u, 14831u, 14843u, 14851u, 14867u, 14869u, 14879u, 14887u, 14891u, 
         14897u, 14923u, 14929u, 14939u, 14947u, 14951u, 14957u, 14969u, 14983u, 
         15013u, 15017u, 15031u, 15053u, 15061u, 15073u, 15077u, 15083u, 15091u, 
         15101u, 15107u, 15121u, 15131u, 15137u, 15139u, 15149u, 15161u, 15173u, 
         15187u, 15193u, 15199u, 15217u, 15227u, 15233u, 15241u, 15259u, 15263u, 
         15269u, 15271u, 15277u, 15287u, 15289u, 15299u, 15307u, 15313u, 15319u, 
         15329u, 15331u, 15349u, 15359u, 15361u, 15373u, 15377u, 15383u, 15391u, 
         15401u, 15413u, 15427u, 15439u, 15443u, 15451u, 15461u, 15467u, 15473u, 
         15493u, 15497u, 15511u, 15527u, 15541u, 15551u, 15559u, 15569u, 15581u, 
         15583u, 15601u, 15607u, 15619u, 15629u, 15641u, 15643u, 15647u, 15649u, 
         15661u, 15667u, 15671u, 15679u, 15683u, 15727u, 15731u, 15733u, 15737u, 
         15739u, 15749u, 15761u, 15767u, 15773u, 15787u, 15791u, 15797u, 15803u, 
         15809u, 15817u, 15823u, 15859u, 15877u, 15881u, 15887u, 15889u, 15901u, 
         15907u, 15913u, 15919u, 15923u, 15937u, 15959u, 15971u, 15973u, 15991u, 
         16001u, 16007u, 16033u, 16057u, 16061u, 16063u, 16067u, 16069u, 16073u, 
         16087u, 16091u, 16097u, 16103u, 16111u, 16127u, 16139u, 16141u, 16183u, 
         16187u, 16189u, 16193u, 16217u, 16223u, 16229u, 16231u, 16249u, 16253u, 
         16267u, 16273u, 16301u, 16319u, 16333u, 16339u, 16349u, 16361u, 16363u, 
         16369u, 16381u, 16411u, 16417u, 16421u, 16427u, 16433u, 16447u, 16451u, 
         16453u, 16477u, 16481u, 16487u, 16493u, 16519u, 16529u, 16547u, 16553u, 
         16561u, 16567u, 16573u, 16603u, 16607u, 16619u, 16631u, 16633u, 16649u, 
         16651u, 16657u, 16661u, 16673u, 16691u, 16693u, 16699u, 16703u, 16729u, 
         16741u, 16747u, 16759u, 16763u, 16787u, 16811u, 16823u, 16829u, 16831u, 
         16843u, 16871u, 16879u, 16883u, 16889u, 16901u, 16903u, 16921u, 16927u, 
         16931u, 16937u, 16943u, 16963u, 16979u, 16981u, 16987u, 16993u, 17011u, 
         17021u, 17027u, 17029u, 17033u, 17041u, 17047u, 17053u, 17077u, 17093u, 
         17099u, 17107u, 17117u, 17123u, 17137u, 17159u, 17167u, 17183u, 17189u, 
         17191u, 17203u, 17207u, 17209u, 17231u, 17239u, 17257u, 17291u, 17293u, 
         17299u, 17317u, 17321u, 17327u, 17333u, 17341u, 17351u, 17359u, 17377u, 
         17383u, 17387u, 17389u, 17393u, 17401u, 17417u, 17419u, 17431u, 17443u, 
         17449u, 17467u, 17471u, 17477u, 17483u, 17489u, 17491u, 17497u, 17509u, 
         17519u, 17539u, 17551u, 17569u, 17573u, 17579u, 17581u, 17597u, 17599u, 
         17609u, 17623u, 17627u, 17657u, 17659u, 17669u, 17681u, 17683u, 17707u, 
         17713u, 17729u, 17737u, 17747u, 17749u, 17761u, 17783u, 17789u, 17791u, 
         17807u, 17827u, 17837u, 17839u, 17851u, 17863u, 17881u, 17891u, 17903u, 
         17909u, 17911u, 17921u, 17923u, 17929u, 17939u, 17957u, 17959u, 17971u, 
         17977u, 17981u, 17987u, 17989u, 18013u, 18041u, 18043u, 18047u, 18049u, 
         18059u, 18061u, 18077u, 18089u, 18097u, 18119u, 18121u, 18127u, 18131u, 
         18133u, 18143u, 18149u, 18169u, 18181u, 18191u, 18199u, 18211u, 18217u, 
         18223u, 18229u, 18233u, 18251u, 18253u, 18257u, 18269u, 18287u, 18289u, 
         18301u, 18307u, 18311u, 18313u, 18329u, 18341u, 18353u, 18367u, 18371u, 
         18379u, 18397u, 18401u, 18413u, 18427u, 18433u, 18439u, 18443u, 18451u, 
         18457u, 18461u, 18481u, 18493u, 18503u, 18517u, 18521u, 18523u, 18539u, 
         18541u, 18553u, 18583u, 18587u, 18593u, 18617u, 18637u, 18661u, 18671u, 
         18679u, 18691u, 18701u, 18713u, 18719u, 18731u, 18743u, 18749u, 18757u, 
         18773u, 18787u, 18793u, 18797u, 18803u, 18839u, 18859u, 18869u, 18899u, 
         18911u, 18913u, 18917u, 18919u, 18947u, 18959u, 18973u, 18979u, 19001u, 
         19009u, 19013u, 19031u, 19037u, 19051u, 19069u, 19073u, 19079u, 19081u, 
         19087u, 19121u, 19139u, 19141u, 19157u, 19163u, 19181u, 19183u, 19207u, 
         19211u, 19213u, 19219u, 19231u, 19237u, 19249u, 19259u, 19267u, 19273u, 
         19289u, 19301u, 19309u, 19319u, 19333u, 19373u, 19379u, 19381u, 19387u, 
         19391u, 19403u, 19417u, 19421u, 19423u, 19427u, 19429u, 19433u, 19441u, 
         19447u, 19457u, 19463u, 19469u, 19471u, 19477u, 19483u, 19489u, 19501u, 
         19507u, 19531u, 19541u, 19543u, 19553u, 19559u, 19571u, 19577u, 19583u, 
         19597u, 19603u, 19609u, 19661u, 19681u, 19687u, 19697u, 19699u, 19709u, 
         19717u, 19727u, 19739u, 19751u, 19753u, 19759u, 19763u, 19777u, 19793u, 
         19801u, 19813u, 19819u, 19841u, 19843u, 19853u, 19861u, 19867u, 19889u, 
         19891u, 19913u, 19919u, 19927u, 19937u, 19949u, 19961u, 19963u, 19973u, 
         19979u, 19991u, 19993u, 19997u, 20011u, 20021u, 20023u, 20029u, 20047u, 
         20051u, 20063u, 20071u, 20089u, 20101u, 20107u, 20113u, 20117u, 20123u, 
         20129u, 20143u, 20147u, 20149u, 20161u, 20173u, 20177u, 20183u, 20201u, 
         20219u, 20231u, 20233u, 20249u, 20261u, 20269u, 20287u, 20297u, 20323u, 
         20327u, 20333u, 20341u, 20347u, 20353u, 20357u, 20359u, 20369u, 20389u, 
         20393u, 20399u, 20407u, 20411u, 20431u, 20441u, 20443u, 20477u, 20479u, 
         20483u, 20507u, 20509u, 20521u, 20533u, 20543u, 20549u, 20551u, 20563u, 
         20593u, 20599u, 20611u, 20627u, 20639u, 20641u, 20663u, 20681u, 20693u, 
         20707u, 20717u, 20719u, 20731u, 20743u, 20747u, 20749u, 20753u, 20759u, 
         20771u, 20773u, 20789u, 20807u, 20809u, 20849u, 20857u, 20873u, 20879u, 
         20887u, 20897u, 20899u, 20903u, 20921u, 20929u, 20939u, 20947u, 20959u, 
         20963u, 20981u, 20983u, 21001u, 21011u, 21013u, 21017u, 21019u, 21023u, 
         21031u, 21059u, 21061u, 21067u, 21089u, 21101u, 21107u, 21121u, 21139u, 
         21143u, 21149u, 21157u, 21163u, 21169u, 21179u, 21187u, 21191u, 21193u, 
         21211u, 21221u, 21227u, 21247u, 21269u, 21277u, 21283u, 21313u, 21317u, 
         21319u, 21323u, 21341u, 21347u, 21377u, 21379u, 21383u, 21391u, 21397u, 
         21401u, 21407u, 21419u, 21433u, 21467u, 21481u, 21487u, 21491u, 21493u, 
         21499u, 21503u, 21517u, 21521u, 21523u, 21529u, 21557u, 21559u, 21563u, 
         21569u, 21577u, 21587u, 21589u, 21599u, 21601u, 21611u, 21613u, 21617u, 
         21647u, 21649u, 21661u, 21673u, 21683u, 21701u, 21713u, 21727u, 21737u, 
         21739u, 21751u, 21757u, 21767u, 21773u, 21787u, 21799u, 21803u, 21817u, 
         21821u, 21839u, 21841u, 21851u, 21859u, 21863u, 21871u, 21881u, 21893u, 
         21911u, 21929u, 21937u, 21943u, 21961u, 21977u, 21991u, 21997u, 22003u, 
         22013u, 22027u, 22031u, 22037u, 22039u, 22051u, 22063u, 22067u, 22073u, 
         22079u, 22091u, 22093u, 22109u, 22111u, 22123u, 22129u, 22133u, 22147u, 
         22153u, 22157u, 22159u, 22171u, 22189u, 22193u, 22229u, 22247u, 22259u, 
         22271u, 22273u, 22277u, 22279u, 22283u, 22291u, 22303u, 22307u, 22343u, 
         22349u, 22367u, 22369u, 22381u, 22391u, 22397u, 22409u, 22433u, 22441u, 
         22447u, 22453u, 22469u, 22481u, 22483u, 22501u, 22511u, 22531u, 22541u, 
         22543u, 22549u, 22567u, 22571u, 22573u, 22613u, 22619u, 22621u, 22637u, 
         22639u, 22643u, 22651u, 22669u, 22679u, 22691u, 22697u, 22699u, 22709u, 
         22717u, 22721u, 22727u, 22739u, 22741u, 22751u, 22769u, 22777u, 22783u, 
         22787u, 22807u, 22811u, 22817u, 22853u, 22859u, 22861u, 22871u, 22877u, 
         22901u, 22907u, 22921u, 22937u, 22943u, 22961u, 22963u, 22973u, 22993u, 
         23003u, 23011u, 23017u, 23021u, 23027u, 23029u, 23039u, 23041u, 23053u, 
         23057u, 23059u, 23063u, 23071u, 23081u, 23087u, 23099u, 23117u, 23131u, 
         23143u, 23159u, 23167u, 23173u, 23189u, 23197u, 23201u, 23203u, 23209u, 
         23227u, 23251u, 23269u, 23279u, 23291u, 23293u, 23297u, 23311u, 23321u, 
         23327u, 23333u, 23339u, 23357u, 23369u, 23371u, 23399u, 23417u, 23431u, 
         23447u, 23459u, 23473u, 23497u, 23509u, 23531u, 23537u, 23539u, 23549u, 
         23557u, 23561u, 23563u, 23567u, 23581u, 23593u, 23599u, 23603u, 23609u, 
         23623u, 23627u, 23629u, 23633u, 23663u, 23669u, 23671u, 23677u, 23687u, 
         23689u, 23719u, 23741u, 23743u, 23747u, 23753u, 23761u, 23767u, 23773u, 
         23789u, 23801u, 23813u, 23819u, 23827u, 23831u, 23833u, 23857u, 23869u, 
         23873u, 23879u, 23887u, 23893u, 23899u, 23909u, 23911u, 23917u, 23929u, 
         23957u, 23971u, 23977u, 23981u, 23993u, 24001u, 24007u, 24019u, 24023u, 
         24029u, 24043u, 24049u, 24061u, 24071u, 24077u, 24083u, 24091u, 24097u, 
         24103u, 24107u, 24109u, 24113u, 24121u, 24133u, 24137u, 24151u, 24169u, 
         24179u, 24181u, 24197u, 24203u, 24223u, 24229u, 24239u, 24247u, 24251u, 
         24281u, 24317u, 24329u, 24337u, 24359u, 24371u, 24373u, 24379u, 24391u, 
         24407u, 24413u, 24419u, 24421u, 24439u, 24443u, 24469u, 24473u, 24481u, 
         24499u, 24509u, 24517u, 24527u, 24533u, 24547u, 24551u, 24571u, 24593u, 
         24611u, 24623u, 24631u, 24659u, 24671u, 24677u, 24683u, 24691u, 24697u, 
         24709u, 24733u, 24749u, 24763u, 24767u, 24781u, 24793u, 24799u, 24809u, 
         24821u, 24841u, 24847u, 24851u, 24859u, 24877u, 24889u, 24907u, 24917u, 
         24919u, 24923u, 24943u, 24953u, 24967u, 24971u, 24977u, 24979u, 24989u, 
         25013u, 25031u, 25033u, 25037u, 25057u, 25073u, 25087u, 25097u, 25111u, 
         25117u, 25121u, 25127u, 25147u, 25153u, 25163u, 25169u, 25171u, 25183u, 
         25189u, 25219u, 25229u, 25237u, 25243u, 25247u, 25253u, 25261u, 25301u, 
         25303u, 25307u, 25309u, 25321u, 25339u, 25343u, 25349u, 25357u, 25367u, 
         25373u, 25391u, 25409u, 25411u, 25423u, 25439u, 25447u, 25453u, 25457u, 
         25463u, 25469u, 25471u, 25523u, 25537u, 25541u, 25561u, 25577u, 25579u, 
         25583u, 25589u, 25601u, 25603u, 25609u, 25621u, 25633u, 25639u, 25643u, 
         25657u, 25667u, 25673u, 25679u, 25693u, 25703u, 25717u, 25733u, 25741u, 
         25747u, 25759u, 25763u, 25771u, 25793u, 25799u, 25801u, 25819u, 25841u, 
         25847u, 25849u, 25867u, 25873u, 25889u, 25903u, 25913u, 25919u, 25931u, 
         25933u, 25939u, 25943u, 25951u, 25969u, 25981u, 25997u, 25999u, 26003u, 
         26017u, 26021u, 26029u, 26041u, 26053u, 26083u, 26099u, 26107u, 26111u, 
         26113u, 26119u, 26141u, 26153u, 26161u, 26171u, 26177u, 26183u, 26189u, 
         26203u, 26209u, 26227u, 26237u, 26249u, 26251u, 26261u, 26263u, 26267u, 
         26293u, 26297u, 26309u, 26317u, 26321u, 26339u, 26347u, 26357u, 26371u, 
         26387u, 26393u, 26399u, 26407u, 26417u, 26423u, 26431u, 26437u, 26449u, 
         26459u, 26479u, 26489u, 26497u, 26501u, 26513u, 26539u, 26557u, 26561u, 
         26573u, 26591u, 26597u, 26627u, 26633u, 26641u, 26647u, 26669u, 26681u, 
         26683u, 26687u, 26693u, 26699u, 26701u, 26711u, 26713u, 26717u, 26723u, 
         26729u, 26731u, 26737u, 26759u, 26777u, 26783u, 26801u, 26813u, 26821u, 
         26833u, 26839u, 26849u, 26861u, 26863u, 26879u, 26881u, 26891u, 26893u, 
         26903u, 26921u, 26927u, 26947u, 26951u, 26953u, 26959u, 26981u, 26987u, 
         26993u, 27011u, 27017u, 27031u, 27043u, 27059u, 27061u, 27067u, 27073u, 
         27077u, 27091u, 27103u, 27107u, 27109u, 27127u, 27143u, 27179u, 27191u, 
         27197u, 27211u, 27239u, 27241u, 27253u, 27259u, 27271u, 27277u, 27281u, 
         27283u, 27299u, 27329u, 27337u, 27361u, 27367u, 27397u, 27407u, 27409u, 
         27427u, 27431u, 27437u, 27449u, 27457u, 27479u, 27481u, 27487u, 27509u, 
         27527u, 27529u, 27539u, 27541u, 27551u, 27581u, 27583u, 27611u, 27617u, 
         27631u, 27647u, 27653u, 27673u, 27689u, 27691u, 27697u, 27701u, 27733u, 
         27737u, 27739u, 27743u, 27749u, 27751u, 27763u, 27767u, 27773u, 27779u, 
         27791u, 27793u, 27799u, 27803u, 27809u, 27817u, 27823u, 27827u, 27847u, 
         27851u, 27883u, 27893u, 27901u, 27917u, 27919u, 27941u, 27943u, 27947u, 
         27953u, 27961u, 27967u, 27983u, 27997u, 28001u, 28019u, 28027u, 28031u, 
         28051u, 28057u, 28069u, 28081u, 28087u, 28097u, 28099u, 28109u, 28111u, 
         28123u, 28151u, 28163u, 28181u, 28183u, 28201u, 28211u, 28219u, 28229u, 
         28277u, 28279u, 28283u, 28289u, 28297u, 28307u, 28309u, 28319u, 28349u, 
         28351u, 28387u, 28393u, 28403u, 28409u, 28411u, 28429u, 28433u, 28439u, 
         28447u, 28463u, 28477u, 28493u, 28499u, 28513u, 28517u, 28537u, 28541u, 
         28547u, 28549u, 28559u, 28571u, 28573u, 28579u, 28591u, 28597u, 28603u, 
         28607u, 28619u, 28621u, 28627u, 28631u, 28643u, 28649u, 28657u, 28661u, 
         28663u, 28669u, 28687u, 28697u, 28703u, 28711u, 28723u, 28729u, 28751u, 
         28753u, 28759u, 28771u, 28789u, 28793u, 28807u, 28813u, 28817u, 28837u, 
         28843u, 28859u, 28867u, 28871u, 28879u, 28901u, 28909u, 28921u, 28927u, 
         28933u, 28949u, 28961u, 28979u, 29009u, 29017u, 29021u, 29023u, 29027u, 
         29033u, 29059u, 29063u, 29077u, 29101u, 29123u, 29129u, 29131u, 29137u, 
         29147u, 29153u, 29167u, 29173u, 29179u, 29191u, 29201u, 29207u, 29209u, 
         29221u, 29231u, 29243u, 29251u, 29269u, 29287u, 29297u, 29303u, 29311u, 
         29327u, 29333u, 29339u, 29347u, 29363u, 29383u, 29387u, 29389u, 29399u, 
         29401u, 29411u, 29423u, 29429u, 29437u, 29443u, 29453u, 29473u, 29483u, 
         29501u, 29527u, 29531u, 29537u, 29567u, 29569u, 29573u, 29581u, 29587u, 
         29599u, 29611u, 29629u, 29633u, 29641u, 29663u, 29669u, 29671u, 29683u, 
         29717u, 29723u, 29741u, 29753u, 29759u, 29761u, 29789u, 29803u, 29819u, 
         29833u, 29837u, 29851u, 29863u, 29867u, 29873u, 29879u, 29881u, 29917u, 
         29921u, 29927u, 29947u, 29959u, 29983u, 29989u, 30011u, 30013u, 30029u, 
         30047u, 30059u, 30071u, 30089u, 30091u, 30097u, 30103u, 30109u, 30113u, 
         30119u, 30133u, 30137u, 30139u, 30161u, 30169u, 30181u, 30187u, 30197u, 
         30203u, 30211u, 30223u, 30241u, 30253u, 30259u, 30269u, 30271u, 30293u, 
         30307u, 30313u, 30319u, 30323u, 30341u, 30347u, 30367u, 30389u, 30391u, 
         30403u, 30427u, 30431u, 30449u, 30467u, 30469u, 30491u, 30493u, 30497u, 
         30509u, 30517u, 30529u, 30539u, 30553u, 30557u, 30559u, 30577u, 30593u, 
         30631u, 30637u, 30643u, 30649u, 30661u, 30671u, 30677u, 30689u, 30697u, 
         30703u, 30707u, 30713u, 30727u, 30757u, 30763u, 30773u, 30781u, 30803u, 
         30809u, 30817u, 30829u, 30839u, 30841u, 30851u, 30853u, 30859u, 30869u, 
         30871u, 30881u, 30893u, 30911u, 30931u, 30937u, 30941u, 30949u, 30971u, 
         30977u, 30983u, 31013u, 31019u, 31033u, 31039u, 31051u, 31063u, 31069u, 
         31079u, 31081u, 31091u, 31121u, 31123u, 31139u, 31147u, 31151u, 31153u, 
         31159u, 31177u, 31181u, 31183u, 31189u, 31193u, 31219u, 31223u, 31231u, 
         31237u, 31247u, 31249u, 31253u, 31259u, 31267u, 31271u, 31277u, 31307u, 
         31319u, 31321u, 31327u, 31333u, 31337u, 31357u, 31379u, 31387u, 31391u, 
         31393u, 31397u, 31469u, 31477u, 31481u, 31489u, 31511u, 31513u, 31517u, 
         31531u, 31541u, 31543u, 31547u, 31567u, 31573u, 31583u, 31601u, 31607u, 
         31627u, 31643u, 31649u, 31657u, 31663u, 31667u, 31687u, 31699u, 31721u, 
         31723u, 31727u, 31729u, 31741u, 31751u, 31769u, 31771u, 31793u, 31799u, 
         31817u, 31847u, 31849u, 31859u, 31873u, 31883u, 31891u, 31907u, 31957u, 
         31963u, 31973u, 31981u, 31991u, 32003u, 32009u, 32027u, 32029u, 32051u, 
         32057u, 32059u, 32063u, 32069u, 32077u, 32083u, 32089u, 32099u, 32117u, 
         32119u, 32141u, 32143u, 32159u, 32173u, 32183u, 32189u, 32191u, 32203u, 
         32213u, 32233u, 32237u, 32251u, 32257u, 32261u, 32297u, 32299u, 32303u, 
         32309u, 32321u, 32323u, 32327u, 32341u, 32353u, 32359u, 32363u, 32369u, 
         32371u, 32377u, 32381u, 32401u, 32411u, 32413u, 32423u, 32429u, 32441u, 
         32443u, 32467u, 32479u, 32491u, 32497u, 32503u, 32507u, 32531u, 32533u, 
         32537u, 32561u, 32563u, 32569u, 32573u, 32579u, 32587u, 32603u, 32609u, 
         32611u, 32621u, 32633u, 32647u, 32653u, 32687u, 32693u, 32707u, 32713u, 
         32717u, 32719u, 32749u, 32771u, 32779u, 32783u, 32789u, 32797u, 32801u, 
         32803u, 32831u, 32833u, 32839u, 32843u, 32869u, 32887u, 32909u, 32911u, 
         32917u, 32933u, 32939u, 32941u, 32957u, 32969u, 32971u, 32983u, 32987u, 
         32993u, 32999u, 33013u, 33023u, 33029u, 33037u, 33049u, 33053u, 33071u, 
         33073u, 33083u, 33091u, 33107u, 33113u, 33119u, 33149u, 33151u, 33161u, 
         33179u, 33181u, 33191u, 33199u, 33203u, 33211u, 33223u, 33247u, 33287u, 
         33289u, 33301u, 33311u, 33317u, 33329u, 33331u, 33343u, 33347u, 33349u, 
         33353u, 33359u, 33377u, 33391u, 33403u, 33409u, 33413u, 33427u, 33457u, 
         33461u, 33469u, 33479u, 33487u, 33493u, 33503u, 33521u, 33529u, 33533u, 
         33547u, 33563u, 33569u, 33577u, 33581u, 33587u, 33589u, 33599u, 33601u, 
         33613u, 33617u, 33619u, 33623u, 33629u, 33637u, 33641u, 33647u, 33679u, 
         33703u, 33713u, 33721u, 33739u, 33749u, 33751u, 33757u, 33767u, 33769u, 
         33773u, 33791u, 33797u, 33809u, 33811u, 33827u, 33829u, 33851u, 33857u, 
         33863u, 33871u, 33889u, 33893u, 33911u, 33923u, 33931u, 33937u, 33941u, 
         33961u, 33967u, 33997u, 34019u, 34031u, 34033u, 34039u, 34057u, 34061u, 
         34123u, 34127u, 34129u, 34141u, 34147u, 34157u, 34159u, 34171u, 34183u, 
         34211u, 34213u, 34217u, 34231u, 34253u, 34259u, 34261u, 34267u, 34273u, 
         34283u, 34297u, 34301u, 34303u, 34313u, 34319u, 34327u, 34337u, 34351u, 
         34361u, 34367u, 34369u, 34381u, 34403u, 34421u, 34429u, 34439u, 34457u, 
         34469u, 34471u, 34483u, 34487u, 34499u, 34501u, 34511u, 34513u, 34519u, 
         34537u, 34543u, 34549u, 34583u, 34589u, 34591u, 34603u, 34607u, 34613u, 
         34631u, 34649u, 34651u, 34667u, 34673u, 34679u, 34687u, 34693u, 34703u, 
         34721u, 34729u, 34739u, 34747u, 34757u, 34759u, 34763u, 34781u, 34807u, 
         34819u, 34841u, 34843u, 34847u, 34849u, 34871u, 34877u, 34883u, 34897u, 
         34913u, 34919u, 34939u, 34949u, 34961u, 34963u, 34981u, 35023u, 35027u, 
         35051u, 35053u, 35059u, 35069u, 35081u, 35083u, 35089u, 35099u, 35107u, 
         35111u, 35117u, 35129u, 35141u, 35149u, 35153u, 35159u, 35171u, 35201u, 
         35221u, 35227u, 35251u, 35257u, 35267u, 35279u, 35281u, 35291u, 35311u, 
         35317u, 35323u, 35327u, 35339u, 35353u, 35363u, 35381u, 35393u, 35401u, 
         35407u, 35419u, 35423u, 35437u, 35447u, 35449u, 35461u, 35491u, 35507u, 
         35509u, 35521u, 35527u, 35531u, 35533u, 35537u, 35543u, 35569u, 35573u, 
         35591u, 35593u, 35597u, 35603u, 35617u, 35671u, 35677u, 35729u, 35731u, 
         35747u, 35753u, 35759u, 35771u, 35797u, 35801u, 35803u, 35809u, 35831u, 
         35837u, 35839u, 35851u, 35863u, 35869u, 35879u, 35897u, 35899u, 35911u, 
         35923u, 35933u, 35951u, 35963u, 35969u, 35977u, 35983u, 35993u, 35999u, 
         36007u, 36011u, 36013u, 36017u, 36037u, 36061u, 36067u, 36073u, 36083u, 
         36097u, 36107u, 36109u, 36131u, 36137u, 36151u, 36161u, 36187u, 36191u, 
         36209u, 36217u, 36229u, 36241u, 36251u, 36263u, 36269u, 36277u, 36293u, 
         36299u, 36307u, 36313u, 36319u, 36341u, 36343u, 36353u, 36373u, 36383u, 
         36389u, 36433u, 36451u, 36457u, 36467u, 36469u, 36473u, 36479u, 36493u, 
         36497u, 36523u, 36527u, 36529u, 36541u, 36551u, 36559u, 36563u, 36571u, 
         36583u, 36587u, 36599u, 36607u, 36629u, 36637u, 36643u, 36653u, 36671u, 
         36677u, 36683u, 36691u, 36697u, 36709u, 36713u, 36721u, 36739u, 36749u, 
         36761u, 36767u, 36779u, 36781u, 36787u, 36791u, 36793u, 36809u, 36821u, 
         36833u, 36847u, 36857u, 36871u, 36877u, 36887u, 36899u, 36901u, 36913u, 
         36919u, 36923u, 36929u, 36931u, 36943u, 36947u, 36973u, 36979u, 36997u, 
         37003u, 37013u, 37019u, 37021u, 37039u, 37049u, 37057u, 37061u, 37087u, 
         37097u, 37117u, 37123u, 37139u, 37159u, 37171u, 37181u, 37189u, 37199u, 
         37201u, 37217u, 37223u, 37243u, 37253u, 37273u, 37277u, 37307u, 37309u, 
         37313u, 37321u, 37337u, 37339u, 37357u, 37361u, 37363u, 37369u, 37379u, 
         37397u, 37409u, 37423u, 37441u, 37447u, 37463u, 37483u, 37489u, 37493u, 
         37501u, 37507u, 37511u, 37517u, 37529u, 37537u, 37547u, 37549u, 37561u, 
         37567u, 37571u, 37573u, 37579u, 37589u, 37591u, 37607u, 37619u, 37633u, 
         37643u, 37649u, 37657u, 37663u, 37691u, 37693u, 37699u, 37717u, 37747u, 
         37781u, 37783u, 37799u, 37811u, 37813u, 37831u, 37847u, 37853u, 37861u, 
         37871u, 37879u, 37889u, 37897u, 37907u, 37951u, 37957u, 37963u, 37967u, 
         37987u, 37991u, 37993u, 37997u, 38011u, 38039u, 38047u, 38053u, 38069u, 
         38083u, 38113u, 38119u, 38149u, 38153u, 38167u, 38177u, 38183u, 38189u, 
         38197u, 38201u, 38219u, 38231u, 38237u, 38239u, 38261u, 38273u, 38281u, 
         38287u, 38299u, 38303u, 38317u, 38321u, 38327u, 38329u, 38333u, 38351u, 
         38371u, 38377u, 38393u, 38431u, 38447u, 38449u, 38453u, 38459u, 38461u, 
         38501u, 38543u, 38557u, 38561u, 38567u, 38569u, 38593u, 38603u, 38609u, 
         38611u, 38629u, 38639u, 38651u, 38653u, 38669u, 38671u, 38677u, 38693u, 
         38699u, 38707u, 38711u, 38713u, 38723u, 38729u, 38737u, 38747u, 38749u, 
         38767u, 38783u, 38791u, 38803u, 38821u, 38833u, 38839u, 38851u, 38861u, 
         38867u, 38873u, 38891u, 38903u, 38917u, 38921u, 38923u, 38933u, 38953u, 
         38959u, 38971u, 38977u, 38993u, 39019u, 39023u, 39041u, 39043u, 39047u, 
         39079u, 39089u, 39097u, 39103u, 39107u, 39113u, 39119u, 39133u, 39139u, 
         39157u, 39161u, 39163u, 39181u, 39191u, 39199u, 39209u, 39217u, 39227u, 
         39229u, 39233u, 39239u, 39241u, 39251u, 39293u, 39301u, 39313u, 39317u, 
         39323u, 39341u, 39343u, 39359u, 39367u, 39371u, 39373u, 39383u, 39397u, 
         39409u, 39419u, 39439u, 39443u, 39451u, 39461u, 39499u, 39503u, 39509u, 
         39511u, 39521u, 39541u, 39551u, 39563u, 39569u, 39581u, 39607u, 39619u, 
         39623u, 39631u, 39659u, 39667u, 39671u, 39679u, 39703u, 39709u, 39719u, 
         39727u, 39733u, 39749u, 39761u, 39769u, 39779u, 39791u, 39799u, 39821u, 
         39827u, 39829u, 39839u, 39841u, 39847u, 39857u, 39863u, 39869u, 39877u, 
         39883u, 39887u, 39901u, 39929u, 39937u, 39953u, 39971u, 39979u, 39983u, 
         39989u, 40009u, 40013u, 40031u, 40037u, 40039u, 40063u, 40087u, 40093u, 
         40099u, 40111u, 40123u, 40127u, 40129u, 40151u, 40153u, 40163u, 40169u, 
         40177u, 40189u, 40193u, 40213u, 40231u, 40237u, 40241u, 40253u, 40277u, 
         40283u, 40289u, 40343u, 40351u, 40357u, 40361u, 40387u, 40423u, 40427u, 
         40429u, 40433u, 40459u, 40471u, 40483u, 40487u, 40493u, 40499u, 40507u, 
         40519u, 40529u, 40531u, 40543u, 40559u, 40577u, 40583u, 40591u, 40597u, 
         40609u, 40627u, 40637u, 40639u, 40693u, 40697u, 40699u, 40709u, 40739u, 
         40751u, 40759u, 40763u, 40771u, 40787u, 40801u, 40813u, 40819u, 40823u, 
         40829u, 40841u, 40847u, 40849u, 40853u, 40867u, 40879u, 40883u, 40897u, 
         40903u, 40927u, 40933u, 40939u, 40949u, 40961u, 40973u, 40993u, 41011u, 
         41017u, 41023u, 41039u, 41047u, 41051u, 41057u, 41077u, 41081u, 41113u, 
         41117u, 41131u, 41141u, 41143u, 41149u, 41161u, 41177u, 41179u, 41183u, 
         41189u, 41201u, 41203u, 41213u, 41221u, 41227u, 41231u, 41233u, 41243u, 
         41257u, 41263u, 41269u, 41281u, 41299u, 41333u, 41341u, 41351u, 41357u, 
         41381u, 41387u, 41389u, 41399u, 41411u, 41413u, 41443u, 41453u, 41467u, 
         41479u, 41491u, 41507u, 41513u, 41519u, 41521u, 41539u, 41543u, 41549u, 
         41579u, 41593u, 41597u, 41603u, 41609u, 41611u, 41617u, 41621u, 41627u, 
         41641u, 41647u, 41651u, 41659u, 41669u, 41681u, 41687u, 41719u, 41729u, 
         41737u, 41759u, 41761u, 41771u, 41777u, 41801u, 41809u, 41813u, 41843u, 
         41849u, 41851u, 41863u, 41879u, 41887u, 41893u, 41897u, 41903u, 41911u, 
         41927u, 41941u, 41947u, 41953u, 41957u, 41959u, 41969u, 41981u, 41983u, 
         41999u, 42013u, 42017u, 42019u, 42023u, 42043u, 42061u, 42071u, 42073u, 
         42083u, 42089u, 42101u, 42131u, 42139u, 42157u, 42169u, 42179u, 42181u, 
         42187u, 42193u, 42197u, 42209u, 42221u, 42223u, 42227u, 42239u, 42257u, 
         42281u, 42283u, 42293u, 42299u, 42307u, 42323u, 42331u, 42337u, 42349u, 
         42359u, 42373u, 42379u, 42391u, 42397u, 42403u, 42407u, 42409u, 42433u, 
         42437u, 42443u, 42451u, 42457u, 42461u, 42463u, 42467u, 42473u, 42487u, 
         42491u, 42499u, 42509u, 42533u, 42557u, 42569u, 42571u, 42577u, 42589u, 
         42611u, 42641u, 42643u, 42649u, 42667u, 42677u, 42683u, 42689u, 42697u, 
         42701u, 42703u, 42709u, 42719u, 42727u, 42737u, 42743u, 42751u, 42767u, 
         42773u, 42787u, 42793u, 42797u, 42821u, 42829u, 42839u, 42841u, 42853u, 
         42859u, 42863u, 42899u, 42901u, 42923u, 42929u, 42937u, 42943u, 42953u, 
         42961u, 42967u, 42979u, 42989u, 43003u, 43013u, 43019u, 43037u, 43049u, 
         43051u, 43063u, 43067u, 43093u, 43103u, 43117u, 43133u, 43151u, 43159u, 
         43177u, 43189u, 43201u, 43207u, 43223u, 43237u, 43261u, 43271u, 43283u, 
         43291u, 43313u, 43319u, 43321u, 43331u, 43391u, 43397u, 43399u, 43403u, 
         43411u, 43427u, 43441u, 43451u, 43457u, 43481u, 43487u, 43499u, 43517u, 
         43541u, 43543u, 43573u, 43577u, 43579u, 43591u, 43597u, 43607u, 43609u, 
         43613u, 43627u, 43633u, 43649u, 43651u, 43661u, 43669u, 43691u, 43711u, 
         43717u, 43721u, 43753u, 43759u, 43777u, 43781u, 43783u, 43787u, 43789u, 
         43793u, 43801u, 43853u, 43867u, 43889u, 43891u, 43913u, 43933u, 43943u, 
         43951u, 43961u, 43963u, 43969u, 43973u, 43987u, 43991u, 43997u, 44017u, 
         44021u, 44027u, 44029u, 44041u, 44053u, 44059u, 44071u, 44087u, 44089u, 
         44101u, 44111u, 44119u, 44123u, 44129u, 44131u, 44159u, 44171u, 44179u, 
         44189u, 44201u, 44203u, 44207u, 44221u, 44249u, 44257u, 44263u, 44267u, 
         44269u, 44273u, 44279u, 44281u, 44293u, 44351u, 44357u, 44371u, 44381u, 
         44383u, 44389u, 44417u, 44449u, 44453u, 44483u, 44491u, 44497u, 44501u, 
         44507u, 44519u, 44531u, 44533u, 44537u, 44543u, 44549u, 44563u, 44579u, 
         44587u, 44617u, 44621u, 44623u, 44633u, 44641u, 44647u, 44651u, 44657u, 
         44683u, 44687u, 44699u, 44701u, 44711u, 44729u, 44741u, 44753u, 44771u, 
         44773u, 44777u, 44789u, 44797u, 44809u, 44819u, 44839u, 44843u, 44851u, 
         44867u, 44879u, 44887u, 44893u, 44909u, 44917u, 44927u, 44939u, 44953u, 
         44959u, 44963u, 44971u, 44983u, 44987u, 45007u, 45013u, 45053u, 45061u, 
         45077u, 45083u, 45119u, 45121u, 45127u, 45131u, 45137u, 45139u, 45161u, 
         45179u, 45181u, 45191u, 45197u, 45233u, 45247u, 45259u, 45263u, 45281u, 
         45289u, 45293u, 45307u, 45317u, 45319u, 45329u, 45337u, 45341u, 45343u, 
         45361u, 45377u, 45389u, 45403u, 45413u, 45427u, 45433u, 45439u, 45481u, 
         45491u, 45497u, 45503u, 45523u, 45533u, 45541u, 45553u, 45557u, 45569u, 
         45587u, 45589u, 45599u, 45613u, 45631u, 45641u, 45659u, 45667u, 45673u, 
         45677u, 45691u, 45697u, 45707u, 45737u, 45751u, 45757u, 45763u, 45767u, 
         45779u, 45817u, 45821u, 45823u, 45827u, 45833u, 45841u, 45853u, 45863u, 
         45869u, 45887u, 45893u, 45943u, 45949u, 45953u, 45959u, 45971u, 45979u, 
         45989u, 46021u, 46027u, 46049u, 46051u, 46061u, 46073u, 46091u, 46093u, 
         46099u, 46103u, 46133u, 46141u, 46147u, 46153u, 46171u, 46181u, 46183u, 
         46187u, 46199u, 46219u, 46229u, 46237u, 46261u, 46271u, 46273u, 46279u, 
         46301u, 46307u, 46309u, 46327u, 46337u, 46349u, 46351u, 46381u, 46399u, 
         46411u, 46439u, 46441u, 46447u, 46451u, 46457u, 46471u, 46477u, 46489u, 
         46499u, 46507u, 46511u, 46523u, 46549u, 46559u, 46567u, 46573u, 46589u, 
         46591u, 46601u, 46619u, 46633u, 46639u, 46643u, 46649u, 46663u, 46679u, 
         46681u, 46687u, 46691u, 46703u, 46723u, 46727u, 46747u, 46751u, 46757u, 
         46769u, 46771u, 46807u, 46811u, 46817u, 46819u, 46829u, 46831u, 46853u, 
         46861u, 46867u, 46877u, 46889u, 46901u, 46919u, 46933u, 46957u, 46993u, 
         46997u, 47017u, 47041u, 47051u, 47057u, 47059u, 47087u, 47093u, 47111u, 
         47119u, 47123u, 47129u, 47137u, 47143u, 47147u, 47149u, 47161u, 47189u, 
         47207u, 47221u, 47237u, 47251u, 47269u, 47279u, 47287u, 47293u, 47297u, 
         47303u, 47309u, 47317u, 47339u, 47351u, 47353u, 47363u, 47381u, 47387u, 
         47389u, 47407u, 47417u, 47419u, 47431u, 47441u, 47459u, 47491u, 47497u, 
         47501u, 47507u, 47513u, 47521u, 47527u, 47533u, 47543u, 47563u, 47569u, 
         47581u, 47591u, 47599u, 47609u, 47623u, 47629u, 47639u, 47653u, 47657u, 
         47659u, 47681u, 47699u, 47701u, 47711u, 47713u, 47717u, 47737u, 47741u, 
         47743u, 47777u, 47779u, 47791u, 47797u, 47807u, 47809u, 47819u, 47837u, 
         47843u, 47857u, 47869u, 47881u, 47903u, 47911u, 47917u, 47933u, 47939u, 
         47947u, 47951u, 47963u, 47969u, 47977u, 47981u, 48017u, 48023u, 48029u, 
         48049u, 48073u, 48079u, 48091u, 48109u, 48119u, 48121u, 48131u, 48157u, 
         48163u, 48179u, 48187u, 48193u, 48197u, 48221u, 48239u, 48247u, 48259u, 
         48271u, 48281u, 48299u, 48311u, 48313u, 48337u, 48341u, 48353u, 48371u, 
         48383u, 48397u, 48407u, 48409u, 48413u, 48437u, 48449u, 48463u, 48473u, 
         48479u, 48481u, 48487u, 48491u, 48497u, 48523u, 48527u, 48533u, 48539u, 
         48541u, 48563u, 48571u, 48589u, 48593u, 48611u, 48619u, 48623u, 48647u, 
         48649u, 48661u, 48673u, 48677u, 48679u, 48731u, 48733u, 48751u, 48757u, 
         48761u, 48767u, 48779u, 48781u, 48787u, 48799u, 48809u, 48817u, 48821u, 
         48823u, 48847u, 48857u, 48859u, 48869u, 48871u, 48883u, 48889u, 48907u, 
         48947u, 48953u, 48973u, 48989u, 48991u, 49003u, 49009u, 49019u, 49031u, 
         49033u, 49037u, 49043u, 49057u, 49069u, 49081u, 49103u, 49109u, 49117u, 
         49121u, 49123u, 49139u, 49157u, 49169u, 49171u, 49177u, 49193u, 49199u, 
         49201u, 49207u, 49211u, 49223u, 49253u, 49261u, 49277u, 49279u, 49297u, 
         49307u, 49331u, 49333u, 49339u, 49363u, 49367u, 49369u, 49391u, 49393u, 
         49409u, 49411u, 49417u, 49429u, 49433u, 49451u, 49459u, 49463u, 49477u, 
         49481u, 49499u, 49523u, 49529u, 49531u, 49537u, 49547u, 49549u, 49559u, 
         49597u, 49603u, 49613u, 49627u, 49633u, 49639u, 49663u, 49667u, 49669u, 
         49681u, 49697u, 49711u, 49727u, 49739u, 49741u, 49747u, 49757u, 49783u, 
         49787u, 49789u, 49801u, 49807u, 49811u, 49823u, 49831u, 49843u, 49853u, 
         49871u, 49877u, 49891u, 49919u, 49921u, 49927u, 49937u, 49939u, 49943u, 
         49957u, 49991u, 49993u, 49999u, 50021u, 50023u, 50033u, 50047u, 50051u, 
         50053u, 50069u, 50077u, 50087u, 50093u, 50101u, 50111u, 50119u, 50123u, 
         50129u, 50131u, 50147u, 50153u, 50159u, 50177u, 50207u, 50221u, 50227u, 
         50231u, 50261u, 50263u, 50273u, 50287u, 50291u, 50311u, 50321u, 50329u, 
         50333u, 50341u, 50359u, 50363u, 50377u, 50383u, 50387u, 50411u, 50417u, 
         50423u, 50441u, 50459u, 50461u, 50497u, 50503u, 50513u, 50527u, 50539u, 
         50543u, 50549u, 50551u, 50581u, 50587u, 50591u, 50593u, 50599u, 50627u, 
         50647u, 50651u, 50671u, 50683u, 50707u, 50723u, 50741u, 50753u, 50767u, 
         50773u, 50777u, 50789u, 50821u, 50833u, 50839u, 50849u, 50857u, 50867u, 
         50873u, 50891u, 50893u, 50909u, 50923u, 50929u, 50951u, 50957u, 50969u, 
         50971u, 50989u, 50993u, 51001u, 51031u, 51043u, 51047u, 51059u, 51061u, 
         51071u, 51109u, 51131u, 51133u, 51137u, 51151u, 51157u, 51169u, 51193u, 
         51197u, 51199u, 51203u, 51217u, 51229u, 51239u, 51241u, 51257u, 51263u, 
         51283u, 51287u, 51307u, 51329u, 51341u, 51343u, 51347u, 51349u, 51361u, 
         51383u, 51407u, 51413u, 51419u, 51421u, 51427u, 51431u, 51437u, 51439u, 
         51449u, 51461u, 51473u, 51479u, 51481u, 51487u, 51503u, 51511u, 51517u, 
         51521u, 51539u, 51551u, 51563u, 51577u, 51581u, 51593u, 51599u, 51607u, 
         51613u, 51631u, 51637u, 51647u, 51659u, 51673u, 51679u, 51683u, 51691u, 
         51713u, 51719u, 51721u, 51749u, 51767u, 51769u, 51787u, 51797u, 51803u, 
         51817u, 51827u, 51829u, 51839u, 51853u, 51859u, 51869u, 51871u, 51893u, 
         51899u, 51907u, 51913u, 51929u, 51941u, 51949u, 51971u, 51973u, 51977u, 
         51991u, 52009u, 52021u, 52027u, 52051u, 52057u, 52067u, 52069u, 52081u, 
         52103u, 52121u, 52127u, 52147u, 52153u, 52163u, 52177u, 52181u, 52183u, 
         52189u, 52201u, 52223u, 52237u, 52249u, 52253u, 52259u, 52267u, 52289u, 
         52291u, 52301u, 52313u, 52321u, 52361u, 52363u, 52369u, 52379u, 52387u, 
         52391u, 52433u, 52453u, 52457u, 52489u, 52501u, 52511u, 52517u, 52529u, 
         52541u, 52543u, 52553u, 52561u, 52567u, 52571u, 52579u, 52583u, 52609u, 
         52627u, 52631u, 52639u, 52667u, 52673u, 52691u, 52697u, 52709u, 52711u, 
         52721u, 52727u, 52733u, 52747u, 52757u, 52769u, 52783u, 52807u, 52813u, 
         52817u, 52837u, 52859u, 52861u, 52879u, 52883u, 52889u, 52901u, 52903u, 
         52919u, 52937u, 52951u, 52957u, 52963u, 52967u, 52973u, 52981u, 52999u, 
         53003u, 53017u, 53047u, 53051u, 53069u, 53077u, 53087u, 53089u, 53093u, 
         53101u, 53113u, 53117u, 53129u, 53147u, 53149u, 53161u, 53171u, 53173u, 
         53189u, 53197u, 53201u, 53231u, 53233u, 53239u, 53267u, 53269u, 53279u, 
         53281u, 53299u, 53309u, 53323u, 53327u, 53353u, 53359u, 53377u, 53381u, 
         53401u, 53407u, 53411u, 53419u, 53437u, 53441u, 53453u, 53479u, 53503u, 
         53507u, 53527u, 53549u, 53551u, 53569u, 53591u, 53593u, 53597u, 53609u, 
         53611u, 53617u, 53623u, 53629u, 53633u, 53639u, 53653u, 53657u, 53681u, 
         53693u, 53699u, 53717u, 53719u, 53731u, 53759u, 53773u, 53777u, 53783u, 
         53791u, 53813u, 53819u, 53831u, 53849u, 53857u, 53861u, 53881u, 53887u, 
         53891u, 53897u, 53899u, 53917u, 53923u, 53927u, 53939u, 53951u, 53959u, 
         53987u, 53993u, 54001u, 54011u, 54013u, 54037u, 54049u, 54059u, 54083u, 
         54091u, 54101u, 54121u, 54133u, 54139u, 54151u, 54163u, 54167u, 54181u, 
         54193u, 54217u, 54251u, 54269u, 54277u, 54287u, 54293u, 54311u, 54319u, 
         54323u, 54331u, 54347u, 54361u, 54367u, 54371u, 54377u, 54401u, 54403u, 
         54409u, 54413u, 54419u, 54421u, 54437u, 54443u, 54449u, 54469u, 54493u, 
         54497u, 54499u, 54503u, 54517u, 54521u, 54539u, 54541u, 54547u, 54559u, 
         54563u, 54577u, 54581u, 54583u, 54601u, 54617u, 54623u, 54629u, 54631u, 
         54647u, 54667u, 54673u, 54679u, 54709u, 54713u, 54721u, 54727u, 54751u, 
         54767u, 54773u, 54779u, 54787u, 54799u, 54829u, 54833u, 54851u, 54869u, 
         54877u, 54881u, 54907u, 54917u, 54919u, 54941u, 54949u, 54959u, 54973u, 
         54979u, 54983u, 55001u, 55009u, 55021u, 55049u, 55051u, 55057u, 55061u, 
         55073u, 55079u, 55103u, 55109u, 55117u, 55127u, 55147u, 55163u, 55171u, 
         55201u, 55207u, 55213u, 55217u, 55219u, 55229u, 55243u, 55249u, 55259u, 
         55291u, 55313u, 55331u, 55333u, 55337u, 55339u, 55343u, 55351u, 55373u, 
         55381u, 55399u, 55411u, 55439u, 55441u, 55457u, 55469u, 55487u, 55501u, 
         55511u, 55529u, 55541u, 55547u, 55579u, 55589u, 55603u, 55609u, 55619u, 
         55621u, 55631u, 55633u, 55639u, 55661u, 55663u, 55667u, 55673u, 55681u, 
         55691u, 55697u, 55711u, 55717u, 55721u, 55733u, 55763u, 55787u, 55793u, 
         55799u, 55807u, 55813u, 55817u, 55819u, 55823u, 55829u, 55837u, 55843u, 
         55849u, 55871u, 55889u, 55897u, 55901u, 55903u, 55921u, 55927u, 55931u, 
         55933u, 55949u, 55967u, 55987u, 55997u, 56003u, 56009u, 56039u, 56041u, 
         56053u, 56081u, 56087u, 56093u, 56099u, 56101u, 56113u, 56123u, 56131u, 
         56149u, 56167u, 56171u, 56179u, 56197u, 56207u, 56209u, 56237u, 56239u, 
         56249u, 56263u, 56267u, 56269u, 56299u, 56311u, 56333u, 56359u, 56369u, 
         56377u, 56383u, 56393u, 56401u, 56417u, 56431u, 56437u, 56443u, 56453u, 
         56467u, 56473u, 56477u, 56479u, 56489u, 56501u, 56503u, 56509u, 56519u, 
         56527u, 56531u, 56533u, 56543u, 56569u, 56591u, 56597u, 56599u, 56611u, 
         56629u, 56633u, 56659u, 56663u, 56671u, 56681u, 56687u, 56701u, 56711u, 
         56713u, 56731u, 56737u, 56747u, 56767u, 56773u, 56779u, 56783u, 56807u, 
         56809u, 56813u, 56821u, 56827u, 56843u, 56857u, 56873u, 56891u, 56893u, 
         56897u, 56909u, 56911u, 56921u, 56923u, 56929u, 56941u, 56951u, 56957u, 
         56963u, 56983u, 56989u, 56993u, 56999u, 57037u, 57041u, 57047u, 57059u, 
         57073u, 57077u, 57089u, 57097u, 57107u, 57119u, 57131u, 57139u, 57143u, 
         57149u, 57163u, 57173u, 57179u, 57191u, 57193u, 57203u, 57221u, 57223u, 
         57241u, 57251u, 57259u, 57269u, 57271u, 57283u, 57287u, 57301u, 57329u, 
         57331u, 57347u, 57349u, 57367u, 57373u, 57383u, 57389u, 57397u, 57413u, 
         57427u, 57457u, 57467u, 57487u, 57493u, 57503u, 57527u, 57529u, 57557u, 
         57559u, 57571u, 57587u, 57593u, 57601u, 57637u, 57641u, 57649u, 57653u, 
         57667u, 57679u, 57689u, 57697u, 57709u, 57713u, 57719u, 57727u, 57731u, 
         57737u, 57751u, 57773u, 57781u, 57787u, 57791u, 57793u, 57803u, 57809u, 
         57829u, 57839u, 57847u, 57853u, 57859u, 57881u, 57899u, 57901u, 57917u, 
         57923u, 57943u, 57947u, 57973u, 57977u, 57991u, 58013u, 58027u, 58031u, 
         58043u, 58049u, 58057u, 58061u, 58067u, 58073u, 58099u, 58109u, 58111u, 
         58129u, 58147u, 58151u, 58153u, 58169u, 58171u, 58189u, 58193u, 58199u, 
         58207u, 58211u, 58217u, 58229u, 58231u, 58237u, 58243u, 58271u, 58309u, 
         58313u, 58321u, 58337u, 58363u, 58367u, 58369u, 58379u, 58391u, 58393u, 
         58403u, 58411u, 58417u, 58427u, 58439u, 58441u, 58451u, 58453u, 58477u, 
         58481u, 58511u, 58537u, 58543u, 58549u, 58567u, 58573u, 58579u, 58601u, 
         58603u, 58613u, 58631u, 58657u, 58661u, 58679u, 58687u, 58693u, 58699u, 
         58711u, 58727u, 58733u, 58741u, 58757u, 58763u, 58771u, 58787u, 58789u, 
         58831u, 58889u, 58897u, 58901u, 58907u, 58909u, 58913u, 58921u, 58937u, 
         58943u, 58963u, 58967u, 58979u, 58991u, 58997u, 59009u, 59011u, 59021u, 
         59023u, 59029u, 59051u, 59053u, 59063u, 59069u, 59077u, 59083u, 59093u, 
         59107u, 59113u, 59119u, 59123u, 59141u, 59149u, 59159u, 59167u, 59183u, 
         59197u, 59207u, 59209u, 59219u, 59221u, 59233u, 59239u, 59243u, 59263u, 
         59273u, 59281u, 59333u, 59341u, 59351u, 59357u, 59359u, 59369u, 59377u, 
         59387u, 59393u, 59399u, 59407u, 59417u, 59419u, 59441u, 59443u, 59447u, 
         59453u, 59467u, 59471u, 59473u, 59497u, 59509u, 59513u, 59539u, 59557u, 
         59561u, 59567u, 59581u, 59611u, 59617u, 59621u, 59627u, 59629u, 59651u, 
         59659u, 59663u, 59669u, 59671u, 59693u, 59699u, 59707u, 59723u, 59729u, 
         59743u, 59747u, 59753u, 59771u, 59779u, 59791u, 59797u, 59809u, 59833u, 
         59863u, 59879u, 59887u, 59921u, 59929u, 59951u, 59957u, 59971u, 59981u, 
         59999u, 60013u, 60017u, 60029u, 60037u, 60041u, 60077u, 60083u, 60089u, 
         60091u, 60101u, 60103u, 60107u, 60127u, 60133u, 60139u, 60149u, 60161u, 
         60167u, 60169u, 60209u, 60217u, 60223u, 60251u, 60257u, 60259u, 60271u, 
         60289u, 60293u, 60317u, 60331u, 60337u, 60343u, 60353u, 60373u, 60383u, 
         60397u, 60413u, 60427u, 60443u, 60449u, 60457u, 60493u, 60497u, 60509u, 
         60521u, 60527u, 60539u, 60589u, 60601u, 60607u, 60611u, 60617u, 60623u, 
         60631u, 60637u, 60647u, 60649u, 60659u, 60661u, 60679u, 60689u, 60703u, 
         60719u, 60727u, 60733u, 60737u, 60757u, 60761u, 60763u, 60773u, 60779u, 
         60793u, 60811u, 60821u, 60859u, 60869u, 60887u, 60889u, 60899u, 60901u, 
         60913u, 60917u, 60919u, 60923u, 60937u, 60943u, 60953u, 60961u, 61001u, 
         61007u, 61027u, 61031u, 61043u, 61051u, 61057u, 61091u, 61099u, 61121u, 
         61129u, 61141u, 61151u, 61153u, 61169u, 61211u, 61223u, 61231u, 61253u, 
         61261u, 61283u, 61291u, 61297u, 61331u, 61333u, 61339u, 61343u, 61357u, 
         61363u, 61379u, 61381u, 61403u, 61409u, 61417u, 61441u, 61463u, 61469u, 
         61471u, 61483u, 61487u, 61493u, 61507u, 61511u, 61519u, 61543u, 61547u, 
         61553u, 61559u, 61561u, 61583u, 61603u, 61609u, 61613u, 61627u, 61631u, 
         61637u, 61643u, 61651u, 61657u, 61667u, 61673u, 61681u, 61687u, 61703u, 
         61717u, 61723u, 61729u, 61751u, 61757u, 61781u, 61813u, 61819u, 61837u, 
         61843u, 61861u, 61871u, 61879u, 61909u, 61927u, 61933u, 61949u, 61961u, 
         61967u, 61979u, 61981u, 61987u, 61991u, 62003u, 62011u, 62017u, 62039u, 
         62047u, 62053u, 62057u, 62071u, 62081u, 62099u, 62119u, 62129u, 62131u, 
         62137u, 62141u, 62143u, 62171u, 62189u, 62191u, 62201u, 62207u, 62213u, 
         62219u, 62233u, 62273u, 62297u, 62299u, 62303u, 62311u, 62323u, 62327u, 
         62347u, 62351u, 62383u, 62401u, 62417u, 62423u, 62459u, 62467u, 62473u, 
         62477u, 62483u, 62497u, 62501u, 62507u, 62533u, 62539u, 62549u, 62563u, 
         62581u, 62591u, 62597u, 62603u, 62617u, 62627u, 62633u, 62639u, 62653u, 
         62659u, 62683u, 62687u, 62701u, 62723u, 62731u, 62743u, 62753u, 62761u, 
         62773u, 62791u, 62801u, 62819u, 62827u, 62851u, 62861u, 62869u, 62873u, 
         62897u, 62903u, 62921u, 62927u, 62929u, 62939u, 62969u, 62971u, 62981u, 
         62983u, 62987u, 62989u, 63029u, 63031u, 63059u, 63067u, 63073u, 63079u, 
         63097u, 63103u, 63113u, 63127u, 63131u, 63149u, 63179u, 63197u, 63199u, 
         63211u, 63241u, 63247u, 63277u, 63281u, 63299u, 63311u, 63313u, 63317u, 
         63331u, 63337u, 63347u, 63353u, 63361u, 63367u, 63377u, 63389u, 63391u, 
         63397u, 63409u, 63419u, 63421u, 63439u, 63443u, 63463u, 63467u, 63473u, 
         63487u, 63493u, 63499u, 63521u, 63527u, 63533u, 63541u, 63559u, 63577u, 
         63587u, 63589u, 63599u, 63601u, 63607u, 63611u, 63617u, 63629u, 63647u, 
         63649u, 63659u, 63667u, 63671u, 63689u, 63691u, 63697u, 63703u, 63709u, 
         63719u, 63727u, 63737u, 63743u, 63761u, 63773u, 63781u, 63793u, 63799u, 
         63803u, 63809u, 63823u, 63839u, 63841u, 63853u, 63857u, 63863u, 63901u, 
         63907u, 63913u, 63929u, 63949u, 63977u, 63997u, 64007u, 64013u, 64019u, 
         64033u, 64037u, 64063u, 64067u, 64081u, 64091u, 64109u, 64123u, 64151u, 
         64153u, 64157u, 64171u, 64187u, 64189u, 64217u, 64223u, 64231u, 64237u, 
         64271u, 64279u, 64283u, 64301u, 64303u, 64319u, 64327u, 64333u, 64373u, 
         64381u, 64399u, 64403u, 64433u, 64439u, 64451u, 64453u, 64483u, 64489u, 
         64499u, 64513u, 64553u, 64567u, 64577u, 64579u, 64591u, 64601u, 64609u, 
         64613u, 64621u, 64627u, 64633u, 64661u, 64663u, 64667u, 64679u, 64693u, 
         64709u, 64717u, 64747u, 64763u, 64781u, 64783u, 64793u, 64811u, 64817u, 
         64849u, 64853u, 64871u, 64877u, 64879u, 64891u, 64901u, 64919u, 64921u, 
         64927u, 64937u, 64951u, 64969u, 64997u, 65003u, 65011u, 65027u, 65029u, 
         65033u, 65053u, 65063u, 65071u, 65089u, 65099u, 65101u, 65111u, 65119u, 
         65123u, 65129u, 65141u, 65147u, 65167u, 65171u, 65173u, 65179u, 65183u, 
         65203u, 65213u, 65239u, 65257u, 65267u, 65269u, 65287u, 65293u, 65309u, 
         65323u, 65327u, 65353u, 65357u, 65371u, 65381u, 65393u, 65407u, 65413u, 
         65419u, 65423u, 65437u, 65447u, 65449u, 65479u, 65497u, 65519u, 65521u
      }};
#ifdef BOOST_MATH_HAVE_CONSTEXPR_TABLES
      constexpr std::array<boost::uint16_t, 3458> a3 = {{
#else
      static const boost::array<boost::uint16_t, 3458> a3 = {{
#endif
         2u, 4u, 8u, 16u, 22u, 28u, 44u, 
         46u, 52u, 64u, 74u, 82u, 94u, 98u, 112u, 
         116u, 122u, 142u, 152u, 164u, 166u, 172u, 178u, 
         182u, 184u, 194u, 196u, 226u, 242u, 254u, 274u, 
         292u, 296u, 302u, 304u, 308u, 316u, 332u, 346u, 
         364u, 386u, 392u, 394u, 416u, 422u, 428u, 446u, 
         448u, 458u, 494u, 502u, 506u, 512u, 532u, 536u, 
         548u, 554u, 568u, 572u, 574u, 602u, 626u, 634u, 
         638u, 644u, 656u, 686u, 704u, 736u, 758u, 766u, 
         802u, 808u, 812u, 824u, 826u, 838u, 842u, 848u, 
         868u, 878u, 896u, 914u, 922u, 928u, 932u, 956u, 
         964u, 974u, 988u, 994u, 998u, 1006u, 1018u, 1034u, 
         1036u, 1052u, 1058u, 1066u, 1082u, 1094u, 1108u, 1118u, 
         1148u, 1162u, 1166u, 1178u, 1186u, 1198u, 1204u, 1214u, 
         1216u, 1228u, 1256u, 1262u, 1274u, 1286u, 1306u, 1316u, 
         1318u, 1328u, 1342u, 1348u, 1354u, 1384u, 1388u, 1396u, 
         1408u, 1412u, 1414u, 1424u, 1438u, 1442u, 1468u, 1486u, 
         1498u, 1508u, 1514u, 1522u, 1526u, 1538u, 1544u, 1568u, 
         1586u, 1594u, 1604u, 1606u, 1618u, 1622u, 1634u, 1646u, 
         1652u, 1654u, 1676u, 1678u, 1682u, 1684u, 1696u, 1712u, 
         1726u, 1736u, 1738u, 1754u, 1772u, 1804u, 1808u, 1814u, 
         1834u, 1856u, 1864u, 1874u, 1876u, 1886u, 1892u, 1894u, 
         1898u, 1912u, 1918u, 1942u, 1946u, 1954u, 1958u, 1964u, 
         1976u, 1988u, 1996u, 2002u, 2012u, 2024u, 2032u, 2042u, 
         2044u, 2054u, 2066u, 2072u, 2084u, 2096u, 2116u, 2144u, 
         2164u, 2174u, 2188u, 2198u, 2206u, 2216u, 2222u, 2224u, 
         2228u, 2242u, 2248u, 2254u, 2266u, 2272u, 2284u, 2294u, 
         2308u, 2318u, 2332u, 2348u, 2356u, 2366u, 2392u, 2396u, 
         2398u, 2404u, 2408u, 2422u, 2426u, 2432u, 2444u, 2452u, 
         2458u, 2488u, 2506u, 2518u, 2524u, 2536u, 2552u, 2564u, 
         2576u, 2578u, 2606u, 2612u, 2626u, 2636u, 2672u, 2674u, 
         2678u, 2684u, 2692u, 2704u, 2726u, 2744u, 2746u, 2776u, 
         2794u, 2816u, 2836u, 2854u, 2864u, 2902u, 2908u, 2912u, 
         2914u, 2938u, 2942u, 2948u, 2954u, 2956u, 2966u, 2972u, 
         2986u, 2996u, 3004u, 3008u, 3032u, 3046u, 3062u, 3076u, 
         3098u, 3104u, 3124u, 3134u, 3148u, 3152u, 3164u, 3176u, 
         3178u, 3194u, 3202u, 3208u, 3214u, 3232u, 3236u, 3242u, 
         3256u, 3278u, 3284u, 3286u, 3328u, 3344u, 3346u, 3356u, 
         3362u, 3364u, 3368u, 3374u, 3382u, 3392u, 3412u, 3428u, 
         3458u, 3466u, 3476u, 3484u, 3494u, 3496u, 3526u, 3532u, 
         3538u, 3574u, 3584u, 3592u, 3608u, 3614u, 3616u, 3628u, 
         3656u, 3658u, 3662u, 3668u, 3686u, 3698u, 3704u, 3712u, 
         3722u, 3724u, 3728u, 3778u, 3782u, 3802u, 3806u, 3836u, 
         3844u, 3848u, 3854u, 3866u, 3868u, 3892u, 3896u, 3904u, 
         3922u, 3928u, 3932u, 3938u, 3946u, 3956u, 3958u, 3962u, 
         3964u, 4004u, 4022u, 4058u, 4088u, 4118u, 4126u, 4142u, 
         4156u, 4162u, 4174u, 4202u, 4204u, 4226u, 4228u, 4232u, 
         4244u, 4274u, 4286u, 4292u, 4294u, 4298u, 4312u, 4322u, 
         4324u, 4342u, 4364u, 4376u, 4394u, 4396u, 4406u, 4424u, 
         4456u, 4462u, 4466u, 4468u, 4474u, 4484u, 4504u, 4516u, 
         4526u, 4532u, 4544u, 4564u, 4576u, 4582u, 4586u, 4588u, 
         4604u, 4606u, 4622u, 4628u, 4642u, 4646u, 4648u, 4664u, 
         4666u, 4672u, 4688u, 4694u, 4702u, 4706u, 4714u, 4736u, 
         4754u, 4762u, 4774u, 4778u, 4786u, 4792u, 4816u, 4838u, 
         4844u, 4846u, 4858u, 4888u, 4894u, 4904u, 4916u, 4922u, 
         4924u, 4946u, 4952u, 4954u, 4966u, 4972u, 4994u, 5002u, 
         5014u, 5036u, 5038u, 5048u, 5054u, 5072u, 5084u, 5086u, 
         5092u, 5104u, 5122u, 5128u, 5132u, 5152u, 5174u, 5182u, 
         5194u, 5218u, 5234u, 5248u, 5258u, 5288u, 5306u, 5308u, 
         5314u, 5318u, 5332u, 5342u, 5344u, 5356u, 5366u, 5378u, 
         5384u, 5386u, 5402u, 5414u, 5416u, 5422u, 5434u, 5444u, 
         5446u, 5456u, 5462u, 5464u, 5476u, 5488u, 5504u, 5524u, 
         5534u, 5546u, 5554u, 5584u, 5594u, 5608u, 5612u, 5618u, 
         5626u, 5632u, 5636u, 5656u, 5674u, 5698u, 5702u, 5714u, 
         5722u, 5726u, 5728u, 5752u, 5758u, 5782u, 5792u, 5794u, 
         5798u, 5804u, 5806u, 5812u, 5818u, 5824u, 5828u, 5852u, 
         5854u, 5864u, 5876u, 5878u, 5884u, 5894u, 5902u, 5908u, 
         5918u, 5936u, 5938u, 5944u, 5948u, 5968u, 5992u, 6002u, 
         6014u, 6016u, 6028u, 6034u, 6058u, 6062u, 6098u, 6112u, 
         6128u, 6136u, 6158u, 6164u, 6172u, 6176u, 6178u, 6184u, 
         6206u, 6226u, 6242u, 6254u, 6272u, 6274u, 6286u, 6302u, 
         6308u, 6314u, 6326u, 6332u, 6344u, 6346u, 6352u, 6364u, 
         6374u, 6382u, 6398u, 6406u, 6412u, 6428u, 6436u, 6448u, 
         6452u, 6458u, 6464u, 6484u, 6496u, 6508u, 6512u, 6518u, 
         6538u, 6542u, 6554u, 6556u, 6566u, 6568u, 6574u, 6604u, 
         6626u, 6632u, 6634u, 6638u, 6676u, 6686u, 6688u, 6692u, 
         6694u, 6716u, 6718u, 6734u, 6736u, 6742u, 6752u, 6772u, 
         6778u, 6802u, 6806u, 6818u, 6832u, 6844u, 6848u, 6886u, 
         6896u, 6926u, 6932u, 6934u, 6946u, 6958u, 6962u, 6968u, 
         6998u, 7012u, 7016u, 7024u, 7042u, 7078u, 7082u, 7088u, 
         7108u, 7112u, 7114u, 7126u, 7136u, 7138u, 7144u, 7154u, 
         7166u, 7172u, 7184u, 7192u, 7198u, 7204u, 7228u, 7232u, 
         7262u, 7282u, 7288u, 7324u, 7334u, 7336u, 7348u, 7354u, 
         7358u, 7366u, 7372u, 7376u, 7388u, 7396u, 7402u, 7414u, 
         7418u, 7424u, 7438u, 7442u, 7462u, 7474u, 7478u, 7484u, 
         7502u, 7504u, 7508u, 7526u, 7528u, 7544u, 7556u, 7586u, 
         7592u, 7598u, 7606u, 7646u, 7654u, 7702u, 7708u, 7724u, 
         7742u, 7756u, 7768u, 7774u, 7792u, 7796u, 7816u, 7826u, 
         7828u, 7834u, 7844u, 7852u, 7882u, 7886u, 7898u, 7918u, 
         7924u, 7936u, 7942u, 7948u, 7982u, 7988u, 7994u, 8012u, 
         8018u, 8026u, 8036u, 8048u, 8054u, 8062u, 8072u, 8074u, 
         8078u, 8102u, 8108u, 8116u, 8138u, 8144u, 8146u, 8158u, 
         8164u, 8174u, 8186u, 8192u, 8216u, 8222u, 8236u, 8248u, 
         8284u, 8288u, 8312u, 8314u, 8324u, 8332u, 8342u, 8348u, 
         8362u, 8372u, 8404u, 8408u, 8416u, 8426u, 8438u, 8464u, 
         8482u, 8486u, 8492u, 8512u, 8516u, 8536u, 8542u, 8558u, 
         8564u, 8566u, 8596u, 8608u, 8614u, 8624u, 8626u, 8632u, 
         8642u, 8654u, 8662u, 8666u, 8668u, 8674u, 8684u, 8696u, 
         8722u, 8744u, 8752u, 8758u, 8762u, 8776u, 8782u, 8788u, 
         8818u, 8822u, 8828u, 8842u, 8846u, 8848u, 8876u, 8878u, 
         8884u, 8906u, 8914u, 8918u, 8936u, 8954u, 8972u, 8974u, 
         8986u, 8992u, 8996u, 9016u, 9026u, 9032u, 9038u, 9052u, 
         9062u, 9074u, 9076u, 9088u, 9118u, 9152u, 9164u, 9172u, 
         9178u, 9182u, 9184u, 9194u, 9196u, 9212u, 9224u, 9226u, 
         9236u, 9244u, 9262u, 9286u, 9292u, 9296u, 9308u, 9322u, 
         9326u, 9334u, 9338u, 9352u, 9356u, 9362u, 9368u, 9388u, 
         9394u, 9398u, 9406u, 9424u, 9476u, 9478u, 9482u, 9494u, 
         9502u, 9506u, 9544u, 9548u, 9574u, 9598u, 9614u, 9626u, 
         9632u, 9634u, 9646u, 9658u, 9674u, 9676u, 9682u, 9688u, 
         9692u, 9704u, 9718u, 9734u, 9742u, 9754u, 9772u, 9788u, 
         9794u, 9802u, 9812u, 9818u, 9832u, 9842u, 9854u, 9856u, 
         9866u, 9868u, 9872u, 9896u, 9902u, 9944u, 9968u, 9976u, 
         9986u, 9992u, 9998u, 10004u, 10006u, 10018u, 10022u, 10036u, 
         10042u, 10048u, 10076u, 10082u, 10084u, 10094u, 10106u, 10118u, 
         10124u, 10144u, 10148u, 10154u, 10168u, 10172u, 10174u, 10186u, 
         10196u, 10208u, 10232u, 10238u, 10246u, 10252u, 10258u, 10262u, 
         10286u, 10298u, 10318u, 10334u, 10348u, 10378u, 10396u, 10402u, 
         10406u, 10432u, 10444u, 10448u, 10454u, 10456u, 10462u, 10466u, 
         10468u, 10496u, 10504u, 10544u, 10546u, 10556u, 10564u, 10568u, 
         10588u, 10594u, 10612u, 10622u, 10624u, 10628u, 10672u, 10678u, 
         10696u, 10708u, 10714u, 10718u, 10724u, 10726u, 10748u, 10754u, 
         10768u, 10798u, 10808u, 10832u, 10834u, 10844u, 10852u, 10868u, 
         10886u, 10888u, 10906u, 10928u, 10936u, 10946u, 10952u, 10958u, 
         10972u, 10976u, 10984u, 11002u, 11006u, 11008u, 11026u, 11044u, 
         11062u, 11068u, 11072u, 11096u, 11114u, 11116u, 11132u, 11138u, 
         11144u, 11162u, 11182u, 11198u, 11218u, 11222u, 11236u, 11242u, 
         11246u, 11266u, 11284u, 11294u, 11296u, 11302u, 11312u, 11336u, 
         11338u, 11348u, 11372u, 11378u, 11384u, 11408u, 11414u, 11426u, 
         11428u, 11456u, 11468u, 11482u, 11488u, 11494u, 11506u, 11512u, 
         11534u, 11546u, 11558u, 11566u, 11602u, 11606u, 11618u, 11632u, 
         11636u, 11656u, 11666u, 11678u, 11702u, 11704u, 11708u, 11714u, 
         11726u, 11728u, 11732u, 11734u, 11744u, 11756u, 11782u, 11788u, 
         11804u, 11812u, 11816u, 11824u, 11834u, 11842u, 11848u, 11882u, 
         11884u, 11896u, 11912u, 11936u, 11942u, 11944u, 11954u, 11956u, 
         11974u, 11978u, 11986u, 11992u, 12008u, 12014u, 12016u, 12022u, 
         12028u, 12034u, 12038u, 12052u, 12056u, 12076u, 12082u, 12086u, 
         12106u, 12112u, 12124u, 12146u, 12152u, 12154u, 12164u, 12176u, 
         12178u, 12184u, 12188u, 12196u, 12208u, 12212u, 12226u, 12238u, 
         12248u, 12262u, 12266u, 12278u, 12304u, 12314u, 12328u, 12332u, 
         12358u, 12364u, 12394u, 12398u, 12416u, 12434u, 12442u, 12448u, 
         12464u, 12472u, 12482u, 12496u, 12506u, 12514u, 12524u, 12544u, 
         12566u, 12586u, 12602u, 12604u, 12622u, 12628u, 12632u, 12638u, 
         12644u, 12656u, 12658u, 12668u, 12694u, 12698u, 12706u, 12724u, 
         12742u, 12748u, 12766u, 12772u, 12776u, 12782u, 12806u, 12812u, 
         12832u, 12866u, 12892u, 12902u, 12904u, 12932u, 12944u, 12952u, 
         12962u, 12974u, 12976u, 12982u, 13004u, 13006u, 13018u, 13034u, 
         13036u, 13042u, 13048u, 13058u, 13072u, 13088u, 13108u, 13114u, 
         13118u, 13156u, 13162u, 13172u, 13178u, 13186u, 13202u, 13244u, 
         13246u, 13252u, 13256u, 13262u, 13268u, 13274u, 13288u, 13304u, 
         13318u, 13322u, 13342u, 13352u, 13354u, 13358u, 13366u, 13384u, 
         13394u, 13406u, 13442u, 13444u, 13454u, 13496u, 13504u, 13508u, 
         13528u, 13552u, 13568u, 13576u, 13598u, 13604u, 13612u, 13616u, 
         13618u, 13624u, 13646u, 13652u, 13658u, 13666u, 13694u, 13696u, 
         13706u, 13724u, 13738u, 13744u, 13748u, 13766u, 13774u, 13784u, 
         13798u, 13802u, 13814u, 13822u, 13832u, 13844u, 13858u, 13862u, 
         13864u, 13876u, 13888u, 13892u, 13898u, 13916u, 13946u, 13958u, 
         13996u, 14002u, 14014u, 14024u, 14026u, 14044u, 14054u, 14066u, 
         14074u, 14078u, 14086u, 14092u, 14096u, 14098u, 14122u, 14134u, 
         14152u, 14156u, 14158u, 14162u, 14164u, 14222u, 14234u, 14242u, 
         14266u, 14276u, 14278u, 14282u, 14288u, 14294u, 14306u, 14308u, 
         14312u, 14326u, 14332u, 14338u, 14354u, 14366u, 14368u, 14372u, 
         14404u, 14408u, 14432u, 14438u, 14444u, 14452u, 14462u, 14464u, 
         14486u, 14504u, 14516u, 14536u, 14542u, 14572u, 14576u, 14606u, 
         14612u, 14614u, 14618u, 14632u, 14638u, 14642u, 14656u, 14672u, 
         14674u, 14686u, 14696u, 14698u, 14704u, 14716u, 14728u, 14738u, 
         14744u, 14752u, 14774u, 14782u, 14794u, 14806u, 14812u, 14828u, 
         14834u, 14852u, 14872u, 14894u, 14912u, 14914u, 14936u, 14938u, 
         14954u, 14956u, 14978u, 14992u, 15002u, 15022u, 15032u, 15064u, 
         15068u, 15076u, 15086u, 15092u, 15094u, 15116u, 15122u, 15134u, 
         15136u, 15142u, 15146u, 15148u, 15152u, 15166u, 15178u, 15202u, 
         15212u, 15214u, 15226u, 15242u, 15244u, 15248u, 15254u, 15268u, 
         15274u, 15284u, 15296u, 15298u, 15314u, 15328u, 15362u, 15374u, 
         15376u, 15382u, 15388u, 15394u, 15398u, 15418u, 15428u, 15454u, 
         15466u, 15478u, 15482u, 15484u, 15488u, 15496u, 15506u, 15508u, 
         15512u, 15514u, 15536u, 15542u, 15548u, 15562u, 15566u, 15584u, 
         15596u, 15622u, 15628u, 15638u, 15646u, 15662u, 15664u, 15668u, 
         15688u, 15698u, 15704u, 15746u, 15748u, 15758u, 15764u, 15772u, 
         15796u, 15808u, 15814u, 15818u, 15824u, 15836u, 15838u, 15866u, 
         15874u, 15886u, 15904u, 15922u, 15928u, 15974u, 15982u, 15992u, 
         15998u, 16012u, 16016u, 16018u, 16024u, 16028u, 16034u, 16076u, 
         16084u, 16094u, 16102u, 16112u, 16114u, 16132u, 16136u, 16142u, 
         16154u, 16166u, 16168u, 16172u, 16192u, 16202u, 16214u, 16226u, 
         16234u, 16238u, 16264u, 16282u, 16304u, 16312u, 16318u, 16334u, 
         16348u, 16364u, 16366u, 16384u, 16394u, 16396u, 16402u, 16408u, 
         16418u, 16432u, 16436u, 16438u, 16468u, 16472u, 16474u, 16478u, 
         16486u, 16496u, 16502u, 16504u, 16516u, 16532u, 16538u, 16594u, 
         16604u, 16606u, 16618u, 16628u, 16636u, 16648u, 16654u, 16658u, 
         16672u, 16682u, 16684u, 16688u, 16696u, 16702u, 16706u, 16726u, 
         16732u, 16744u, 16766u, 16772u, 16804u, 16814u, 16816u, 16826u, 
         16838u, 16852u, 16858u, 16886u, 16922u, 16928u, 16934u, 16936u, 
         16948u, 16952u, 16958u, 16964u, 16972u, 16994u, 16996u, 17014u, 
         17024u, 17026u, 17032u, 17036u, 17056u, 17066u, 17074u, 17078u, 
         17084u, 17098u, 17116u, 17122u, 17164u, 17186u, 17188u, 17192u, 
         17194u, 17222u, 17224u, 17228u, 17246u, 17252u, 17258u, 17264u, 
         17276u, 17278u, 17302u, 17312u, 17348u, 17354u, 17356u, 17368u, 
         17378u, 17404u, 17428u, 17446u, 17462u, 17468u, 17474u, 17488u, 
         17512u, 17524u, 17528u, 17536u, 17542u, 17554u, 17558u, 17566u, 
         17582u, 17602u, 17642u, 17668u, 17672u, 17684u, 17686u, 17692u, 
         17696u, 17698u, 17708u, 17722u, 17732u, 17734u, 17738u, 17764u, 
         17776u, 17804u, 17806u, 17822u, 17848u, 17854u, 17864u, 17866u, 
         17872u, 17882u, 17888u, 17896u, 17902u, 17908u, 17914u, 17924u, 
         17936u, 17942u, 17962u, 18002u, 18022u, 18026u, 18028u, 18044u, 
         18056u, 18062u, 18074u, 18082u, 18086u, 18104u, 18106u, 18118u, 
         18128u, 18154u, 18166u, 18182u, 18184u, 18202u, 18226u, 18238u, 
         18242u, 18256u, 18278u, 18298u, 18308u, 18322u, 18334u, 18338u, 
         18356u, 18368u, 18376u, 18386u, 18398u, 18404u, 18434u, 18448u, 
         18452u, 18476u, 18482u, 18512u, 18518u, 18524u, 18526u, 18532u, 
         18554u, 18586u, 18592u, 18596u, 18602u, 18608u, 18628u, 18644u, 
         18646u, 18656u, 18664u, 18676u, 18686u, 18688u, 18694u, 18704u, 
         18712u, 18728u, 18764u, 18772u, 18778u, 18782u, 18784u, 18812u, 
         18814u, 18842u, 18854u, 18856u, 18866u, 18872u, 18886u, 18896u, 
         18902u, 18908u, 18914u, 18922u, 18928u, 18932u, 18946u, 18964u, 
         18968u, 18974u, 18986u, 18988u, 18998u, 19016u, 19024u, 19054u, 
         19094u, 19096u, 19114u, 19118u, 19124u, 19138u, 19156u, 19162u, 
         19166u, 19178u, 19184u, 19196u, 19202u, 19216u, 19226u, 19252u, 
         19258u, 19274u, 19276u, 19292u, 19322u, 19324u, 19334u, 19336u, 
         19378u, 19384u, 19412u, 19426u, 19432u, 19442u, 19444u, 19456u, 
         19474u, 19486u, 19492u, 19502u, 19514u, 19526u, 19546u, 19552u, 
         19556u, 19558u, 19568u, 19574u, 19586u, 19598u, 19612u, 19624u, 
         19658u, 19664u, 19666u, 19678u, 19688u, 19694u, 19702u, 19708u, 
         19712u, 19724u, 19762u, 19768u, 19778u, 19796u, 19798u, 19826u, 
         19828u, 19834u, 19846u, 19876u, 19892u, 19894u, 19904u, 19912u, 
         19916u, 19918u, 19934u, 19952u, 19978u, 19982u, 19988u, 19996u, 
         20014u, 20036u, 20042u, 20062u, 20066u, 20072u, 20084u, 20086u, 
         20092u, 20104u, 20108u, 20126u, 20132u, 20134u, 20156u, 20168u, 
         20176u, 20182u, 20198u, 20216u, 20246u, 20258u, 20282u, 20284u, 
         20294u, 20296u, 20302u, 20308u, 20312u, 20318u, 20354u, 20368u, 
         20374u, 20396u, 20398u, 20456u, 20464u, 20476u, 20482u, 20492u, 
         20494u, 20534u, 20542u, 20548u, 20576u, 20578u, 20582u, 20596u, 
         20602u, 20608u, 20626u, 20636u, 20644u, 20648u, 20662u, 20666u, 
         20674u, 20704u, 20708u, 20714u, 20722u, 20728u, 20734u, 20752u, 
         20756u, 20758u, 20762u, 20776u, 20788u, 20806u, 20816u, 20818u, 
         20822u, 20834u, 20836u, 20846u, 20854u, 20864u, 20878u, 20888u, 
         20906u, 20918u, 20926u, 20932u, 20942u, 20956u, 20966u, 20974u, 
         20996u, 20998u, 21004u, 21026u, 21038u, 21044u, 21052u, 21064u, 
         21092u, 21094u, 21142u, 21154u, 21158u, 21176u, 21184u, 21194u, 
         21208u, 21218u, 21232u, 21236u, 21248u, 21278u, 21302u, 21308u, 
         21316u, 21322u, 21326u, 21334u, 21388u, 21392u, 21394u, 21404u, 
         21416u, 21424u, 21434u, 21446u, 21458u, 21476u, 21478u, 21502u, 
         21506u, 21514u, 21536u, 21548u, 21568u, 21572u, 21584u, 21586u, 
         21598u, 21614u, 21616u, 21644u, 21646u, 21652u, 21676u, 21686u, 
         21688u, 21716u, 21718u, 21722u, 21742u, 21746u, 21758u, 21764u, 
         21778u, 21782u, 21788u, 21802u, 21824u, 21848u, 21868u, 21872u, 
         21886u, 21892u, 21898u, 21908u, 21938u, 21946u, 21956u, 21974u, 
         21976u, 21982u, 21988u, 22004u, 22006u, 22012u, 22018u, 22022u, 
         22024u, 22048u, 22052u, 22054u, 22078u, 22088u, 22094u, 22096u, 
         22106u, 22108u, 22114u, 22136u, 22144u, 22148u, 22156u, 22162u, 
         22166u, 22184u, 22186u, 22204u, 22208u, 22216u, 22232u, 22258u, 
         22262u, 22268u, 22276u, 22298u, 22318u, 22334u, 22342u, 22346u, 
         22352u, 22376u, 22382u, 22396u, 22408u, 22424u, 22426u, 22438u, 
         22442u, 22456u, 22466u, 22468u, 22472u, 22484u, 22502u, 22534u, 
         22544u, 22558u, 22582u, 22594u, 22634u, 22642u, 22676u, 22688u, 
         22702u, 22706u, 22724u, 22726u, 22754u, 22766u, 22786u, 22792u, 
         22802u, 22804u, 22844u, 22862u, 22876u, 22888u, 22892u, 22928u, 
         22934u, 22936u, 22958u, 22964u, 22978u, 22988u, 23012u, 23054u, 
         23056u, 23072u, 23074u, 23108u, 23116u, 23122u, 23126u, 23128u, 
         23132u, 23146u, 23186u, 23194u, 23206u, 23212u, 23236u, 23254u, 
         23258u, 23264u, 23266u, 23272u, 23276u, 23278u, 23282u, 23284u, 
         23308u, 23318u, 23326u, 23332u, 23338u, 23348u, 23362u, 23368u, 
         23384u, 23402u, 23416u, 23434u, 23458u, 23462u, 23468u, 23474u, 
         23482u, 23486u, 23506u, 23516u, 23522u, 23534u, 23536u, 23548u, 
         23552u, 23566u, 23572u, 23578u, 23584u, 23588u, 23602u, 23618u, 
         23654u, 23668u, 23674u, 23678u, 23692u, 23696u, 23702u, 23726u, 
         23734u, 23738u, 23758u, 23768u, 23782u, 23794u, 23828u, 23836u, 
         23846u, 23852u, 23858u, 23864u, 23878u, 23882u, 23896u, 23908u, 
         23914u, 23924u, 23942u, 23956u, 23966u, 23978u, 23984u, 23986u, 
         23992u, 23998u, 24026u, 24028u, 24032u, 24056u, 24062u, 24064u, 
         24068u, 24076u, 24092u, 24098u, 24118u, 24122u, 24124u, 24134u, 
         24136u, 24146u, 24154u, 24218u, 24224u, 24232u, 24244u, 24248u, 
         24262u, 24274u, 24284u, 24286u, 24298u, 24304u, 24314u, 24332u, 
         24356u, 24362u, 24364u, 24374u, 24382u, 24388u, 24404u, 24424u, 
         24428u, 24442u, 24448u, 24454u, 24466u, 24472u, 24476u, 24482u, 
         24484u, 24488u, 24496u, 24518u, 24524u, 24532u, 24536u, 24538u, 
         24554u, 24572u, 24586u, 24592u, 24614u, 24628u, 24638u, 24652u, 
         24656u, 24662u, 24664u, 24668u, 24682u, 24692u, 24704u, 24712u, 
         24728u, 24736u, 24746u, 24754u, 24778u, 24818u, 24824u, 24836u, 
         24838u, 24844u, 24862u, 24866u, 24868u, 24872u, 24902u, 24904u, 
         24934u, 24938u, 24946u, 24964u, 24976u, 24988u, 24992u, 24994u, 
         24998u, 25012u, 25048u, 25064u, 25082u, 25084u, 25096u, 25106u, 
         25112u, 25124u, 25142u, 25144u, 25162u, 25168u, 25174u, 25196u, 
         25214u, 25252u, 25258u, 25268u, 25286u, 25288u, 25298u, 25306u, 
         25312u, 25328u, 25352u, 25366u, 25372u, 25376u, 25382u, 25396u, 
         25412u, 25436u, 25442u, 25454u, 25462u, 25474u, 25484u, 25498u, 
         25544u, 25546u, 25562u, 25564u, 25586u, 25592u, 25594u, 25604u, 
         25606u, 25616u, 25618u, 25624u, 25628u, 25648u, 25658u, 25664u, 
         25694u, 25702u, 25708u, 25714u, 25718u, 25748u, 25756u, 25762u, 
         25768u, 25774u, 25796u, 25832u, 25834u, 25838u, 25846u, 25852u, 
         25858u, 25862u, 25876u, 25888u, 25898u, 25918u, 25922u, 25924u, 
         25928u, 25958u, 25964u, 25978u, 25994u, 26006u, 26036u, 26038u, 
         26042u, 26048u, 26056u, 26086u, 26096u, 26104u, 26138u, 26156u, 
         26168u, 26176u, 26198u, 26218u, 26222u, 26236u, 26246u, 26266u, 
         26272u, 26276u, 26278u, 26288u, 26302u, 26306u, 26332u, 26338u, 
         26374u, 26386u, 26404u, 26408u, 26416u, 26422u, 26426u, 26432u, 
         26434u, 26462u, 26468u, 26474u, 26498u, 26506u, 26516u, 26542u, 
         26548u, 26572u, 26576u, 26584u, 26608u, 26618u, 26638u, 26642u, 
         26644u, 26654u, 26668u, 26684u, 26686u, 26692u, 26698u, 26702u, 
         26708u, 26716u, 26734u, 26762u, 26776u, 26782u, 26798u, 26812u, 
         26818u, 26822u, 26828u, 26834u, 26842u, 26846u, 26848u, 26852u, 
         26864u, 26866u, 26878u, 26884u, 26896u, 26924u, 26926u, 26932u, 
         26944u, 26954u, 26968u, 26972u, 27016u, 27022u, 27032u, 27034u, 
         27046u, 27058u, 27088u, 27092u, 27104u, 27106u, 27112u, 27122u, 
         27134u, 27136u, 27146u, 27148u, 27158u, 27164u, 27172u, 27182u, 
         27188u, 27202u, 27218u, 27226u, 27232u, 27244u, 27254u, 27256u, 
         27266u, 27274u, 27286u, 27296u, 27314u, 27322u, 27326u, 27328u, 
         27332u, 27358u, 27364u, 27386u, 27392u, 27406u, 27416u, 27422u, 
         27424u, 27452u, 27458u, 27466u, 27512u, 27518u, 27524u, 27542u, 
         27548u, 27554u, 27562u, 27568u, 27578u, 27596u, 27598u, 27604u, 
         27616u, 27634u, 27644u, 27652u, 27664u, 27694u, 27704u, 27706u, 
         27716u, 27718u, 27722u, 27728u, 27746u, 27748u, 27752u, 27772u, 
         27784u, 27788u, 27794u, 27802u, 27836u, 27842u, 27848u, 27872u, 
         27884u, 27892u, 27928u, 27944u, 27946u, 27952u, 27956u, 27958u, 
         27962u, 27968u, 27988u, 27994u, 28018u, 28022u, 28024u, 28028u, 
         28046u, 28066u, 28072u, 28094u, 28102u, 28148u, 28166u, 28168u, 
         28184u, 28204u, 28226u, 28228u, 28252u, 28274u, 28276u, 28292u, 
         28316u, 28336u, 28352u, 28354u, 28358u, 28366u, 28376u, 28378u, 
         28388u, 28402u, 28406u, 28414u, 28432u, 28436u, 28444u, 28448u, 
         28462u, 28472u, 28474u, 28498u, 28514u, 28522u, 28528u, 28544u, 
         28564u, 28574u, 28576u, 28582u, 28586u, 28616u, 28618u, 28634u, 
         28666u, 28672u, 28684u, 28694u, 28718u, 28726u, 28738u, 28756u, 
         28772u, 28774u, 28786u, 28792u, 28796u, 28808u, 28814u, 28816u, 
         28844u, 28862u, 28864u, 28886u, 28892u, 28898u, 28904u, 28906u, 
         28912u, 28928u, 28942u, 28948u, 28978u, 28994u, 28996u, 29006u, 
         29008u, 29012u, 29024u, 29026u, 29038u, 29048u, 29062u, 29068u, 
         29078u, 29086u, 29114u, 29116u, 29152u, 29158u, 29174u, 29188u, 
         29192u, 29212u, 29236u, 29242u, 29246u, 29254u, 29258u, 29276u, 
         29284u, 29288u, 29302u, 29306u, 29312u, 29314u, 29338u, 29354u, 
         29368u, 29372u, 29398u, 29414u, 29416u, 29426u, 29458u, 29464u, 
         29468u, 29474u, 29486u, 29492u, 29528u, 29536u, 29548u, 29552u, 
         29554u, 29558u, 29566u, 29572u, 29576u, 29596u, 29608u, 29618u, 
         29642u, 29654u, 29656u, 29668u, 29678u, 29684u, 29696u, 29698u, 
         29704u, 29722u, 29726u, 29732u, 29738u, 29744u, 29752u, 29776u, 
         29782u, 29792u, 29804u, 29834u, 29848u, 29858u, 29866u, 29878u, 
         29884u, 29894u, 29906u, 29908u, 29926u, 29932u, 29936u, 29944u, 
         29948u, 29972u, 29992u, 29996u, 30004u, 30014u, 30026u, 30034u, 
         30046u, 30062u, 30068u, 30082u, 30086u, 30094u, 30098u, 30116u, 
         30166u, 30172u, 30178u, 30182u, 30188u, 30196u, 30202u, 30212u, 
         30238u, 30248u, 30254u, 30256u, 30266u, 30268u, 30278u, 30284u, 
         30322u, 30334u, 30338u, 30346u, 30356u, 30376u, 30382u, 30388u, 
         30394u, 30412u, 30422u, 30424u, 30436u, 30452u, 30454u, 30466u, 
         30478u, 30482u, 30508u, 30518u, 30524u, 30544u, 30562u, 30602u, 
         30614u, 30622u, 30632u, 30644u, 30646u, 30664u, 30676u, 30686u, 
         30688u, 30698u, 30724u, 30728u, 30734u, 30746u, 30754u, 30758u, 
         30788u, 30794u, 30796u, 30802u, 30818u, 30842u, 30866u, 30884u, 
         30896u, 30908u, 30916u, 30922u, 30926u, 30934u, 30944u, 30952u, 
         30958u, 30962u, 30982u, 30992u, 31018u, 31022u, 31046u, 31052u, 
         31054u, 31066u, 31108u, 31126u, 31132u, 31136u, 31162u, 31168u, 
         31196u, 31202u, 31204u, 31214u, 31222u, 31228u, 31234u, 31244u, 
         31252u, 31262u, 31264u, 31286u, 31288u, 31292u, 31312u, 31316u, 
         31322u, 31358u, 31372u, 31376u, 31396u, 31418u, 31424u, 31438u, 
         31444u, 31454u, 31462u, 31466u, 31468u, 31472u, 31486u, 31504u, 
         31538u, 31546u, 31568u, 31582u, 31592u, 31616u, 31622u, 31624u, 
         31634u, 31636u, 31642u, 31652u, 31678u, 31696u, 31706u, 31724u, 
         31748u, 31766u, 31768u, 31792u, 31832u, 31834u, 31838u, 31844u, 
         31846u, 31852u, 31862u, 31888u, 31894u, 31906u, 31918u, 31924u, 
         31928u, 31964u, 31966u, 31976u, 31988u, 32012u, 32014u, 32018u, 
         32026u, 32036u, 32042u, 32044u, 32048u, 32072u, 32074u, 32078u, 
         32114u, 32116u, 32138u, 32152u, 32176u, 32194u, 32236u, 32242u, 
         32252u, 32254u, 32278u, 32294u, 32306u, 32308u, 32312u, 32314u, 
         32324u, 32326u, 32336u, 32344u, 32348u, 32384u, 32392u, 32396u, 
         32408u, 32426u, 32432u, 32438u, 32452u, 32474u, 32476u, 32482u, 
         32506u, 32512u, 32522u, 32546u, 32566u, 32588u, 32594u, 32608u, 
         32644u, 32672u, 32678u, 32686u, 32692u, 32716u, 32722u, 32734u, 
         32762u, 32764u, 32782u, 32786u, 32788u, 32792u, 32812u, 32834u, 
         32842u, 32852u, 32854u, 32872u, 32876u, 32884u, 32894u, 32908u, 
         32918u, 32924u, 32932u, 32938u, 32944u, 32956u, 32972u, 32984u, 
         32998u, 33008u, 33026u, 33028u, 33038u, 33062u, 33086u, 33092u, 
         33104u, 33106u, 33128u, 33134u, 33154u, 33176u, 33178u, 33182u, 
         33194u, 33196u, 33202u, 33238u, 33244u, 33266u, 33272u, 33274u, 
         33302u, 33314u, 33332u, 33334u, 33338u, 33352u, 33358u, 33362u, 
         33364u, 33374u, 33376u, 33392u, 33394u, 33404u, 33412u, 33418u, 
         33428u, 33446u, 33458u, 33464u, 33478u, 33482u, 33488u, 33506u, 
         33518u, 33544u, 33548u, 33554u, 33568u, 33574u, 33584u, 33596u, 
         33598u, 33602u, 33604u, 33614u, 33638u, 33646u, 33656u, 33688u, 
         33698u, 33706u, 33716u, 33722u, 33724u, 33742u, 33754u, 33782u, 
         33812u, 33814u, 33832u, 33836u, 33842u, 33856u, 33862u, 33866u, 
         33874u, 33896u, 33904u, 33934u, 33952u, 33962u, 33988u, 33992u, 
         33994u, 34016u, 34024u, 34028u, 34036u, 34042u, 34046u, 34072u, 
         34076u, 34088u, 34108u, 34126u, 34132u, 34144u, 34154u, 34172u, 
         34174u, 34178u, 34184u, 34186u, 34198u, 34226u, 34232u, 34252u, 
         34258u, 34274u, 34282u, 34288u, 34294u, 34298u, 34304u, 34324u, 
         34336u, 34342u, 34346u, 34366u, 34372u, 34388u, 34394u, 34426u, 
         34436u, 34454u, 34456u, 34468u, 34484u, 34508u, 34514u, 34522u, 
         34534u, 34568u, 34574u, 34594u, 34616u, 34618u, 34634u, 34648u, 
         34654u, 34658u, 34672u, 34678u, 34702u, 34732u, 34736u, 34744u, 
         34756u, 34762u, 34778u, 34798u, 34808u, 34822u, 34826u, 34828u, 
         34844u, 34856u, 34858u, 34868u, 34876u, 34882u, 34912u, 34924u, 
         34934u, 34948u, 34958u, 34966u, 34976u, 34982u, 34984u, 34988u, 
         35002u, 35012u, 35014u, 35024u, 35056u, 35074u, 35078u, 35086u, 
         35114u, 35134u, 35138u, 35158u, 35164u, 35168u, 35198u, 35206u, 
         35212u, 35234u, 35252u, 35264u, 35266u, 35276u, 35288u, 35294u, 
         35312u, 35318u, 35372u, 35378u, 35392u, 35396u, 35402u, 35408u, 
         35422u, 35446u, 35452u, 35464u, 35474u, 35486u, 35492u, 35516u, 
         35528u, 35546u, 35554u, 35572u, 35576u, 35578u, 35582u, 35584u, 
         35606u, 35614u, 35624u, 35626u, 35638u, 35648u, 35662u, 35668u, 
         35672u, 35674u, 35686u, 35732u, 35738u, 35744u, 35746u, 35752u, 
         35758u, 35788u, 35798u, 35806u, 35812u, 35824u, 35828u, 35842u, 
         35848u, 35864u, 35876u, 35884u, 35894u, 35914u, 35932u, 35942u, 
         35948u, 35954u, 35966u, 35968u, 35978u, 35992u, 35996u, 35998u, 
         36002u, 36026u, 36038u, 36046u, 36064u, 36068u, 36076u, 36092u, 
         36106u, 36118u, 36128u, 36146u, 36158u, 36166u, 36184u, 36188u, 
         36202u, 36206u, 36212u, 36214u, 36236u, 36254u, 36262u, 36272u, 
         36298u, 36302u, 36304u, 36328u, 36334u, 36338u, 36344u, 36356u, 
         36382u, 36386u, 36394u, 36404u, 36422u, 36428u, 36442u, 36452u, 
         36464u, 36466u, 36478u, 36484u, 36488u, 36496u, 36508u, 36524u, 
         36526u, 36536u, 36542u, 36544u, 36566u, 36568u, 36572u, 36586u, 
         36604u, 36614u, 36626u, 36646u, 36656u, 36662u, 36664u, 36668u, 
         36682u, 36694u, 36698u, 36706u, 36716u, 36718u, 36724u, 36758u, 
         36764u, 36766u, 36782u, 36794u, 36802u, 36824u, 36832u, 36862u, 
         36872u, 36874u, 36898u, 36902u, 36916u, 36926u, 36946u, 36962u, 
         36964u, 36968u, 36988u, 36998u, 37004u, 37012u, 37016u, 37024u, 
         37028u, 37052u, 37058u, 37072u, 37076u, 37108u, 37112u, 37118u, 
         37132u, 37138u, 37142u, 37144u, 37166u, 37226u, 37228u, 37234u, 
         37258u, 37262u, 37276u, 37294u, 37306u, 37324u, 37336u, 37342u, 
         37346u, 37376u, 37378u, 37394u, 37396u, 37418u, 37432u, 37448u, 
         37466u, 37472u, 37508u, 37514u, 37532u, 37534u, 37544u, 37552u, 
         37556u, 37558u, 37564u, 37588u, 37606u, 37636u, 37642u, 37648u, 
         37682u, 37696u, 37702u, 37754u, 37756u, 37772u, 37784u, 37798u, 
         37814u, 37822u, 37852u, 37856u, 37858u, 37864u, 37874u, 37886u, 
         37888u, 37916u, 37922u, 37936u, 37948u, 37976u, 37994u, 38014u, 
         38018u, 38026u, 38032u, 38038u, 38042u, 38048u, 38056u, 38078u, 
         38084u, 38108u, 38116u, 38122u, 38134u, 38146u, 38152u, 38164u, 
         38168u, 38188u, 38234u, 38252u, 38266u, 38276u, 38278u, 38302u, 
         38306u, 38308u, 38332u, 38354u, 38368u, 38378u, 38384u, 38416u, 
         38428u, 38432u, 38434u, 38444u, 38446u, 38456u, 38458u, 38462u, 
         38468u, 38474u, 38486u, 38498u, 38512u, 38518u, 38524u, 38552u, 
         38554u, 38572u, 38578u, 38584u, 38588u, 38612u, 38614u, 38626u, 
         38638u, 38644u, 38648u, 38672u, 38696u, 38698u, 38704u, 38708u, 
         38746u, 38752u, 38762u, 38774u, 38776u, 38788u, 38792u, 38812u, 
         38834u, 38846u, 38848u, 38858u, 38864u, 38882u, 38924u, 38936u, 
         38938u, 38944u, 38956u, 38978u, 38992u, 39002u, 39008u, 39014u, 
         39016u, 39026u, 39044u, 39058u, 39062u, 39088u, 39104u, 39116u, 
         39124u, 39142u, 39146u, 39148u, 39158u, 39166u, 39172u, 39176u, 
         39182u, 39188u, 39194u
      }};

      if(n <= b1)
         return a1[n];
      if(n <= b2)
         return a2[n - b1 - 1];
      if(n >= b3)
      {
         return boost::math::policies::raise_domain_error<boost::uint32_t>(
            "boost::math::prime<%1%>", "Argument n out of range: got %1%", n, pol);
      }
      return static_cast<boost::uint32_t>(a3[n - b2 - 1]) + 0xFFFFu;
   }

   inline BOOST_MATH_CONSTEXPR_TABLE_FUNCTION boost::uint32_t prime(unsigned n)
   {
      return boost::math::prime(n, boost::math::policies::policy<>());
   }

   static const unsigned max_prime = 10000;

}} // namespace boost and math

#endif // BOOST_MATH_SF_PRIME_HPP
