/*
 * Copyright (c) 2016 Samsung Electronics Co., Ltd All Rights Reserved
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
/**
 * @file        test/common/cache/monitorcache.cpp
 * @author      Aleksander Zdyb <a.zdyb@samsung.com>
 * @version     1.0
 * @brief       Tests of MonitorCache
 */

#include <gtest/gtest.h>
#include <gmock/gmock.h>

#include <ctime>

#include "../../helpers.h"

#include <common/types/MonitorEntry.h>
#include <client-common/cache/MonitorCache.h>

using namespace Cynara;

namespace {
    int constClock(int, struct timespec *ts) {
        ts->tv_sec = 42;
        ts->tv_nsec = 666;
        return 0;
    }

    int halfMaxAgeIncrementClock(int, struct timespec *ts) {
        static unsigned callCount = 1;
        ts->tv_sec = callCount * (MonitorCache::MAX_LOG_AGE_SEC / 2);
        ts->tv_nsec = 0;
        ++callCount;
        return 0;
    }
}

TEST(MonitorCache, addOne) {
    using ::testing::ElementsAre;

    MonitorCache monitorCache(constClock);

    MonitorEntry testEntry{{"c", "u", "p"}, 1, {42, 666}};
    monitorCache.update(testEntry.key(), testEntry.result());
    ASSERT_THAT(monitorCache.entries(), ElementsAre(testEntry));
}

TEST(MonitorCache, clear) {
    using ::testing::ElementsAre;
    using ::testing::IsEmpty;

    MonitorCache monitorCache(constClock);

    MonitorEntry testEntry{{"c", "u", "p"}, 1, {42, 666}};
    monitorCache.update(testEntry.key(), testEntry.result());
    ASSERT_THAT(monitorCache.entries(), ElementsAre(testEntry));
    monitorCache.clear();
    ASSERT_THAT(monitorCache.entries(), IsEmpty());
}

TEST(MonitorCache, shouldFlushAge) {
    MonitorCache monitorCache(halfMaxAgeIncrementClock);

    monitorCache.update({"c", "u", "p"}, 1);
    ASSERT_FALSE(monitorCache.shouldFlush());
    monitorCache.update({"c", "u", "p"}, 1);
    ASSERT_FALSE(monitorCache.shouldFlush());
    monitorCache.update({"c", "u", "p"}, 1);
    ASSERT_TRUE(monitorCache.shouldFlush());
}

TEST(MonitorCache, shouldFlushCapacity) {
    MonitorCache monitorCache(constClock);

    for (std::size_t i = 0; i < MonitorCache::CACHE_CAPACITY - 1; ++i)
        monitorCache.update({"c", "u", "p"}, 1);

    ASSERT_FALSE(monitorCache.shouldFlush());

    monitorCache.update({"c", "u", "p"}, 1);
    ASSERT_TRUE(monitorCache.shouldFlush());
}