// Copyright 2010 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CC_LAYERS_TEXTURE_LAYER_H_
#define CC_LAYERS_TEXTURE_LAYER_H_

#include <string>

#include "base/callback.h"
#include "base/macros.h"
#include "base/synchronization/lock.h"
#include "base/threading/thread_checker.h"
#include "cc/cc_export.h"
#include "cc/layers/layer.h"
#include "components/viz/common/resources/transferable_resource.h"

namespace gpu {
struct SyncToken;
}

namespace viz {
class SingleReleaseCallback;
}

namespace cc {
class SingleReleaseCallback;
class TextureLayerClient;

// A Layer containing a the rendered output of a plugin instance.
class CC_EXPORT TextureLayer : public Layer {
 public:
  class CC_EXPORT TransferableResourceHolder
      : public base::RefCountedThreadSafe<TransferableResourceHolder> {
   public:
    class CC_EXPORT MainThreadReference {
     public:
      explicit MainThreadReference(TransferableResourceHolder* holder);
      ~MainThreadReference();
      TransferableResourceHolder* holder() { return holder_.get(); }

     private:
      scoped_refptr<TransferableResourceHolder> holder_;
      DISALLOW_COPY_AND_ASSIGN(MainThreadReference);
    };

    const viz::TransferableResource& resource() const { return resource_; }
    void Return(const gpu::SyncToken& sync_token, bool is_lost);

    // Gets a viz::ReleaseCallback that can be called from another thread. Note:
    // the caller must ensure the callback is called.
    std::unique_ptr<viz::SingleReleaseCallback> GetCallbackForImplThread(
        scoped_refptr<base::SequencedTaskRunner> main_thread_task_runner);

   protected:
    friend class TextureLayer;

    // Protected visiblity so only TextureLayer and unit tests can create these.
    static std::unique_ptr<MainThreadReference> Create(
        const viz::TransferableResource& resource,
        std::unique_ptr<viz::SingleReleaseCallback> release_callback);
    virtual ~TransferableResourceHolder();

   private:
    friend class base::RefCountedThreadSafe<TransferableResourceHolder>;
    friend class MainThreadReference;
    explicit TransferableResourceHolder(
        const viz::TransferableResource& resource,
        std::unique_ptr<viz::SingleReleaseCallback> release_callback);

    void InternalAddRef();
    void InternalRelease();
    void ReturnAndReleaseOnImplThread(
        const scoped_refptr<base::SequencedTaskRunner>& main_thread_task_runner,
        const gpu::SyncToken& sync_token,
        bool is_lost);

    // These members are only accessed on the main thread, or on the impl thread
    // during commit where the main thread is blocked.
    unsigned internal_references_;
    viz::TransferableResource resource_;
    std::unique_ptr<viz::SingleReleaseCallback> release_callback_;

    // This lock guards the sync_token_ and is_lost_ fields because they can be
    // accessed on both the impl and main thread. We do this to ensure that the
    // values of these fields are well-ordered such that the last call to
    // ReturnAndReleaseOnImplThread() defines their values.
    base::Lock arguments_lock_;
    gpu::SyncToken sync_token_;
    bool is_lost_;
    base::ThreadChecker main_thread_checker_;
    DISALLOW_COPY_AND_ASSIGN(TransferableResourceHolder);
  };

  // Used when mailbox names are specified instead of texture IDs.
  static scoped_refptr<TextureLayer> CreateForMailbox(
      TextureLayerClient* client);

  // Resets the client, which also resets the texture.
  void ClearClient();

  // Resets the texture.
  void ClearTexture();

  std::unique_ptr<LayerImpl> CreateLayerImpl(LayerTreeImpl* tree_impl) override;

  // Sets whether this texture should be Y-flipped at draw time. Defaults to
  // true.
  void SetFlipped(bool flipped);
  bool flipped() const { return flipped_; }

  // Sets whether this texture should use nearest neighbor interpolation as
  // opposed to bilinear. Defaults to false.
  void SetNearestNeighbor(bool nearest_neighbor);

  // Sets a UV transform to be used at draw time. Defaults to (0, 0) and (1, 1).
  void SetUV(const gfx::PointF& top_left, const gfx::PointF& bottom_right);

  // Sets an opacity value per vertex. It will be multiplied by the layer
  // opacity value.
  void SetVertexOpacity(float bottom_left,
                        float top_left,
                        float top_right,
                        float bottom_right);

  // Sets whether the alpha channel is premultiplied or unpremultiplied.
  // Defaults to true.
  void SetPremultipliedAlpha(bool premultiplied_alpha);

  // Sets whether the texture should be blended with the background color
  // at draw time. Defaults to false.
  void SetBlendBackgroundColor(bool blend);

  // Code path for plugins which supply their own mailbox.
  void SetTransferableResource(
      const viz::TransferableResource& resource,
      std::unique_ptr<viz::SingleReleaseCallback> release_callback);

  void SetNeedsDisplayRect(const gfx::Rect& dirty_rect) override;

  void SetLayerTreeHost(LayerTreeHost* layer_tree_host) override;
  bool Update() override;
  bool IsSnapped() override;
  void PushPropertiesTo(LayerImpl* layer) override;

 protected:
  explicit TextureLayer(TextureLayerClient* client);
  ~TextureLayer() override;
  bool HasDrawableContent() const override;

 private:
  void SetTransferableResourceInternal(
      const viz::TransferableResource& resource,
      std::unique_ptr<viz::SingleReleaseCallback> release_callback,
      bool requires_commit);

  TextureLayerClient* client_;

  bool flipped_ = true;
  bool nearest_neighbor_ = false;
  gfx::PointF uv_top_left_ = gfx::PointF();
  gfx::PointF uv_bottom_right_ = gfx::PointF(1.f, 1.f);
  // [bottom left, top left, top right, bottom right]
  float vertex_opacity_[4] = {1.f, 1.f, 1.f, 1.f};
  bool premultiplied_alpha_ = true;
  bool blend_background_color_ = false;

  std::unique_ptr<TransferableResourceHolder::MainThreadReference> holder_ref_;
  bool needs_set_resource_ = false;

  DISALLOW_COPY_AND_ASSIGN(TextureLayer);
};

}  // namespace cc
#endif  // CC_LAYERS_TEXTURE_LAYER_H_
