/*
 * Copyright (c) 2019 TOYOTA MOTOR CORPORATION
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "hs-proxy.h"
#include "hmi-debug.h"

const char _afm_main[] = "afm-main";


/**
 * the callback function
 *
 * #### Parameters
 *  - closure : the user defined closure pointer 'closure'
 *  - object : a JSON object returned (can be NULL)
 *  - error : a string not NULL in case of error but NULL on success
 *  - info : a string handling some info (can be NULL)
 *  - api : the api
 *
 * #### Return
 *  None
 *
 */
static void api_callback(void *closure, struct json_object *object, const char *error, const char *info, afb_api_t api)
{
    HMI_DEBUG("homescreen-service","asynchronous call, error=%s, info=%s, object=%s.", error, info, json_object_get_string(object));
}

/**
 * call api asynchronous
 *
 * #### Parameters
 *  - api : the api serving the request
 *  - service : the api name of service
 *  - verb : the verb of service
 *  - args : parameter
 *
 * #### Return
 *  None
 *
 */
static void api_call(afb_api_t api, const char *service, const char *verb, struct json_object *args)
{
    HMI_DEBUG("homescreen-service","service=%s verb=%s, args=%s.", service, verb, json_object_get_string(args));
    afb_api_call(api, service, verb, args, api_callback, nullptr);
}

/**
 * call api synchronous
 *
 * #### Parameters
 *  - api : the api serving the request
 *  - service : the api name of service
 *  - verb : the verb of afm-main
 *  - args : parameter
 *  - object : return the details of application
 *
 * #### Return
 *  0 : success
 *  1 : fail
 *
 */
static int api_call_sync(afb_api_t api, const char *service, const char *verb, struct json_object *args, struct json_object **object)
{
    char *error = nullptr, *info = nullptr;
    int ret = afb_api_call_sync(api, service, verb, args, object, &error, &info);
    HMI_DEBUG("homescreen-service","synchronous call, error=%s, info=%s.", error, info);
    return ret;
}

/**
 * get runnables application list
 *
 * #### Parameters
 *  - api : the api serving the request
 *  - object : return the details of appid
 *
 * #### Return
 *  0 : success
 *  1 : fail
 *
 */
int HS_AfmMainProxy::runnables(afb_api_t api, struct json_object **object)
{
    return api_call_sync(api, _afm_main, __FUNCTION__, nullptr, object);
}

/**
 * get details of application
 *
 * #### Parameters
 *  - api : the api serving the request
 *  - id : the id to get details,liked "dashboard@0.1"
 *  - object : return the details of application
 *
 * #### Return
 *  0 : success
 *  1 : fail
 *
 */
int HS_AfmMainProxy::detail(afb_api_t api, const std::string &id, struct json_object **object)
{
    struct json_object *args = json_object_new_string(id.c_str());
    return api_call_sync(api, _afm_main, __FUNCTION__, args, object);
}

/**
 * start application
 *
 * #### Parameters
 *  - request : the request
 *  - id : the application id liked "dashboard@0.1"
 *
 * #### Return
 *  None
 *
 */
void HS_AfmMainProxy::start(afb_req_t request, const std::string &id)
{
    struct json_object *args = json_object_new_string(id.c_str());
    api_call(request->api, _afm_main, __FUNCTION__, args);
}