/*
 * Copyright (C) 2016 Red Hat, Inc.
 *
 * Author: Nikos Mavrogiannopoulos
 *
 * This file is part of GnuTLS.
 *
 * GnuTLS is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * GnuTLS is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GnuTLS; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <gnutls/gnutls.h>
#include <gnutls/x509.h>

#ifdef ENABLE_OCSP

#include "cert-common.h"
#include "utils.h"

/* Tests whether setting an OCSP response to a server 
 * is working as expected */

static time_t mytime(time_t * t)
{
	time_t then = 1469186559;
	if (t)
		*t = then;

	return then;
}

static const unsigned char _resp[] = {
0x30, 0x82, 0x06, 0x45, 0x0A, 0x01, 0x00, 0xA0,
0x82, 0x06, 0x3E, 0x30, 0x82, 0x06, 0x3A, 0x06,
0x09, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30,
0x01, 0x01, 0x04, 0x82, 0x06, 0x2B, 0x30, 0x82,
0x06, 0x27, 0x30, 0x81, 0x9E, 0xA2, 0x16, 0x04,
0x14, 0x1E, 0xA5, 0xBD, 0xCA, 0x59, 0x64, 0x55,
0x85, 0xAC, 0xDA, 0x54, 0x34, 0x23, 0x40, 0xD1,
0xF6, 0xBD, 0xC3, 0xB0, 0xF6, 0x18, 0x0F, 0x32,
0x30, 0x31, 0x37, 0x31, 0x31, 0x31, 0x39, 0x30,
0x39, 0x34, 0x33, 0x34, 0x37, 0x5A, 0x30, 0x73,
0x30, 0x71, 0x30, 0x49, 0x30, 0x09, 0x06, 0x05,
0x2B, 0x0E, 0x03, 0x02, 0x1A, 0x05, 0x00, 0x04,
0x14, 0xD1, 0xB1, 0x64, 0x8B, 0x8C, 0x9F, 0x0D,
0xD1, 0x6B, 0xA3, 0x8A, 0xCD, 0x2B, 0x50, 0x17,
0xD5, 0xF9, 0xCF, 0xC0, 0x64, 0x04, 0x14, 0x5F,
0x60, 0xCF, 0x61, 0x90, 0x55, 0xDF, 0x84, 0x43,
0x14, 0x8A, 0x60, 0x2A, 0xB2, 0xF5, 0x7A, 0xF4,
0x43, 0x18, 0xEF, 0x02, 0x10, 0x28, 0x2E, 0x96,
0xB3, 0x6B, 0x76, 0xD6, 0xD8, 0x52, 0x46, 0xED,
0xBB, 0x31, 0xB2, 0x0C, 0x98, 0x80, 0x00, 0x18,
0x0F, 0x32, 0x30, 0x31, 0x37, 0x31, 0x31, 0x31,
0x39, 0x30, 0x39, 0x34, 0x33, 0x34, 0x37, 0x5A,
0xA0, 0x11, 0x18, 0x0F, 0x32, 0x30, 0x31, 0x37,
0x31, 0x31, 0x32, 0x36, 0x30, 0x39, 0x34, 0x33,
0x34, 0x37, 0x5A, 0x30, 0x0D, 0x06, 0x09, 0x2A,
0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x05,
0x05, 0x00, 0x03, 0x82, 0x01, 0x01, 0x00, 0x54,
0x66, 0x9D, 0x96, 0x6B, 0x9D, 0x71, 0x18, 0x86,
0x90, 0x5D, 0xD9, 0x54, 0x9C, 0xC4, 0x7F, 0x18,
0x51, 0xE9, 0xFE, 0xF3, 0xE6, 0x48, 0x60, 0x89,
0x74, 0xFD, 0xF1, 0x6D, 0xDB, 0x1F, 0x5A, 0x28,
0x3D, 0x16, 0xEA, 0xA6, 0xD6, 0xE0, 0xAA, 0x42,
0xF9, 0x5B, 0x76, 0xA1, 0x59, 0xDA, 0x30, 0x8D,
0x08, 0x18, 0xDD, 0x60, 0x39, 0x0B, 0x90, 0x64,
0x11, 0x1E, 0x9D, 0xA2, 0x70, 0x18, 0xAD, 0xC6,
0x27, 0xD3, 0xF1, 0xBA, 0x11, 0x4E, 0xF6, 0x9D,
0x6C, 0xC5, 0xEB, 0xD6, 0xB7, 0x43, 0x9D, 0x32,
0x31, 0xC9, 0x24, 0x19, 0xB9, 0x47, 0x1C, 0x61,
0x09, 0x8F, 0xAA, 0x42, 0x5B, 0xAF, 0x66, 0x0F,
0x23, 0xAA, 0x80, 0xC0, 0x85, 0x7F, 0x00, 0x08,
0xCA, 0x30, 0xE4, 0xC8, 0xDA, 0x2F, 0xC4, 0xD2,
0x7E, 0x86, 0xCC, 0xDA, 0x6D, 0xD4, 0x7E, 0x40,
0x66, 0xD8, 0x5C, 0x27, 0x83, 0xDA, 0x10, 0x8F,
0x91, 0xA8, 0xE6, 0x9D, 0x44, 0x13, 0xF1, 0x04,
0x4E, 0xC9, 0xF9, 0xC8, 0xA2, 0xED, 0x9C, 0x9F,
0x05, 0xDA, 0xFA, 0x4A, 0xEA, 0xD2, 0x72, 0xF9,
0xF1, 0xF6, 0xDB, 0xFF, 0xF8, 0x55, 0x0E, 0x92,
0x75, 0xD6, 0x83, 0xBC, 0x7A, 0x95, 0xBE, 0xBF,
0x8D, 0xD5, 0xA3, 0x23, 0x02, 0x32, 0xF8, 0x60,
0xF7, 0x7C, 0x46, 0xC6, 0x69, 0x7E, 0xB7, 0x23,
0xE1, 0x36, 0xC2, 0xEE, 0xBD, 0xFF, 0x3C, 0x05,
0x5E, 0x07, 0x0C, 0xA6, 0x64, 0x65, 0x82, 0x46,
0xC9, 0x67, 0x73, 0xC9, 0x15, 0xC8, 0xFA, 0x0F,
0x73, 0xB5, 0x48, 0x0F, 0x0E, 0x6F, 0x43, 0xE8,
0x8D, 0x7A, 0x21, 0x88, 0x12, 0x08, 0x37, 0x18,
0x67, 0x66, 0x05, 0xD1, 0x2C, 0x4D, 0xE8, 0xA6,
0x1B, 0x4D, 0x29, 0xD4, 0xEF, 0x79, 0x83, 0xDB,
0xCA, 0x6E, 0xBC, 0xE4, 0xCA, 0x50, 0xB0, 0x73,
0xEF, 0xD6, 0xC7, 0x69, 0xF6, 0x16, 0x1E, 0xA0,
0x82, 0x04, 0x6E, 0x30, 0x82, 0x04, 0x6A, 0x30,
0x82, 0x04, 0x66, 0x30, 0x82, 0x03, 0x4E, 0xA0,
0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x6F, 0x93,
0x87, 0x5C, 0x4B, 0x9E, 0x94, 0x93, 0xF8, 0x5F,
0x16, 0xA7, 0x05, 0x86, 0x82, 0x8C, 0x30, 0x0D,
0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D,
0x01, 0x01, 0x0B, 0x05, 0x00, 0x30, 0x7E, 0x31,
0x0B, 0x30, 0x09, 0x06, 0x03, 0x55, 0x04, 0x06,
0x13, 0x02, 0x55, 0x53, 0x31, 0x1D, 0x30, 0x1B,
0x06, 0x03, 0x55, 0x04, 0x0A, 0x13, 0x14, 0x53,
0x79, 0x6D, 0x61, 0x6E, 0x74, 0x65, 0x63, 0x20,
0x43, 0x6F, 0x72, 0x70, 0x6F, 0x72, 0x61, 0x74,
0x69, 0x6F, 0x6E, 0x31, 0x1F, 0x30, 0x1D, 0x06,
0x03, 0x55, 0x04, 0x0B, 0x13, 0x16, 0x53, 0x79,
0x6D, 0x61, 0x6E, 0x74, 0x65, 0x63, 0x20, 0x54,
0x72, 0x75, 0x73, 0x74, 0x20, 0x4E, 0x65, 0x74,
0x77, 0x6F, 0x72, 0x6B, 0x31, 0x2F, 0x30, 0x2D,
0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x26, 0x53,
0x79, 0x6D, 0x61, 0x6E, 0x74, 0x65, 0x63, 0x20,
0x43, 0x6C, 0x61, 0x73, 0x73, 0x20, 0x33, 0x20,
0x53, 0x65, 0x63, 0x75, 0x72, 0x65, 0x20, 0x53,
0x65, 0x72, 0x76, 0x65, 0x72, 0x20, 0x43, 0x41,
0x20, 0x2D, 0x20, 0x47, 0x34, 0x30, 0x1E, 0x17,
0x0D, 0x31, 0x37, 0x31, 0x30, 0x31, 0x30, 0x30,
0x30, 0x30, 0x30, 0x30, 0x30, 0x5A, 0x17, 0x0D,
0x31, 0x38, 0x30, 0x31, 0x30, 0x38, 0x32, 0x33,
0x35, 0x39, 0x35, 0x39, 0x5A, 0x30, 0x40, 0x31,
0x3E, 0x30, 0x3C, 0x06, 0x03, 0x55, 0x04, 0x03,
0x13, 0x35, 0x53, 0x79, 0x6D, 0x61, 0x6E, 0x74,
0x65, 0x63, 0x20, 0x43, 0x6C, 0x61, 0x73, 0x73,
0x20, 0x33, 0x20, 0x53, 0x65, 0x63, 0x75, 0x72,
0x65, 0x20, 0x53, 0x65, 0x72, 0x76, 0x65, 0x72,
0x20, 0x43, 0x41, 0x20, 0x2D, 0x20, 0x47, 0x34,
0x20, 0x4F, 0x43, 0x53, 0x50, 0x20, 0x52, 0x65,
0x73, 0x70, 0x6F, 0x6E, 0x64, 0x65, 0x72, 0x30,
0x82, 0x01, 0x22, 0x30, 0x0D, 0x06, 0x09, 0x2A,
0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01,
0x05, 0x00, 0x03, 0x82, 0x01, 0x0F, 0x00, 0x30,
0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00,
0xBA, 0xB1, 0x3E, 0xBD, 0xF0, 0x1E, 0x19, 0x16,
0xEA, 0x20, 0x44, 0x73, 0x1F, 0xD8, 0x85, 0x17,
0xC4, 0xBF, 0x86, 0xF0, 0x75, 0x46, 0x02, 0xA8,
0x5B, 0x7F, 0xA8, 0xF8, 0xB2, 0x08, 0x08, 0x55,
0x01, 0xDD, 0x5B, 0xA4, 0x0B, 0xBD, 0x8A, 0x0A,
0x87, 0x90, 0x62, 0x21, 0x59, 0x67, 0x33, 0x36,
0x77, 0x49, 0xAB, 0x69, 0x4B, 0xDB, 0xB8, 0xFC,
0x27, 0xA9, 0x81, 0x4A, 0x1F, 0x5F, 0x7D, 0x5C,
0xC2, 0xE6, 0x54, 0x12, 0xFB, 0xA7, 0xEB, 0x9F,
0xB5, 0xAC, 0x05, 0xBE, 0xA9, 0x58, 0xAA, 0x49,
0x32, 0xEE, 0x73, 0xE8, 0x2F, 0xB1, 0xD3, 0x2E,
0x13, 0xBC, 0x26, 0x23, 0xA0, 0x82, 0xD4, 0x25,
0x20, 0x34, 0xAE, 0x16, 0x48, 0xFB, 0x55, 0x2B,
0x58, 0xC9, 0xC4, 0x84, 0xAC, 0xF7, 0xC4, 0x78,
0x62, 0xB7, 0xBF, 0xA2, 0x32, 0xC7, 0x34, 0x1C,
0xDF, 0x9E, 0xFE, 0xA8, 0x04, 0x85, 0xAF, 0xCB,
0x5A, 0xD6, 0xC6, 0x68, 0x9F, 0x28, 0x03, 0xB7,
0x98, 0x8E, 0xD4, 0xA5, 0xE1, 0x18, 0xD1, 0x64,
0x79, 0x67, 0x04, 0x33, 0x6C, 0x4B, 0xE0, 0xCF,
0x34, 0xFC, 0x81, 0x27, 0x98, 0x16, 0xBB, 0xA3,
0x9F, 0xE1, 0x4D, 0x2B, 0x71, 0x21, 0x41, 0x90,
0xFF, 0x20, 0xB8, 0x4A, 0xCF, 0xB2, 0x2D, 0xB1,
0xF8, 0x89, 0x40, 0xBC, 0xB3, 0x9F, 0x94, 0x1C,
0xF4, 0x68, 0xEA, 0x7B, 0x31, 0x29, 0xDA, 0x71,
0xCC, 0x37, 0x9A, 0xF9, 0x36, 0x0B, 0x58, 0x11,
0x6F, 0x28, 0x14, 0x6F, 0xAF, 0x57, 0x6B, 0xD7,
0xBD, 0x36, 0x98, 0xF4, 0x6C, 0x84, 0xF8, 0x48,
0xF1, 0xBF, 0x88, 0xEB, 0x5C, 0x06, 0x8B, 0x02,
0xF1, 0xDF, 0x6A, 0xFD, 0x61, 0xCF, 0x05, 0x5E,
0xB5, 0x99, 0x85, 0x31, 0x41, 0x1D, 0xE5, 0x67,
0x5C, 0x83, 0xA2, 0xBA, 0x9C, 0x9C, 0x37, 0x44,
0xEF, 0xBC, 0x0E, 0xDE, 0xBF, 0x91, 0x5B, 0x1F,
0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, 0x82, 0x01,
0x1C, 0x30, 0x82, 0x01, 0x18, 0x30, 0x0F, 0x06,
0x09, 0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x30,
0x01, 0x05, 0x04, 0x02, 0x05, 0x00, 0x30, 0x22,
0x06, 0x03, 0x55, 0x1D, 0x11, 0x04, 0x1B, 0x30,
0x19, 0xA4, 0x17, 0x30, 0x15, 0x31, 0x13, 0x30,
0x11, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x0A,
0x54, 0x47, 0x56, 0x2D, 0x45, 0x2D, 0x33, 0x32,
0x35, 0x36, 0x30, 0x1F, 0x06, 0x03, 0x55, 0x1D,
0x23, 0x04, 0x18, 0x30, 0x16, 0x80, 0x14, 0x5F,
0x60, 0xCF, 0x61, 0x90, 0x55, 0xDF, 0x84, 0x43,
0x14, 0x8A, 0x60, 0x2A, 0xB2, 0xF5, 0x7A, 0xF4,
0x43, 0x18, 0xEF, 0x30, 0x1D, 0x06, 0x03, 0x55,
0x1D, 0x0E, 0x04, 0x16, 0x04, 0x14, 0x1E, 0xA5,
0xBD, 0xCA, 0x59, 0x64, 0x55, 0x85, 0xAC, 0xDA,
0x54, 0x34, 0x23, 0x40, 0xD1, 0xF6, 0xBD, 0xC3,
0xB0, 0xF6, 0x30, 0x0C, 0x06, 0x03, 0x55, 0x1D,
0x13, 0x01, 0x01, 0xFF, 0x04, 0x02, 0x30, 0x00,
0x30, 0x6E, 0x06, 0x03, 0x55, 0x1D, 0x20, 0x04,
0x67, 0x30, 0x65, 0x30, 0x63, 0x06, 0x0B, 0x60,
0x86, 0x48, 0x01, 0x86, 0xF8, 0x45, 0x01, 0x07,
0x17, 0x03, 0x30, 0x54, 0x30, 0x26, 0x06, 0x08,
0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x02, 0x01,
0x16, 0x1A, 0x68, 0x74, 0x74, 0x70, 0x3A, 0x2F,
0x2F, 0x77, 0x77, 0x77, 0x2E, 0x73, 0x79, 0x6D,
0x61, 0x75, 0x74, 0x68, 0x2E, 0x63, 0x6F, 0x6D,
0x2F, 0x63, 0x70, 0x73, 0x30, 0x2A, 0x06, 0x08,
0x2B, 0x06, 0x01, 0x05, 0x05, 0x07, 0x02, 0x02,
0x30, 0x1E, 0x1A, 0x1C, 0x20, 0x20, 0x68, 0x74,
0x74, 0x70, 0x3A, 0x2F, 0x2F, 0x77, 0x77, 0x77,
0x2E, 0x73, 0x79, 0x6D, 0x61, 0x75, 0x74, 0x68,
0x2E, 0x63, 0x6F, 0x6D, 0x2F, 0x72, 0x70, 0x61,
0x30, 0x13, 0x06, 0x03, 0x55, 0x1D, 0x25, 0x04,
0x0C, 0x30, 0x0A, 0x06, 0x08, 0x2B, 0x06, 0x01,
0x05, 0x05, 0x07, 0x03, 0x09, 0x30, 0x0E, 0x06,
0x03, 0x55, 0x1D, 0x0F, 0x01, 0x01, 0xFF, 0x04,
0x04, 0x03, 0x02, 0x07, 0x80, 0x30, 0x0D, 0x06,
0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01,
0x01, 0x0B, 0x05, 0x00, 0x03, 0x82, 0x01, 0x01,
0x00, 0x45, 0xFF, 0xCA, 0xD2, 0xAC, 0x71, 0xBE,
0xA5, 0x62, 0x86, 0x93, 0x30, 0xD0, 0xE5, 0xE5,
0x87, 0xFC, 0xAA, 0x29, 0x73, 0x36, 0xD6, 0x66,
0x33, 0xC4, 0xCB, 0xC5, 0x6E, 0xC6, 0x2C, 0x8C,
0x8E, 0xEE, 0x4D, 0xC2, 0xFA, 0xB3, 0xC0, 0xE7,
0x11, 0x02, 0x69, 0x7A, 0xC5, 0x89, 0x28, 0x86,
0x31, 0xD5, 0x14, 0x43, 0x5A, 0x20, 0xB7, 0xBD,
0x1C, 0x0B, 0x1C, 0x3C, 0x84, 0x58, 0xBA, 0x56,
0x14, 0x5B, 0xB8, 0x38, 0x97, 0x18, 0x66, 0xD6,
0x12, 0x51, 0x4B, 0x5A, 0x2D, 0x0D, 0x00, 0xA7,
0xBA, 0x5A, 0xC9, 0x0C, 0x4B, 0x10, 0xDE, 0xF5,
0xAE, 0x56, 0xA5, 0x24, 0xC6, 0x3E, 0x5E, 0xD9,
0xF1, 0x39, 0x76, 0x0C, 0xD7, 0x4A, 0xBF, 0x19,
0x1F, 0x14, 0xA4, 0x18, 0xEC, 0x0F, 0x5D, 0x47,
0x00, 0x75, 0xF0, 0x4E, 0xB3, 0xA1, 0xB4, 0x81,
0x7B, 0x97, 0xAC, 0x0A, 0xA8, 0x5E, 0x92, 0xCC,
0xB0, 0x80, 0x53, 0x66, 0xFF, 0xC6, 0x1B, 0x71,
0xAF, 0xE3, 0x46, 0x55, 0x9D, 0x26, 0x51, 0x97,
0xB0, 0x66, 0x9D, 0x06, 0x70, 0xC5, 0x04, 0x78,
0xBC, 0x99, 0x42, 0xBA, 0x77, 0x82, 0x0E, 0xE8,
0x92, 0x18, 0x4A, 0x72, 0x92, 0x13, 0x25, 0x7F,
0x40, 0x15, 0xF7, 0xA8, 0x07, 0xA2, 0xAD, 0x03,
0xBA, 0x1C, 0xF2, 0x93, 0xBE, 0x14, 0x72, 0x69,
0x2B, 0x85, 0xAC, 0x2E, 0x2C, 0xBF, 0x1C, 0xC6,
0x6C, 0x91, 0xF3, 0x2F, 0xF0, 0xB0, 0x8A, 0xC3,
0xB8, 0xAC, 0x9B, 0xD1, 0xA1, 0x4C, 0xB7, 0x34,
0xCA, 0xC6, 0x90, 0x15, 0xA7, 0x39, 0xB4, 0xF1,
0xED, 0x54, 0x53, 0x5C, 0x29, 0x6F, 0xCE, 0x97,
0x3E, 0x72, 0x79, 0x24, 0xEA, 0xC8, 0x87, 0x21,
0x5F, 0x40, 0xBF, 0x53, 0x37, 0x8E, 0xCA, 0x0B,
0x44, 0xD0, 0x4B, 0x6E, 0xAD, 0x94, 0xFB, 0x0F,
0x33, 0xFE, 0x86, 0xDF, 0x4C, 0xE9, 0x94, 0xBB,
0x3F };

static gnutls_datum_t ocsp_resp1 =
    { (unsigned char *) _resp, sizeof(_resp) };

static void check_response(gnutls_session_t session, void *priv)
{
	int ret;
	gnutls_datum_t resp;
	gnutls_datum_t *exp_resp = priv;

	ret = gnutls_ocsp_status_request_get(session, &resp);
	if (ret < 0) {
		if (priv == NULL)
			return;
		fail("no response was received\n");
	}

	if (priv == NULL) {
		fail("not expected response, but received one\n");
	}

	if (resp.size != exp_resp->size || memcmp(resp.data, exp_resp->data, resp.size) != 0) {
		fail("did not receive the expected response\n");
	}

	/* we provide a response, but an invalid one */
	ret = gnutls_ocsp_status_request_is_checked(session, 0);
	if (ret != 0) {
		fail("did not receive the expected value (%d)\n", ret);
	}

	ret = gnutls_ocsp_status_request_is_checked(session, GNUTLS_OCSP_SR_IS_AVAIL);
	if (ret == 0) {
		fail("did not receive the expected value (%d)\n", ret);
	}
}

void doit(void)
{
	int ret;
	gnutls_certificate_credentials_t xcred;
	gnutls_certificate_credentials_t clicred;
	const char *certfile;
	const char *ocspfile1;
	char certname[TMPNAME_SIZE], ocspname1[TMPNAME_SIZE];
	time_t t;
	FILE *fp;

	global_init();
	gnutls_global_set_time_function(mytime);

	assert(gnutls_certificate_allocate_credentials(&xcred) >= 0);
	assert(gnutls_certificate_allocate_credentials(&clicred) >= 0);

	certfile = get_tmpname(certname);

	fp = fopen(certfile, "wb");
	if (fp == NULL)
		fail("error in fopen\n");
	assert(fwrite(server_localhost_ca3_cert_chain_pem, 1, strlen(server_localhost_ca3_cert_chain_pem), fp)>0);
	assert(fwrite(server_ca3_key_pem, 1, strlen((char*)server_ca3_key_pem), fp)>0);
	fclose(fp);

	/* set cert with localhost name */
	ret = gnutls_certificate_set_x509_key_file2(xcred, certfile, certfile,
						    GNUTLS_X509_FMT_PEM, NULL, 0);
	if (ret < 0)
		fail("set_x509_key_file failed: %s\n", gnutls_strerror(ret));

	fp = fopen(certfile, "wb");
	if (fp == NULL)
		fail("error in fopen\n");
	assert(fwrite(server_localhost6_ca3_cert_chain_pem, 1, strlen(server_localhost6_ca3_cert_chain_pem), fp)>0);
	assert(fwrite(server_ca3_key_pem, 1, strlen((char*)server_ca3_key_pem), fp)>0);
	fclose(fp);

	gnutls_certificate_set_flags(xcred, GNUTLS_CERTIFICATE_SKIP_OCSP_RESPONSE_CHECK);

	/* set OCSP response */
	ocspfile1 = get_tmpname(ocspname1);
	fp = fopen(ocspfile1, "wb");
	if (fp == NULL)
		fail("error in fopen\n");
	assert(fwrite(ocsp_resp1.data, 1, ocsp_resp1.size, fp)>0);
	fclose(fp);

	ret = gnutls_certificate_set_ocsp_status_request_file(xcred, ocspfile1, 0);
	if (ret < 0)
		fail("ocsp file set failed: %s\n", gnutls_strerror(ret));

	t = gnutls_certificate_get_ocsp_expiration(xcred, 0, 0, 0);
	if (t != 1511689427)
		fail("error in OCSP validity time: %ld\n", (long int)t);

	t = gnutls_certificate_get_ocsp_expiration(xcred, 0, 1, 0);
	if (t != -1)
		fail("error in OCSP validity time: %ld\n", (long int)t);

	t = gnutls_certificate_get_ocsp_expiration(xcred, 0, -1, 0);
	if (t != 1511689427)
		fail("error in OCSP validity time: %ld\n", (long int)t);


	/* make sure that our invalid OCSP responses are not considered in verification
	 */
	gnutls_certificate_set_verify_flags(clicred, GNUTLS_VERIFY_DISABLE_CRL_CHECKS);
	if (gnutls_certificate_get_verify_flags(clicred) != GNUTLS_VERIFY_DISABLE_CRL_CHECKS)
		fail("error in gnutls_certificate_set_verify_flags\n");

	ret = gnutls_certificate_set_x509_trust_mem(clicred, &ca3_cert, GNUTLS_X509_FMT_PEM);
	if (ret < 0) {
		fail("error in setting trust cert: %s\n", gnutls_strerror(ret));
	}

	test_cli_serv(xcred, clicred, "NORMAL:-VERS-TLS-ALL:+VERS-TLS1.2", "localhost", &ocsp_resp1, check_response, NULL); /* the DNS name of the first cert */

	test_cli_serv(xcred, clicred, "NORMAL:-VERS-TLS-ALL:+VERS-TLS1.3", "localhost", &ocsp_resp1, check_response, NULL); /* the DNS name of the first cert */

	gnutls_certificate_free_credentials(xcred);
	gnutls_certificate_free_credentials(clicred);
	gnutls_global_deinit();
	remove(ocspfile1);
	remove(certfile);
}

#else
void doit(void)
{
	exit(77);
}
#endif
