# Copyright (C) 2020 Konsulko Group
# Author: Edi Feschiyan
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


import asyncio
import os
import pytest
from random import randint
from pyagl.services.base import AFBResponse, AFBT
from pyagl.services.bluetooth_map import BTMAPService as BMP
import logging

pytestmark = [pytest.mark.asyncio, pytest.mark.bluetooth_map]


@pytest.fixture(scope='module')
def event_loop():
    loop = asyncio.get_event_loop()
    yield loop


@pytest.fixture(scope='module')
async def service():
    address = os.environ.get('AGL_TGT_IP', 'localhost')
    port = os.environ.get('AGL_TGT_PORT', None)

    ams = await BMP(ip=address, port=port)
    yield ams
    await ams.websocket.close()


@pytest.fixture(scope='module')
def recipient():
    phonenumber = os.environ.get('AGL_BTMAP_RECIPIENT', None)
    if phonenumber is None:
        pytest.xfail('No phone number was set, please "export AGL_BTMAP_RECIPIENT=" with a phonenumber')
    return phonenumber


@pytest.fixture(scope='module')
def composetext():
    text = os.environ.get('AGL_BTMAP_TEXT',
                          f'#{randint(0,9999999)} - Hello from AGL. This message is generated by test_bluetooth_map.py')
    return text


@pytest.mark.hwrequired
@pytest.mark.xfail(reason='Expected to fail if no messages are found')
@pytest.fixture(scope='module')
async def messages(service: BMP):
    msgid = await service.list_messages()
    resp = await service.afbresponse()
    assert 'messages' in resp.data
    assert len(resp.data) > 0, "no messages found "
    yield resp.data['messages']


@pytest.mark.hwrequired
@pytest.mark.dependency
async def test_list_messages(event_loop, service: BMP):
    msgid = await service.list_messages()
    resp = await service.afbresponse()
    assert resp.msgid == msgid


@pytest.mark.hwrequired
@pytest.mark.dependency(depends=['test_list_messages'])
@pytest.mark.xfail(reason='Expected to fail if there are no messages, need message id to test the verb')
async def test_message_verb(event_loop, service, messages):
    assert len(messages), "No existing messages to test the verb with"
    messageids = list(messages.keys())
    msgid = await service.message(messageids.pop())
    resp = await service.afbresponse()
    assert resp.status == 'success'


@pytest.mark.dependency
async def test_subscribe_notifications(event_loop, service: BMP):
    msgid = await service.subscribe('notification')
    resp = await service.afbresponse()
    print(resp)
    assert resp.status == 'success'


@pytest.mark.hwrequired
@pytest.mark.dependency
async def test_compose_message(event_loop, service: BMP, recipient, composetext):
    msgid = await service.compose(recipient, composetext)
    resp = await service.afbresponse()
    assert resp.status == 'success'


@pytest.mark.hwrequired
@pytest.mark.dependency(depends=['test_compose_message', 'test_subscribe_notifications'])
async def test_message_to_self(event_loop, service: BMP, recipient, composetext):
    resp = await service.afbresponse()
    assert resp.api == f'{service.api}/notification'
    assert resp.type == AFBT.EVENT
    assert isinstance(resp.data, dict)
    assert 'message' in resp.data
    assert resp.data['message'] == composetext, 'Message mismatching'


@pytest.mark.hwrequired
@pytest.mark.dependency(depends=['test_subscribe_notifications'])
async def test_unsubscribe_notifications(event_loop, service: BMP):
    msgid = await service.unsubscribe('notification')
    resp = await service.afbresponse()
    assert resp.status == 'success', resp.info
