"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Copyright 2017 Google Inc. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
const Launcher_1 = require("./Launcher");
const BrowserFetcher_1 = require("./BrowserFetcher");
const Errors_1 = require("./Errors");
const DeviceDescriptors_1 = require("./DeviceDescriptors");
const QueryHandler = require("./QueryHandler");
class Puppeteer {
    constructor(projectRoot, preferredRevision, isPuppeteerCore, productName) {
        this._changedProduct = false;
        this._projectRoot = projectRoot;
        this._preferredRevision = preferredRevision;
        this._isPuppeteerCore = isPuppeteerCore;
        // track changes to Launcher configuration via options or environment variables
        this.__productName = productName;
    }
    launch(options = {}) {
        if (options.product)
            this._productName = options.product;
        return this._launcher.launch(options);
    }
    connect(options) {
        if (options.product)
            this._productName = options.product;
        return this._launcher.connect(options);
    }
    set _productName(name) {
        if (this.__productName !== name)
            this._changedProduct = true;
        this.__productName = name;
    }
    get _productName() {
        return this.__productName;
    }
    executablePath() {
        return this._launcher.executablePath();
    }
    get _launcher() {
        if (!this._lazyLauncher || this._lazyLauncher.product !== this._productName || this._changedProduct) {
            // @ts-ignore
            // eslint-disable-next-line @typescript-eslint/no-var-requires
            const packageJson = require('../package.json');
            switch (this._productName) {
                case 'firefox':
                    this._preferredRevision = packageJson.puppeteer.firefox_revision;
                    break;
                case 'chrome':
                default:
                    this._preferredRevision = packageJson.puppeteer.chromium_revision;
            }
            this._changedProduct = false;
            this._lazyLauncher = Launcher_1.default(this._projectRoot, this._preferredRevision, this._isPuppeteerCore, this._productName);
        }
        return this._lazyLauncher;
    }
    get product() {
        return this._launcher.product;
    }
    get devices() {
        return DeviceDescriptors_1.devicesMap;
    }
    get errors() {
        return Errors_1.puppeteerErrors;
    }
    defaultArgs(options) {
        return this._launcher.defaultArgs(options);
    }
    createBrowserFetcher(options) {
        return new BrowserFetcher_1.BrowserFetcher(this._projectRoot, options);
    }
    // eslint-disable-next-line @typescript-eslint/camelcase
    __experimental_registerCustomQueryHandler(name, queryHandler) {
        QueryHandler.registerCustomQueryHandler(name, queryHandler);
    }
    // eslint-disable-next-line @typescript-eslint/camelcase
    __experimental_unregisterCustomQueryHandler(name) {
        QueryHandler.unregisterCustomQueryHandler(name);
    }
    // eslint-disable-next-line @typescript-eslint/camelcase
    __experimental_customQueryHandlers() {
        return QueryHandler.customQueryHandlers();
    }
    // eslint-disable-next-line @typescript-eslint/camelcase
    __experimental_clearQueryHandlers() {
        QueryHandler.clearQueryHandlers();
    }
}
exports.Puppeteer = Puppeteer;
