// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import * as fs from 'fs';
import * as path from 'path';

import {generateClosureBridge, GeneratedCode} from './generate_closure.js';
import {filePathToTypeScriptSourceFile, walkTree} from './walk_tree.js';

const chromeLicense = `// Copyright ${new Date().getFullYear()} The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
`;

export const writeToDisk = (inputFilePath: string, generatedCode: GeneratedCode) => {
  const dir = path.dirname(inputFilePath);
  const baseName = path.basename(inputFilePath, '.ts');
  const outputFileName = `${baseName}_bridge.js`;

  const interfaces = generatedCode.interfaces
                         .map(interfacePart => {
                           return interfacePart.join('\n');
                         })
                         .join('\n');
  const classDeclaration = generatedCode.closureClass.join('\n');
  const creatorFunction = generatedCode.creatorFunction.join('\n');

  const relativeFilePath = path.relative(process.cwd(), inputFilePath);

  const byHandWarning = `/**
* WARNING: do not modify this file by hand!
* it was automatically generated by the bridge generator
* if you made changes to the source code and need to update this file, run:
*  npm run generate-bridge-file ${relativeFilePath}
*/
`;

  // extra \n to ensure ending with a linebreak at end of file
  const finalCode = [chromeLicense, byHandWarning, interfaces, classDeclaration, creatorFunction].join('\n') + '\n';

  fs.writeFileSync(path.join(dir, outputFileName), finalCode, {encoding: 'utf8'});

  return {
    output: path.join(dir, outputFileName),
    code: finalCode,
  };
};

export const parseTypeScriptComponent = (componentSourceFilePath: string) => {
  const file = filePathToTypeScriptSourceFile(componentSourceFilePath);
  const state = walkTree(file, componentSourceFilePath);
  const generatedCode = generateClosureBridge(state);
  return writeToDisk(componentSourceFilePath, generatedCode);
};

const main = (args: string[]) => {
  const bridgeComponentPath = path.resolve(process.cwd(), args[0]);

  if (!bridgeComponentPath || !fs.existsSync(bridgeComponentPath)) {
    throw new Error(`Could not find bridgeComponent path ${bridgeComponentPath}`);
  }

  const {output} = parseTypeScriptComponent(bridgeComponentPath);

  console.log('Wrote bridge file to', output);
};

if (require.main === module) {
  main(process.argv.slice(2));
}
