// Copyright (c) 2018-2022 LG Electronics, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// SPDX-License-Identifier: Apache-2.0

#ifndef AGL_WEB_APP_MANAGER_SERVICE_AGL_H
#define AGL_WEB_APP_MANAGER_SERVICE_AGL_H

#include <memory>

#include "agl_shell_types.h"
#include "timer.h"
#include "web_app_manager_service.h"

constexpr char kStartApp[] = "start-app";
constexpr char kKilledApp[] = "killed-app";
constexpr char kActivateEvent[] = "activate-event";
constexpr char kDeactivateEvent[] = "deactivate-event";

class GrpcClient;
class WamIPCLockFile;

class WebAppManagerServiceAGL : public WebAppManagerService,
                                public TimerReceiver {
 public:
  struct LaunchParams {
    std::string app_id;
    std::string uri;
    AglShellSurfaceType surface_type = AglShellSurfaceType::kNone;
    AglShellPanelEdge panel_edge = AglShellPanelEdge::kNotFound;
    int width = 0;
    int height = 0;
  };

  static WebAppManagerServiceAGL* Instance();

  bool InitializeAsHostService();

  bool IsHostServiceRunning();

  void LaunchOnIdle(const LaunchParams& params);
  void SendEventOnIdle(const std::string& event, const std::string& app_id);
  void LaunchOnHost(const LaunchParams& params);
  void SendEvent(const std::string& event_name, const std::string& app_id);

  bool IsUsingGrpc() const;

  // WebAppManagerService
  bool StartService() override;
  Json::Value launchApp(const Json::Value& request) override;
  Json::Value killApp(const Json::Value& request) override;
  Json::Value pauseApp(const Json::Value& request) override;
  Json::Value logControl(const Json::Value& request) override;
  Json::Value setInspectorEnable(const Json::Value& request) override;
  Json::Value closeAllApps(const Json::Value& request) override;
  Json::Value discardCodeCache(const Json::Value& request) override;
  Json::Value listRunningApps(const Json::Value& request,
                              bool subscribed) override;
  Json::Value getWebProcessSize(const Json::Value& request) override;
  Json::Value clearBrowsingData(const Json::Value& request) override;
  Json::Value webProcessCreated(const Json::Value& request,
                                bool subscribed) override;

  void TriggerStartupApp();

 private:
  WebAppManagerServiceAGL();

  void OnLaunchApp(LaunchParams* launch_data);
  void LaunchStartupAppFromJsonConfig(LaunchParams*);
  void LaunchStartupAppFromURL(LaunchParams*);
  struct EventData {
    std::string app_id;
  };
  void OnActivateEvent(EventData* event_data);
  void OnDeactivateEvent(EventData* event_data);
  void OnKillEvent(EventData* event_data);

  bool using_grpc_ = false;
  std::unique_ptr<GrpcClient> grpc_client_;
  GrpcClient* GetGrpcClient();

  std::unique_ptr<WamIPCLockFile> lock_file_;
};

#endif  // AGL_WEB_APP_MANAGER_SERVICE_AGL_H
