// Copyright 2017 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef COMPONENTS_POLICY_CONTENT_POLICY_BLOCKLIST_NAVIGATION_THROTTLE_H_
#define COMPONENTS_POLICY_CONTENT_POLICY_BLOCKLIST_NAVIGATION_THROTTLE_H_

#include "base/time/time.h"
#include "base/timer/timer.h"
#include "components/policy/content/safe_sites_navigation_throttle.h"
#include "components/policy/core/common/policy_namespace.h"
#include "components/policy/core/common/policy_service.h"
#include "content/public/browser/navigation_throttle.h"

class GURL;
class PolicyBlocklistService;
class PrefService;

// PolicyBlocklistNavigationThrottle provides a simple way to block a navigation
// based on the URLBlocklistManager and Safe Search API. If the URL is on the
// blocklist or allowlist, the throttle will immediately block or allow the
// navigation. Otherwise, the URL will be checked against the Safe Search API if
// the SafeSitesFilterBehavior policy is enabled. This final check may be
// asynchronous if the result hasn't been cached yet.
class PolicyBlocklistNavigationThrottle
    : public content::NavigationThrottle,
      public policy::PolicyService::Observer {
 public:
  PolicyBlocklistNavigationThrottle(
      content::NavigationHandle* navigation_handle,
      content::BrowserContext* context,
      policy::PolicyService* policy_service);
  PolicyBlocklistNavigationThrottle(const PolicyBlocklistNavigationThrottle&) =
      delete;
  PolicyBlocklistNavigationThrottle& operator=(
      const PolicyBlocklistNavigationThrottle&) = delete;
  ~PolicyBlocklistNavigationThrottle() override;

  // NavigationThrottle overrides.
  ThrottleCheckResult WillStartRequest() override;
  ThrottleCheckResult WillRedirectRequest() override;
  const char* GetNameForLogging() override;

  // policy::PolicyService::Observer overrides.
  void OnFirstPoliciesLoaded(policy::PolicyDomain domain) override;

  void OnFirstPoliciesLoadedTimeout();

  void OnFirstPoliciesLoadedImpl(bool timeout);

 private:
  ThrottleCheckResult GetUrlThrottleResult(const GURL& url);
  // To ensure both allow and block policies override Safe Sites,
  // SafeSitesNavigationThrottle must be consulted as part of this throttle
  // rather than added separately to the list of throttles.
  ThrottleCheckResult CheckSafeSitesFilter(const GURL& url);
  void OnDeferredSafeSitesResult(bool is_safe,
                                 ThrottleCheckResult cancel_result);
  SafeSitesNavigationThrottle safe_sites_navigation_throttle_;

  PolicyBlocklistService* blocklist_service_;

  // While this is not null, we are not sure policies from all the possible
  // sources have been loaded. This is null if policies have been loaded.
  policy::PolicyService* policy_service_;

  PrefService* prefs_;

  // Time where the navigation was deferred because all policies were not
  // loaded.
  base::TimeTicks policy_load_throttle_start_time_;

  // Timer to try and continue the deferred navigation in case we are waiting
  // for policies to be loaded and |OnFirstPoliciesLoaded| has not been called
  // before a timeout.
  base::OneShotTimer wait_for_policy_timer_;
};

#endif  // COMPONENTS_POLICY_CONTENT_POLICY_BLOCKLIST_NAVIGATION_THROTTLE_H_
