// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef COMPONENTS_POLICY_CORE_COMMON_CLOUD_ENCRYPTED_REPORTING_JOB_CONFIGURATION_H_
#define COMPONENTS_POLICY_CORE_COMMON_CLOUD_ENCRYPTED_REPORTING_JOB_CONFIGURATION_H_

#include <memory>
#include <string>

#include "base/callback.h"
#include "base/macros.h"
#include "base/optional.h"
#include "base/values.h"
#include "components/policy/core/common/cloud/cloud_policy_client.h"
#include "components/policy/core/common/cloud/device_management_service.h"
#include "components/policy/core/common/cloud/reporting_job_configuration_base.h"
#include "components/policy/policy_export.h"
#include "components/reporting/proto/record_constants.pb.h"

namespace policy {

// EncryptedReportingJobConfiguration configures a payload for the Encrypted
// server endpoint. A JSON version of the payload looks like this:
// {
//   "encryptedRecord": [
//     {
//       "encryptedWrappedRecord": "EncryptedMessage",
//       "encryptionInfo" : {
//         "encryptionKey": "EncryptedMessage",
//         "publicKeyId": 1
//       },
//       "sequencingInformation": {
//         "sequencingId": 1,
//         "generationId": 123456789,
//         "priority": 1
//       }
//     },
//     {
//       "encryptedWrappedRecord": "EncryptedMessage",
//       "encryptionInfo" : {
//         "encryptionKey": "EncryptedMessage",
//         "publicKeyId": 2
//       },
//       "sequencingInformation": {
//         "sequencingId": 2,
//         "generationId": 123456789,
//         "priority": 1
//       }
//     }
//   ],
//   "device": {
//     "client_id": "abcdef1234",
//     "dmToken": "abcdef1234",
//     "name": "George",
//     "osPlatform": "Windows",
//     "osVersion": "10.0.0.0"
//   },
//   "browser": {
//     "browserId": "abcdef1234",
//     "chromeVersion": "10.0.0.0",
//     "machineUser": "abcdef1234",
//     "userAgent": "abcdef1234"
//   }
// }
// "device" and "browser" are populated by the base class,
// the rest needs to be provided as |merging_payload|.
class POLICY_EXPORT EncryptedReportingJobConfiguration
    : public ReportingJobConfigurationBase {
 public:
  EncryptedReportingJobConfiguration(CloudPolicyClient* client,
                                     const std::string& server_url,
                                     base::Value merging_payload,
                                     UploadCompleteCallback complete_cb);
  ~EncryptedReportingJobConfiguration() override;

  // |context| is a base::Value item as generated by |reporting::GetContext|.
  // |context| will be stored and applied to the payload on the proceeding
  // |GetPayload| call. |context| mostly fills out the profile dictionary, but
  // does overwrite a few of the device and browser fields (check
  // reporting::GetContext for specifics).
  void UpdateContext(base::Value& context);

 protected:
  void UpdatePayloadBeforeGetInternal() override;

  std::string GetUmaString() const override;

 private:
  friend class EncryptedReportingJobConfigurationTest;

  std::set<std::string> GetTopLevelKeyAllowList();
};

}  // namespace policy

#endif  // COMPONENTS_POLICY_CORE_COMMON_CLOUD_ENCRYPTED_REPORTING_JOB_CONFIGURATION_H_
