// Copyright 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef COMPONENTS_SAFE_BROWSING_CONTENT_PASSWORD_PROTECTION_PASSWORD_PROTECTION_REQUEST_CONTENT_H_
#define COMPONENTS_SAFE_BROWSING_CONTENT_PASSWORD_PROTECTION_PASSWORD_PROTECTION_REQUEST_CONTENT_H_

#include <memory>
#include <string>
#include <vector>

#include "base/memory/weak_ptr.h"
#include "base/time/time.h"
#include "build/build_config.h"
#include "components/password_manager/core/browser/password_manager_metrics_util.h"
#include "components/password_manager/core/browser/password_reuse_detector.h"
#include "components/safe_browsing/buildflags.h"
#include "components/safe_browsing/core/password_protection/metrics_util.h"
#include "components/safe_browsing/core/password_protection/password_protection_request.h"
#include "components/safe_browsing/core/proto/csd.pb.h"

#if BUILDFLAG(SAFE_BROWSING_AVAILABLE)
#include "components/safe_browsing/content/common/safe_browsing.mojom.h"
#include "mojo/public/cpp/bindings/remote.h"
#include "third_party/skia/include/core/SkBitmap.h"
#endif  // BUILDFLAG(SAFE_BROWSING_AVAILABLE)

class GURL;
class PasswordProtectionServiceBase;
class RequestCanceler;

namespace content {
class WebContents;
}

namespace safe_browsing {

class PasswordProtectionNavigationThrottle;

using password_manager::metrics_util::PasswordType;

class PasswordProtectionRequestContent : public PasswordProtectionRequest {
 public:
  PasswordProtectionRequestContent(
      content::WebContents* web_contents,
      const GURL& main_frame_url,
      const GURL& password_form_action,
      const GURL& password_form_frame_url,
      const std::string& mime_type,
      const std::string& username,
      PasswordType password_type,
      const std::vector<password_manager::MatchingReusedCredential>&
          matching_reused_credentials,
      LoginReputationClientRequest::TriggerType type,
      bool password_field_exists,
      PasswordProtectionServiceBase* pps,
      int request_timeout_in_ms);

  // CancelableRequest implementation
  void Cancel(bool timed_out) override;

  content::WebContents* web_contents() const { return web_contents_; }

  base::WeakPtr<PasswordProtectionRequestContent> AsWeakPtr() {
    return base::AsWeakPtr(this);
  }

  // Keeps track of created navigation throttle.
  void AddThrottle(PasswordProtectionNavigationThrottle* throttle) {
    throttles_.insert(throttle);
  }

  void RemoveThrottle(PasswordProtectionNavigationThrottle* throttle) {
    throttles_.erase(throttle);
  }

  // Cancels navigation if there is modal warning showing, resumes it otherwise.
  void HandleDeferredNavigations();

 private:
  friend class PasswordProtectionServiceTest;
  friend class ChromePasswordProtectionServiceTest;
  ~PasswordProtectionRequestContent() override;

  void MaybeLogPasswordReuseLookupEvent(
      RequestOutcome outcome,
      const LoginReputationClientResponse* response) override;

  void MaybeAddPingToWebUI() override;

  void MaybeAddResponseToWebUI(
      const LoginReputationClientResponse& response) override;

#if BUILDFLAG(SAFE_BROWSING_AVAILABLE)
  bool IsClientSideDetectionEnabled() override;

  // Extracts DOM features.
  void GetDomFeatures() override;

  // Called when the DOM feature extraction is complete.
  void OnGetDomFeatures(mojom::PhishingDetectorResult result,
                        const std::string& verdict);

  // Called when the DOM feature extraction times out.
  void OnGetDomFeatureTimeout();

  bool IsVisualFeaturesEnabled() override;

  // If appropriate, collects visual features, otherwise continues on to sending
  // the request.
  void MaybeCollectVisualFeatures() override;

  // Collects visual features from the current login page.
  void CollectVisualFeatures();

  // Processes the screenshot of the login page into visual features.
  void OnScreenshotTaken(const SkBitmap& bitmap);

  // Called when the visual feature extraction is complete.
  void OnVisualFeatureCollectionDone(
      std::unique_ptr<VisualFeatures> visual_features);
#endif  // BUILDFLAG(SAFE_BROWSING_AVAILABLE)

#if defined(OS_ANDROID)
  void SetReferringAppInfo() override;
#endif  // defined(OS_ANDROID)

  // WebContents of the password protection event.
  content::WebContents* web_contents_;

  // Cancels the request when it is no longer valid.
  std::unique_ptr<RequestCanceler> request_canceler_;

  // Navigation throttles created for this |web_contents_| during |this|'s
  // lifetime. These throttles are owned by their corresponding
  // NavigationHandler instances.
  std::set<PasswordProtectionNavigationThrottle*> throttles_;

  // If a request is sent, this is the token returned by the WebUI.
  int web_ui_token_;

#if BUILDFLAG(SAFE_BROWSING_AVAILABLE)
  // When we start extracting visual features.
  base::TimeTicks visual_feature_start_time_;

  // The Mojo pipe used for extracting DOM features from the renderer.
  mojo::Remote<safe_browsing::mojom::PhishingDetector> phishing_detector_;

  // When we start extracting DOM features. Used to compute the duration of DOM
  // feature extraction, which is logged at
  // PasswordProtection.DomFeatureExtractionDuration.
  base::TimeTicks dom_feature_start_time_;

  // Whether the DOM features collection is finished, either by timeout or by
  // successfully gathering the features.
  bool dom_features_collection_complete_;
#endif  // BUILDFLAG(SAFE_BROWSING_AVAILABLE)
};

}  // namespace safe_browsing

#endif  // COMPONENTS_SAFE_BROWSING_CONTENT_PASSWORD_PROTECTION_PASSWORD_PROTECTION_REQUEST_CONTENT_H_
