/* vi: set expandtab sw=4 sts=4: */
/* str_list.c - the opkg package management system

   Carl D. Worth

   Copyright (C) 2001 University of Southern California

   SPDX-License-Identifier: GPL-2.0-or-later

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2, or (at
   your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
*/

#include "config.h"

#include <fnmatch.h>
#include <stdlib.h>
#include <string.h>

#include "str_list.h"
#include "xfuncs.h"

void str_list_elt_init(str_list_elt_t * elt, char *data)
{
    void_list_elt_init((void_list_elt_t *) elt, data);
}

void str_list_elt_deinit(str_list_elt_t * elt)
{
    if (elt->data)
        free(elt->data);
    void_list_elt_deinit((void_list_elt_t *) elt);
}

str_list_t *str_list_alloc()
{
    str_list_t *list = xcalloc(1, sizeof(str_list_t));
    str_list_init(list);
    return list;
}

void str_list_init(str_list_t * list)
{
    void_list_init((void_list_t *) list);
}

void str_list_deinit(str_list_t * list)
{
    str_list_elt_t *elt;
    while (!void_list_empty(list)) {
        elt = str_list_first(list);
        if (!elt)
            return;
        list_del_init(&elt->node);
        free(elt->data);
        elt->data = NULL;
        free(elt);
    }
}

void str_list_append(str_list_t * list, char *data)
{
    void_list_append((void_list_t *) list, xstrdup(data));
}

str_list_elt_t *str_list_pop(str_list_t * list)
{
    return (str_list_elt_t *) void_list_pop((void_list_t *) list);
}

void str_list_remove(str_list_t * list, str_list_elt_t ** iter)
{
    char *str = void_list_remove((void_list_t *) list,
                                 (void_list_elt_t **) iter);

    if (str)
        free(str);
}

void str_list_remove_elt(str_list_t * list, const char *target_str)
{
    char *str = void_list_remove_elt((void_list_t *) list,
                                     (void *)target_str,
                                     (void_list_cmp_t) strcmp);
    if (str)
        free(str);
}

str_list_elt_t *str_list_first(str_list_t * list)
{
    return (str_list_elt_t *) void_list_first((void_list_t *) list);
}

str_list_elt_t *str_list_next(str_list_t * list, str_list_elt_t * node)
{
    return (str_list_elt_t *) void_list_next((void_list_t *) list,
                                             (void_list_elt_t *) node);
}

void str_list_purge(str_list_t * list)
{
    str_list_deinit(list);
    free(list);
}

int str_list_contains(str_list_t * list, const char *s, int use_glob)
{
    str_list_elt_t *elt;
    const char *candidate;

    for (elt = str_list_first(list); elt; elt = str_list_next(list, elt)) {
        candidate = elt->data;
        if (use_glob) {
            if (fnmatch(candidate, s, 0) == 0)
                return 1;
        } else {
            if (strcmp(candidate, s) == 0)
                return 1;
        }
    }

    return 0;
}
