/*
 * Camera test application
 *
 * Copyright (C) 2016-2023 Renesas Electronics Corporation
 * Copyright (C) 2016-2023 Cogent Embedded, Inc. <source@cogentembedded.com>
 *
 * based on:
 *  V4L2 video capture example
 *  This program is provided with the V4L2 API
 *  see http://linuxtv.org/docs.php for more information
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <stddef.h>
#include <string.h>
#include <assert.h>

#include <getopt.h>             /* getopt_long() */

#include <fcntl.h>              /* low-level i/o */
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/mman.h>
#include <sys/ioctl.h>

#include <linux/videodev2.h>
#include <linux/fb.h>

#ifdef DRM_ENABLE
#include <xf86drm.h>
#include <xf86drmMode.h>
#include <drm_fourcc.h>
#endif /* DRM_ENABLE */

#define ARRAY_SIZE(x)   (sizeof(x) / sizeof((x)[0]))
#define CLEAR(x)	memset(&(x), 0, sizeof(x))

/****************** Format interface ******************/

#define FORMAT_BE_FLAG	(1U << 31)
#define FORMAT_BE_STR	"(BE)"
#define FORMAT_STR_MAX	(sizeof(unsigned) + sizeof(FORMAT_BE_STR))

#define container_of(ptr, type, member)	\
			(type *)((char *)(ptr) - offsetof(type, member))
struct format {
	const char *name;
	unsigned code;
};

unsigned format_find(const char *format_name);
void format_list_supported(FILE *fp, unsigned indent, unsigned tabstop);
void format_to_str(char *buf, unsigned fourcc);
static inline unsigned format_to_bpp(unsigned fourcc);

/****************** Buffer interface ******************/

struct buffer {
	void *start;
	size_t length;
};

/****************** Window interface ******************/

struct window {
	unsigned char *ptr[2];
	unsigned size[2];
	unsigned width;
	unsigned height;
	unsigned stride;
	unsigned fourcc;
	unsigned bpp;
};

int window_init(struct window *window, unsigned width, unsigned height,
		unsigned stride, unsigned fourcc);
int window_bind_buffers(struct window *window, struct buffer *plane1,
			struct buffer *plane2);
int window_subwindow_get(const struct window *src, struct window *dst,
			 unsigned top, unsigned left,
			 unsigned width, unsigned height);
int window_copy(const struct window *src, struct window *dst);

/****************** Framebuffer interface ******************/

struct fbtype {
	unsigned major;
	struct fb* (*create)(int fd, unsigned format);
};

struct fb {
	struct window window;
	int fd;
	void (*release)(struct fb*);
};

struct genfb {
	struct fb fb;
	struct buffer buffer;
};

#ifdef DRM_ENABLE
struct drmfb_info {
	uint32_t xsub[4];
	uint32_t ysub[4];
	uint32_t bpp[4];
	uint32_t size[4];
	uint32_t offsets[4];
	uint32_t pitches[4];
	uint32_t bo_handles[4];
};

struct drmplane {
	struct buffer buffer;
	uint32_t handle;
};

struct drmfb {
	struct fb fb;
	struct drmplane *planes;
	unsigned num_planes;
	drmModeCrtc *saved_crtc;
	uint32_t fb_id;
	uint32_t crtc_id;
	uint32_t conn_id;
};

static inline struct drmfb *to_drmfb(struct fb *fb)
{
	return container_of(fb, struct drmfb, fb);
}
#endif /* DRM_ENABLE */

static inline struct genfb *to_genfb(struct fb *fb)
{
	return container_of(fb, struct genfb, fb);
}

struct fb *fb_create(const char *fbdev_name, unsigned format);
void fb_release(struct fb *fb);
struct window *fb_windows_create(struct fb *fb, unsigned n_windows);

/****************** FPS counter interface ******************/

struct fps_counter {
	unsigned frames;
	struct timeval frame_time;
	unsigned long usec;
};

struct fps_counter *fps_count_create(void);
int fps_count_release(struct fps_counter *fps);
int fps_count(struct fps_counter *fps);

/****************** Camera interface ******************/

enum io_method {
        IO_METHOD_READ,
        IO_METHOD_MMAP,
        IO_METHOD_USERPTR,
};

//#define FIELD V4L2_FIELD_INTERLACED
#define FIELD V4L2_FIELD_NONE

struct camera {
	int fd;
	char *dev_name;
	struct buffer *buffers;
	unsigned n_buffers;
	// Current camera buffer window
	struct window window;
	// FPS counter
	struct fps_counter *fps;
	// Callbacks
	int (*buf_alloc)(struct camera *cam, unsigned size);
	int (*buf_release)(struct camera *cam);
	int (*start)(struct camera *cam);
	int (*stop)(struct camera *cam);
	int (*frame_get)(struct camera *cam, struct v4l2_buffer *buf);
	int (*frame_put)(struct camera *cam, struct v4l2_buffer *buf);
};

struct camera *camera_create(const char *dev_name,
			     unsigned left, unsigned top,
			     unsigned width, unsigned height,
			     unsigned format, enum io_method io,
			     unsigned framerate, int fps_count);
void camera_release(struct camera *cam);

static inline int camera_start(struct camera *cam);
static inline int camera_stop(struct camera *cam);

static inline int camera_fd_set(struct camera *cam, int *nfd, fd_set *fds);
static inline int camera_fd_isset(struct camera *cam, fd_set *fds);

static inline int camera_frame_get(struct camera *cam, struct v4l2_buffer *buf);
static inline int camera_frame_put(struct camera *cam, struct v4l2_buffer *buf);
static inline int camera_frame_dump(struct camera *cam, struct v4l2_buffer *buf,
				    FILE *fp);
static inline const struct window *camera_frame_window_set(struct camera *cam,
							   struct v4l2_buffer *buf);

/****************** Application parameters interface ******************/

struct params {
	const char **dev_names;
	size_t n_dev_names;
	unsigned n_devs;
	const char *fbdev_name;
	const char *format_name;
	enum io_method io;
	FILE *out_fp;
	int fps_count;
	int framerate;
	int frame_count;
	int timeout;
	int left;
	int top;
	int width;
	int height;
};

struct params *params_create(void);
void params_release(struct params *params);
int params_parse(struct params *params, int argc, char *argv[]);
const char *params_get_dev_name(struct params *params, unsigned idx);

/****************** Implementation ******************/

/* Common private functions */

static inline unsigned long usec_elapsed(struct timeval *t2, struct timeval *t1)
{
        return (t2->tv_sec - t1->tv_sec) * 1000000 + t2->tv_usec - t1->tv_usec;
}

static unsigned mean4(unsigned v1, unsigned v2, unsigned v3, unsigned v4)
{
	return (v1 + v2 + v3 + v4) >> 2;
}

static unsigned mean2(unsigned v1, unsigned v2)
{
	return (v1 + v2) >> 1;
}

/* Format private data */

static struct format formats[] = {
	{
		.name = "uyvy", .code = V4L2_PIX_FMT_UYVY,
	}, {
		.name = "yuyv", .code = V4L2_PIX_FMT_YUYV,
	}, {
		.name = "rgb565", .code = V4L2_PIX_FMT_RGB565,
	}, {
		.name = "rgb32", .code = V4L2_PIX_FMT_XBGR32,
	}, {
		.name = "nv12", .code = V4L2_PIX_FMT_NV12,
	}, {
		.name = "nv16", .code = V4L2_PIX_FMT_NV16,
	}, {
		.name = "bggr8", .code = V4L2_PIX_FMT_SBGGR8,
	}, {
		.name = "rggb8", .code = V4L2_PIX_FMT_SRGGB8,
	}, {
		.name = "bggr12", .code = V4L2_PIX_FMT_SBGGR12,
	}, {
		.name = "rggb12", .code = V4L2_PIX_FMT_SRGGB12,
	}, {
		.name = "bggr16", .code = V4L2_PIX_FMT_SBGGR16,
	}, {
		.name = "rggb16", .code = V4L2_PIX_FMT_SRGGB16,
	}, {
		.name ="grey", .code = V4L2_PIX_FMT_GREY,
	}, {
		.name = "raw10", .code = V4L2_PIX_FMT_Y10,
	}, {
		.name = "raw12", .code = V4L2_PIX_FMT_Y12,
	}, {
		.name = "raw14", .code = V4L2_PIX_FMT_Y14,
	}, {
		.name = "raw16", .code = V4L2_PIX_FMT_Y16,
	},
};

/* Format public API */

void format_list_supported(FILE *fp, unsigned indent, unsigned tabstop)
{
	unsigned i;

	for (i = 0; i < ARRAY_SIZE(formats); i++) {
		char fmt[64];
		char str[FORMAT_STR_MAX];

		sprintf(fmt, "%%-%us%%-%us%%s\n", indent, tabstop);
		format_to_str(str, formats[i].code);
		fprintf(fp, fmt, "", formats[i].name, str);
	}
}

unsigned format_find(const char *format_name)
{
	unsigned i;

	if (!format_name)
		return 0;

	for (i = 0; i < ARRAY_SIZE(formats); i++) {
		if (!strcmp(formats[i].name, format_name)) {
			return formats[i].code;
		}
	}

	return 0;
}

void format_to_str(char *buf, unsigned fourcc)
{
	if (fourcc & FORMAT_BE_FLAG) {
		fourcc &= ~FORMAT_BE_FLAG;
		strcpy(buf, FORMAT_BE_STR);
		buf += strlen(FORMAT_BE_STR);
	}

	while (1) {
		*buf = fourcc;

		if (!fourcc)
			break;

		fourcc >>= 8;
		buf++;
	}
}

static inline unsigned format_to_bpp(unsigned fourcc)
{
	switch (fourcc) {
	case V4L2_PIX_FMT_XBGR32:
		return 4;
	case V4L2_PIX_FMT_UYVY:
	case V4L2_PIX_FMT_YUYV:
	case V4L2_PIX_FMT_RGB565:
	case V4L2_PIX_FMT_Y10:
	case V4L2_PIX_FMT_Y12:
	case V4L2_PIX_FMT_Y14:
	case V4L2_PIX_FMT_Y16:
	case V4L2_PIX_FMT_SBGGR12:
	case V4L2_PIX_FMT_SRGGB12:
	case V4L2_PIX_FMT_SBGGR16:
	case V4L2_PIX_FMT_SRGGB16:
		return 2;
	case V4L2_PIX_FMT_GREY:
	case V4L2_PIX_FMT_SBGGR8:
	case V4L2_PIX_FMT_SRGGB8:
	case V4L2_PIX_FMT_NV12:
	case V4L2_PIX_FMT_NV16:
		return 1;
	default:
		break;
	}
	return -1;
}

/* Window private functions */

// GREY to XBGR32
static inline void grey_to_xbgr32(unsigned grey, unsigned shift,
				  unsigned char *rgb)
{
	register unsigned v;

	v = grey >> shift;

	*rgb++ = v; //B
	*rgb++ = v; //G
	*rgb++ = v; //R
	*rgb   = 0; //A
}

static int __window_grey8_to_xbgr32(const struct window *src, struct window *dst)
{
	unsigned x, y, width, height;
	const unsigned char *ptr_src;
	unsigned char *ptr_dst;

	width = src->width < dst->width ? src->width : dst->width;
	height = src->height < dst->height ? src->height : dst->height;
	ptr_src = src->ptr[0];
	ptr_dst = dst->ptr[0];

	for (y = 0; y < height; y++) {
		for (x = 0; x < width; x++)
			grey_to_xbgr32(ptr_src[x], 0, &ptr_dst[x * 4]);
		ptr_dst += dst->stride;
		ptr_src += src->stride;
	}

	return 0;
}

static int __window_grey16_to_xbgr32(const struct window *src, struct window *dst)
{
	unsigned x, y, width, height, shift;
	const unsigned char *ptr_src;
	unsigned char *ptr_dst;

	width = src->width < dst->width ? src->width : dst->width;
	height = src->height < dst->height ? src->height : dst->height;
	ptr_src = src->ptr[0];
	ptr_dst = dst->ptr[0];

	switch (src->fourcc) {
	case V4L2_PIX_FMT_Y10:
		shift = 2;
		break;
	case V4L2_PIX_FMT_Y12:
		shift = 4;
		break;
	case V4L2_PIX_FMT_Y14:
		shift = 6;
		break;
	case V4L2_PIX_FMT_Y16:
		shift = 8;
		break;
	default:
		return -1;
	}

	for (y = 0; y < height; y++) {
		const unsigned short *ptr_src16 = (const unsigned short *)ptr_src;

		for (x = 0; x < width; x++)
			grey_to_xbgr32(ptr_src16[x], shift, &ptr_dst[x * 4]);
		ptr_dst += dst->stride;
		ptr_src += src->stride;
	}

	return 0;
}

// RGB656 to XBGR32
static inline void rgb565_to_xbgr32(unsigned short rgb565, unsigned char *rgb)
{
    register unsigned r, g, b;

    r = (rgb565 >> 8) & 0xf8;
    g = (rgb565 >> 3) & 0xfc;
    b = (rgb565 << 3);

    *rgb++ = b; //B
    *rgb++ = g; //G
    *rgb++ = r; //R
    *rgb   = 0; //A
}

static int __window_rgb565_to_xbgr32(const struct window *src, struct window *dst)
{
	unsigned x, y, width, height;
	const unsigned char *ptr_src;
	unsigned char *ptr_dst;

	width = src->width < dst->width ? src->width : dst->width;
	height = src->height < dst->height ? src->height : dst->height;
	ptr_src = src->ptr[0];
	ptr_dst = dst->ptr[0];

	for (y = 0; y < height; y++) {
		const unsigned short *ptr_src16 = (unsigned short *)ptr_src;

		for (x = 0; x < width; x++) {
			rgb565_to_xbgr32(ptr_src16[x], &ptr_dst[x * 4]);
                }
		ptr_dst += dst->stride;
		ptr_src += src->stride;
	}

	return 0;
}

// UYVY/YUYV to XBGR32
enum yuv_coords {
	Y1 = 0,
	U1,
	Y2,
	V2,
	YUVC_MAX
};

static inline void yuv_to_xbgr32(int y, int u, int v,
				 unsigned shift, unsigned char *rgb)
{
	register int r, g, b;

	y >>= shift;
	u >>= shift;
	v >>= shift;

	r = (1192 * (y - 16) + 1634 * (v - 128) ) >> 10;
	g = (1192 * (y - 16) - 833 * (v - 128) - 400 * (u -128) ) >> 10;
	b = (1192 * (y - 16) + 2066 * (u - 128) ) >> 10;

	r = r > 255 ? 255 : r < 0 ? 0 : r;
	g = g > 255 ? 255 : g < 0 ? 0 : g;
	b = b > 255 ? 255 : b < 0 ? 0 : b;

	*rgb++ = b; //B
	*rgb++ = g; //G
	*rgb++ = r; //R
	*rgb   = 0; //A
}

static int __window_yuv8_to_xbgr32(const struct window *src, struct window *dst)
{
	/* for UYVY from camera: covert UYVY to XBGR32 */
	unsigned char cy1, cy2, cu1, cv2;
	unsigned x, y, width, height;
	const unsigned char *ptr_src;
	unsigned char *ptr_dst;

	width = src->width < dst->width ? src->width : dst->width;
	height = src->height < dst->height ? src->height : dst->height;

	ptr_src = src->ptr[0];
	ptr_dst = dst->ptr[0];

	int o[YUVC_MAX];

	switch (src->fourcc) {
	case V4L2_PIX_FMT_UYVY:
		o[U1] = 0;
		o[Y1] = 1;
		o[V2] = 2;
		o[Y2] = 3;
		break;
	case V4L2_PIX_FMT_YUYV:
		o[Y1] = 0;
		o[U1] = 1;
		o[Y2] = 2;
		o[V2] = 3;
		break;
	default:
		return -1;
	}

	for (y = 0; y < height; y++) {
		for (x = 0; x < width; x += 2) {
			cu1 = ptr_src[x * 2 + o[U1]];
			cy1 = ptr_src[x * 2 + o[Y1]];
			cv2 = ptr_src[x * 2 + o[V2]];
			cy2 = ptr_src[x * 2 + o[Y2]];
			yuv_to_xbgr32(cy1, cu1, cv2, 0, &ptr_dst[x * 4]);
			yuv_to_xbgr32(cy2, cu1, cv2, 0, &ptr_dst[x * 4 + 4]);
		}
		ptr_src += src->stride;
		ptr_dst += dst->stride;
	}

	return 0;
}

// NV12/NV16 to XBGR32
static int __window_nv12_to_xbgr32(const struct window *src, struct window *dst)
{
	const unsigned char *bufY = src->ptr[0];
	const unsigned char *bufUV = src->ptr[1];
        unsigned char cy, cu, cv;
	unsigned x, y, width, height;
	unsigned char *ptr_dst = dst->ptr[0];
	unsigned strideUV[2] = { 0, src->stride };

	width = src->width < dst->width ? src->width : dst->width;
	height = src->height < dst->height ? src->height : dst->height;

	switch (src->fourcc) {
	case V4L2_PIX_FMT_NV12:
		break;
	case V4L2_PIX_FMT_NV16:
		strideUV[0] = src->stride;
		break;
	default:
		return -1;
	}

	for (y = 0; y < height; y++) {
		for (x = 0; x < width; x++) {
			int o = x & ~1;

			cy = bufY[x];
			cu = bufUV[o];
			cv = bufUV[o + 1];

			yuv_to_xbgr32(cy, cu, cv, 0, &ptr_dst[4*x]);
		}

		ptr_dst += dst->stride;
		bufY += src->stride;
		bufUV += strideUV[y & 1];
	}

	return 0;
}

// Bayer to XBGR32
enum bayer_type {
	BGGR = 0x0,
	RGGB = 0x3,
};

enum bayer_coords {
	NW = 0,
	NN,
	NE,
	WW,
	CC,
	EE,
	SW,
	SS,
	SE,
	BC_MAX
};

static inline void bayer_to_xbgr32(enum bayer_type bt, unsigned shift,
				   unsigned x, unsigned y,
				   unsigned *w, unsigned char *rgb)
{
	register int r,g,b;

	unsigned code = (x & 1) | ((y & 1) << 1);
	code ^= bt;

	switch (code) {
	case 0x0:
		r = mean4(w[NW], w[NE], w[SW], w[SE]);
		g = mean4(w[NN], w[WW], w[EE], w[SS]);
		b = w[CC];
		break;
	case 0x1:
		r = mean2(w[NN], w[SS]);
		g = w[CC];
		b = mean2(w[WW], w[EE]);
		break;
	case 0x2:
		r = mean2(w[WW], w[EE]);
		g = w[CC];
		b = mean2(w[NN], w[SS]);
		break;
	case 0x3:
		r = w[CC];
		g = mean4(w[NN], w[WW], w[EE], w[SS]);
		b = mean4(w[NW], w[NE], w[SW], w[SE]);
		break;
	}

	*rgb++ = b >> shift; //B
	*rgb++ = g >> shift; //G
	*rgb++ = r >> shift; //R
	*rgb   = 0; //A
}

static int __window_bayer8_to_xbgr32(const struct window *src, struct window *dst)
{
	unsigned w[BC_MAX];
	int o[BC_MAX] = {
		-src->width - 1, -src->width, -src->width + 1,
		-1, 0, 1,
		src->width - 1, src->width, src->width + 1,
	};
	int x, y, width, height, k;
	enum bayer_type bt;
	const unsigned char *ptr_src;
	unsigned char *ptr_dst;

	if (src->width < 2 || src->height < 2)
		return -1;

	switch (src->fourcc) {
	case V4L2_PIX_FMT_SBGGR8:
		bt = BGGR;
		break;
	case V4L2_PIX_FMT_SRGGB8:
		bt = RGGB;
		break;
	default:
		return  -1;
	}

	x = y = k = 0;

	width = src->width - 1;
	height = src->height -1;
	if (width > dst->width)
		width = dst->width;
	if (height > dst->height)
		height = dst->height;

	ptr_src = src->ptr[0];
	ptr_dst = dst->ptr[0];
	// Top-left corner;
	w[NW] = 0;
	w[NN] = 0;
	w[NE] = 0;
	w[WW] = 0;
	w[SW] = 0;
	w[CC] = ptr_src[x + o[CC]];
	w[EE] = ptr_src[x + o[EE]];
	w[SS] = ptr_src[x + o[SS]];
	w[SE] = ptr_src[x + o[SE]];

	bayer_to_xbgr32(bt, 0, x, y, w, &ptr_dst[k]);
	k += 4;

	// First line
	for (x = 1; x < width; x++) {
		w[WW] = ptr_src[x + o[WW]];
		w[CC] = ptr_src[x + o[CC]];
		w[EE] = ptr_src[x + o[EE]];
		w[SS] = ptr_src[x + o[SS]];
		w[SE] = ptr_src[x + o[SE]];
		bayer_to_xbgr32(bt, 0, x, y, w, &ptr_dst[k]);
		k += 4;
	}

	if (x < dst->width) {
		// Top-right corner
		w[WW] = ptr_src[x + o[WW]];
		w[CC] = ptr_src[x + o[CC]];
		w[EE] = 0;
		w[SW] = ptr_src[x + o[SW]];
		w[SS] = ptr_src[x + o[SS]];
		w[SE] = 0;

		bayer_to_xbgr32(bt, 0, x, y, w, &ptr_dst[k]);
	}
	ptr_dst += dst->stride;
	ptr_src += src->stride;
	k = 0;

	// Middle lines
	for (y = 1; y < height; y++) {
		// Leftmost pixel
		x = 0;
		w[NW] = 0;
		w[NN] = ptr_src[x + o[NN]];
		w[NE] = ptr_src[x + o[NE]];
		w[WW] = 0;
		w[CC] = ptr_src[x + o[CC]];
		w[EE] = ptr_src[x + o[EE]];
		w[SW] = 0;
		w[SS] = ptr_src[x + o[SS]];
		w[SE] = ptr_src[x + o[SE]];
		bayer_to_xbgr32(bt, 0, x, y, w, &ptr_dst[k]);
		k += 4;
		// Middle pixels
		for (x = 1; x < width; x++) {
			w[NW] = ptr_src[x + o[NW]];
			w[NN] = ptr_src[x + o[NN]];
			w[NE] = ptr_src[x + o[NE]];
			w[WW] = ptr_src[x + o[WW]];
			w[CC] = ptr_src[x + o[CC]];
			w[EE] = ptr_src[x + o[EE]];
			w[SW] = ptr_src[x + o[SW]];
			w[SS] = ptr_src[x + o[SS]];
			w[SE] = ptr_src[x + o[SE]];
			bayer_to_xbgr32(bt, 0, x, y, w, &ptr_dst[k]);
			k += 4;
		}
		if (x < dst->width) {
			// Rightmost pixel
			w[NW] = ptr_src[x + o[NW]];
			w[NN] = ptr_src[x + o[NN]];
			w[NE] = 0;
			w[WW] = ptr_src[x + o[WW]];
			w[CC] = ptr_src[x + o[CC]];
			w[EE] = 0;
			w[SW] = ptr_src[x + o[SW]];
			w[SS] = ptr_src[x + o[SS]];
			w[SE] = 0;
			bayer_to_xbgr32(bt, 0, x, y, w, &ptr_dst[k]);
		}
		ptr_dst += dst->stride;
		ptr_src += src->stride;
		k = 0;
	}
	if (y < dst->height) {
		// Bottom-left corner
		x = 0;
		w[NW] = 0;
		w[NN] = ptr_src[x + o[NN]];
		w[NE] = ptr_src[x + o[NE]];
		w[WW] = 0;
		w[CC] = ptr_src[x + o[CC]];
		w[EE] = ptr_src[x + o[EE]];
		w[SW] = 0;
		w[SS] = 0;
		w[SE] = 0;
		bayer_to_xbgr32(bt, 0, x, y, w, &ptr_dst[k]);
		k += 4;
		// Last line
		for (x = 1; x < width; x++) {
			w[NW] = ptr_src[x + o[NW]];
			w[NN] = ptr_src[x + o[NN]];
			w[NE] = ptr_src[x + o[NE]];
			w[WW] = ptr_src[x + o[WW]];
			w[CC] = ptr_src[x + o[CC]];
			w[EE] = ptr_src[x + o[EE]];
			bayer_to_xbgr32(bt, 0, x, y, w, &ptr_dst[k]);
			k += 4;
		}
		if (x < dst->width) {
			// Bottom-right corner
			w[NW] = ptr_src[x + o[NW]];
			w[NN] = ptr_src[x + o[NN]];
			w[NE] = 0;
			w[WW] = ptr_src[x + o[WW]];
			w[CC] = ptr_src[x + o[CC]];
			w[EE] = 0;
			w[SW] = 0;
			w[SS] = 0;
			w[SE] = 0;
			bayer_to_xbgr32(bt, 0, x, y, w, &ptr_dst[k]);
		}
	}

	return 0;
}

static int __window_bayer16_to_xbgr32(const struct window *src, struct window *dst)
{
	unsigned w[BC_MAX];
	int o[BC_MAX] = {
		-src->width - 1, -src->width, -src->width + 1,
		-1, 0, 1,
		src->width - 1, src->width, src->width + 1,
	};
	int x, y, width, height, k;
	enum bayer_type bt;

	unsigned shift;
	unsigned char *ptr_dst = dst->ptr[0];
	const unsigned char *ptr_src16 = (const unsigned char *)src->ptr[0];
	unsigned stride16 = src->stride >> 1;

	if (src->width < 2 || src->height < 2)
		return -1;

	switch (src->fourcc) {
	case V4L2_PIX_FMT_SBGGR12:
		bt = BGGR;
		shift = 4;
		break;
	case V4L2_PIX_FMT_SBGGR16:
		bt = BGGR;
		shift = 8;
		break;
	case V4L2_PIX_FMT_SRGGB12:
		shift = 4;
		bt = RGGB;
		break;
	case V4L2_PIX_FMT_SRGGB16:
		shift = 8;
		bt = RGGB;
		break;
	default:
		return -1;
	}

	x = y = k = 0;

	width = src->width - 1;
	height = src->height -1;
	if (width > dst->width)
		width = dst->width;
	if (height > dst->height)
		height = dst->height;

	// Top-left corner;
	w[NW] = 0;
	w[NN] = 0;
	w[NE] = 0;
	w[WW] = 0;
	w[SW] = 0;
	w[CC] = ptr_src16[x + o[CC]];
	w[EE] = ptr_src16[x + o[EE]];
	w[SS] = ptr_src16[x + o[SS]];
	w[SE] = ptr_src16[x + o[SE]];

	bayer_to_xbgr32(bt, shift, x, y, w, &ptr_dst[k]);
	k += 4;

	// First line
	for (x = 1; x < width; x++) {
		w[WW] = ptr_src16[x + o[WW]];
		w[CC] = ptr_src16[x + o[CC]];
		w[EE] = ptr_src16[x + o[EE]];
		w[SS] = ptr_src16[x + o[SS]];
		w[SE] = ptr_src16[x + o[SE]];
		bayer_to_xbgr32(bt, shift, x, y, w, &ptr_dst[k]);
		k += 4;
	}

	if (x < dst->width) {
		// Top-right corner
		w[WW] = ptr_src16[x + o[WW]];
		w[CC] = ptr_src16[x + o[CC]];
		w[EE] = 0;
		w[SW] = ptr_src16[x + o[SW]];
		w[SS] = ptr_src16[x + o[SS]];
		w[SE] = 0;

		bayer_to_xbgr32(bt, shift, x, y, w, &ptr_dst[k]);
	}
        ptr_dst += dst->stride;
	ptr_src16 += stride16;
	k = 0;

	// Middle lines
	for (y = 1; y < height; y++) {
		// Leftmost pixel
		x = 0;
		w[NW] = 0;
		w[NN] = ptr_src16[x + o[NN]];
		w[NE] = ptr_src16[x + o[NE]];
		w[WW] = 0;
		w[CC] = ptr_src16[x + o[CC]];
		w[EE] = ptr_src16[x + o[EE]];
		w[SW] = 0;
		w[SS] = ptr_src16[x + o[SS]];
		w[SE] = ptr_src16[x + o[SE]];
		bayer_to_xbgr32(bt, shift, x, y, w, &ptr_dst[k]);
		k += 4;
		// Middle pixels
		for (x = 1; x < width; x++) {
			w[NW] = ptr_src16[x + o[NW]];
			w[NN] = ptr_src16[x + o[NN]];
			w[NE] = ptr_src16[x + o[NE]];
			w[WW] = ptr_src16[x + o[WW]];
			w[CC] = ptr_src16[x + o[CC]];
			w[EE] = ptr_src16[x + o[EE]];
			w[SW] = ptr_src16[x + o[SW]];
			w[SS] = ptr_src16[x + o[SS]];
			w[SE] = ptr_src16[x + o[SE]];
			bayer_to_xbgr32(bt, shift, x, y, w, &ptr_dst[k]);
			k += 4;
		}
		if (x < dst->width) {
			// Rightmost pixel
			w[NW] = ptr_src16[x + o[NW]];
			w[NN] = ptr_src16[x + o[NN]];
			w[NE] = 0;
			w[WW] = ptr_src16[x + o[WW]];
			w[CC] = ptr_src16[x + o[CC]];
			w[EE] = 0;
			w[SW] = ptr_src16[x + o[SW]];
			w[SS] = ptr_src16[x + o[SS]];
			w[SE] = 0;
			bayer_to_xbgr32(bt, 0, x, y, w, &ptr_dst[k]);
		}
	        ptr_dst += dst->stride;
	        ptr_src16 += stride16;
		k = 0;
	}
	if (y < dst->height) {
		// Bottom-left corner
		x = 0;
		w[NW] = 0;
		w[NN] = ptr_src16[x + o[NN]];
		w[NE] = ptr_src16[x + o[NE]];
		w[WW] = 0;
		w[CC] = ptr_src16[x + o[CC]];
		w[EE] = ptr_src16[x + o[EE]];
		w[SW] = 0;
		w[SS] = 0;
		w[SE] = 0;
		bayer_to_xbgr32(bt, 0, x, y, w, &ptr_dst[k]);
		k += 4;
		// Last line
		for (x = 1; x < width; x++) {
			w[NW] = ptr_src16[x + o[NW]];
			w[NN] = ptr_src16[x + o[NN]];
			w[NE] = ptr_src16[x + o[NE]];
			w[WW] = ptr_src16[x + o[WW]];
			w[CC] = ptr_src16[x + o[CC]];
			w[EE] = ptr_src16[x + o[EE]];
			bayer_to_xbgr32(bt, 0, x, y, w, &ptr_dst[k]);
			k += 4;
		}
		if (x < dst->width) {
			// Bottom-right corner
			w[NW] = ptr_src16[x + o[NW]];
			w[NN] = ptr_src16[x + o[NN]];
			w[NE] = 0;
			w[WW] = ptr_src16[x + o[WW]];
			w[CC] = ptr_src16[x + o[CC]];
			w[EE] = 0;
			w[SW] = 0;
			w[SS] = 0;
			w[SE] = 0;
			bayer_to_xbgr32(bt, 0, x, y, w, &ptr_dst[k]);
		}
	}
}

/* FPS counter public API */
int fps_count(struct fps_counter *fps)
{
	struct timeval t;
	int f;

	gettimeofday(&t, NULL);
	fps->usec += fps->frames++ ? usec_elapsed(&t, &fps->frame_time) : 0;
	fps->frame_time = t;
	if (fps->usec < 1000000)
		return -EAGAIN;

	f= ((unsigned long long)fps->frames * 10000000 + fps->usec - 1) / fps->usec;
	fps->usec = 0;
	fps->frames = 0;

	return f;
}

int fps_count_release(struct fps_counter *fps)
{
	free(fps);
	return 0;
}

struct fps_counter *fps_count_create(void)
{
	struct fps_counter *fps;

	fps = calloc(1, sizeof(*fps));
	if (!fps) {
		fprintf(stderr, "Out of memory\n");
		return NULL;
	}

	return fps;
}

/* Window public API */
int window_init(struct window *window, unsigned width, unsigned height,
		unsigned stride, unsigned fourcc)
{
	unsigned bpp;

	bpp = format_to_bpp(fourcc);
	if (bpp < 1)
		return -1;

	if (stride < bpp * width)
		return -1;

	window->ptr[0] = NULL;
	window->ptr[1] = NULL;
	window->size[0] = stride * height;
	window->fourcc = fourcc;
	window->bpp = bpp;
	window->width = width;
	window->height = height;
	window->stride = stride;

	switch (fourcc) {
	case V4L2_PIX_FMT_NV12:
		window->size[1] = window->size[0] / 2;
		break;
	case V4L2_PIX_FMT_NV16:
		window->size[1] = window->size[0];
		break;
	default:
		window->size[1] = 0;
		break;
	}

	return 0;
}

int window_bind_buffers(struct window *window, struct buffer *plane1,
			struct buffer *plane2)
{
	unsigned char *ptr[2];
	unsigned size;

	if (!plane1)
		return -1;

	ptr[0] = plane1->start;
	size = window->size[0];
	if (!window->size[1]) {
		ptr[1] = NULL;
	} else if (!plane2) {
		size += window->size[1];
		ptr[1] = plane1->start + window->size[0];
	} else {
		if (plane2->length < window->size[1])
			return -1;
		ptr[1] = plane2->start;
	}

	if (plane1->length < size)
		return -1;

	window->ptr[0] = ptr[0];
	window->ptr[1] = ptr[1];
	return 0;
}

int window_subwindow_get(const struct window *src, struct window *dst,
			 unsigned top, unsigned left,
			 unsigned width, unsigned height)
{
	size_t x, y, bpp, offset;

	if (!dst || !src || !src->ptr[0])
		return -1;

	x = src->width - left;
	y = src->height - top;
	if (x < 0 || y < 0)
		return -1;

	dst->width = width < x ? width : x;
	dst->height = height < y ? height : y;
	dst->fourcc = src->fourcc;
	dst->stride = src->stride;
	top *= src->stride;
	bpp = src->bpp;
	offset = top + bpp * left;
	dst->bpp = bpp;
	dst->ptr[0] = src->ptr[0] + offset;
	dst->size[0] = src->size[0] - offset;

	switch (src->fourcc) {
	case V4L2_PIX_FMT_NV12:
		top /= 2;
		break;
	case V4L2_PIX_FMT_NV16:
		break;
	default:
		dst->ptr[1] = NULL;
		dst->size[1] = 0;
		return 0;
	}

	left /= 2;
	bpp *= 2;
	offset = top + bpp * left;
	dst->ptr[1] = src->ptr[1] + offset;
	dst->size[1] = src->size[1] - offset;
	return 0;
}

static int __window_copy(const struct window *src, struct window *dst)
{
	unsigned y, width, height, bpp;
	const unsigned char *ptr_src;
	unsigned char *ptr_dst;

	width = src->width < dst->width ? src->width : dst->width;
	height = src->height < dst->height ? src->height : dst->height;
	bpp = src->bpp;
	ptr_src = src->ptr[0];
	ptr_dst = dst->ptr[0];

	for (y = 0; y < height; y++) {
		memcpy(ptr_dst, ptr_src, width * bpp);
		ptr_dst += dst->stride;
		ptr_src += src->stride;
	}

	switch (src->fourcc) {
	case V4L2_PIX_FMT_NV12:
		height /= 2;
		break;
	case V4L2_PIX_FMT_NV16:
		break;
	default:
		return 0;
	}

	bpp *= 2;
	width /= 2;
	ptr_src = src->ptr[1];
	ptr_dst = dst->ptr[1];
	for (y = 0; y < height; y++) {
		memcpy(ptr_dst, ptr_src, width * bpp);
		ptr_dst += dst->stride;
		ptr_src += src->stride;
	}

	return 0;
}

int window_copy(const struct window *src, struct window *dst)
{
	if (src->fourcc == dst->fourcc)
		return __window_copy(src, dst);

	if (dst->fourcc != V4L2_PIX_FMT_XBGR32)
		return -1;

	switch (src->fourcc) {
	case V4L2_PIX_FMT_UYVY:
	case V4L2_PIX_FMT_YUYV:
		return __window_yuv8_to_xbgr32(src, dst);
	case V4L2_PIX_FMT_RGB565:
		return __window_rgb565_to_xbgr32(src, dst);
	case V4L2_PIX_FMT_NV12:
	case V4L2_PIX_FMT_NV16:
		return __window_nv12_to_xbgr32(src, dst);
	case V4L2_PIX_FMT_GREY:
		return __window_grey8_to_xbgr32(src, dst);
	case V4L2_PIX_FMT_Y10:
	case V4L2_PIX_FMT_Y12:
	case V4L2_PIX_FMT_Y14:
	case V4L2_PIX_FMT_Y16:
		return __window_grey16_to_xbgr32(src, dst);
	case V4L2_PIX_FMT_SBGGR8:
	case V4L2_PIX_FMT_SRGGB8:
		return __window_bayer8_to_xbgr32(src, dst);
	case V4L2_PIX_FMT_SBGGR12:
	case V4L2_PIX_FMT_SRGGB12:
	case V4L2_PIX_FMT_SBGGR16:
	case V4L2_PIX_FMT_SRGGB16:
		return __window_bayer16_to_xbgr32(src, dst);
	default:
		break;
	}
	return -1;
}

/* Camera private functions */
static int __camera_buffers_release(struct camera *cam)
{
	free(cam->buffers);
	cam->buffers = NULL;
	cam->n_buffers = 0;
	return 0;
}

static int __camera_buffers_alloc(struct camera *cam, unsigned n_bufs)
{
	cam->buffers = calloc(n_bufs, sizeof(*cam->buffers));
        if (!cam->buffers) {
		fprintf(stderr, "Out of memory\n");
		return -1;
        }
	cam->n_buffers = n_bufs;
	return 0;
}

// IO_METHOD_READ
static int __camera_buf_release_read(struct camera *cam)
{
	unsigned i;

	for (i = 0; i < cam->n_buffers; i++) {
		free(cam->buffers[i].start);
		cam->buffers[i].start = NULL;
		cam->buffers[i].length = 0;
	}

	__camera_buffers_release(cam);
	return 0;
}

static int __camera_buf_alloc_read(struct camera *cam, unsigned size)
{
	unsigned i;
	int retval;

	retval = __camera_buffers_alloc(cam, 1);
	if (retval)
		return retval;

	for (i = 0; i < cam->n_buffers; i++) {
		cam->buffers[i].start = malloc(size);
		if (!(cam->buffers[i].start)) {
			fprintf(stderr, "Out of memory\n");
			break;
		}
	        cam->buffers[i].length = size;
	}

	if (i == cam->n_buffers)
		return 0;

	__camera_buf_release_read(cam);
	return -1;
}

#define __camera_frame_put_read	NULL

static int __camera_frame_get_read(struct camera *cam, struct v4l2_buffer *buf)
{
	size_t r;

	r = read(cam->fd, cam->buffers[0].start, cam->buffers[0].length);
	if (r < 0) {
		fprintf(stderr, "%s: read: %s\n",
			cam->dev_name, strerror(errno));
		return -1;
	}

	buf->bytesused = r;
	buf->type = 0;
	buf->memory = 0;
	buf->index = 0;
	return 0;
}

#define __camera_stop_read	NULL
#define __camera_start_read	NULL

// IO_METHOD_MMAP
static int __camera_buf_release_mmap(struct camera *cam)
{
	unsigned i;

	for (i = 0; i < cam->n_buffers; i++) {
		if (cam->buffers[i].start) {
			if (-1 == munmap(cam->buffers[i].start, cam->buffers[i].length))
				fprintf(stderr, "%s: munmap: %s\n",
					cam->dev_name, strerror(errno));
			cam->buffers[i].start = NULL;
		}
		cam->buffers[i].length = 0;
	}

	__camera_buffers_release(cam);
	return 0;
}

static int __camera_buf_alloc_mmap(struct camera *cam, unsigned size)
{
        struct v4l2_requestbuffers req;
	unsigned i;
	int retval;

	(void)size;

	CLEAR(req);

	req.count = 7;
	req.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	req.memory = V4L2_MEMORY_MMAP;

	if (-1 == ioctl(cam->fd, VIDIOC_REQBUFS, &req)) {
		if (EINVAL == errno)
			fprintf(stderr, "%s does not support memory mapping\n",
				cam->dev_name);
		else
			fprintf(stderr, "%s: VIDIOC_REQBUFS: %s\n",
				cam->dev_name, strerror(errno));
		return -1;
        }

	if (req.count < 2) {
		fprintf(stderr, "Insufficient buffer memory on %s\n",
			cam->dev_name);
		return -1;
        }

	retval = __camera_buffers_alloc(cam, req.count);
	if (retval)
		return retval;

	for (i = 0; i < cam->n_buffers; ++i) {
		struct v4l2_buffer buf;

		CLEAR(buf);

		buf.type        = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		buf.memory      = V4L2_MEMORY_MMAP;
		buf.index       = i;

		if (-1 == ioctl(cam->fd, VIDIOC_QUERYBUF, &buf)) {
			fprintf(stderr, "%s: VIDIOC_QUERYBUF: %s\n",
				cam->dev_name, strerror(errno));
			break;
		}

		cam->buffers[i].start = mmap(NULL /* start anywhere */,
					     buf.length,
					     PROT_READ | PROT_WRITE /* required */,
					     MAP_SHARED /* recommended */,
					     cam->fd, buf.m.offset);
		if (MAP_FAILED == cam->buffers[i].start) {
			cam->buffers[i].start = NULL;
			fprintf(stderr, "%s: mmap: %s\n",
				cam->dev_name, strerror(errno));
			break;
		}
		cam->buffers[i].length = buf.length;
        }

	if (i == cam->n_buffers)
		return 0;

	__camera_buf_release_mmap(cam);
	return -1;
}

static int __camera_frame_put_mmap(struct camera *cam, struct v4l2_buffer *buf)
{
	if (-1 == ioctl(cam->fd, VIDIOC_QBUF, buf)) {
		fprintf(stderr, "%s: VIDIOC_QBUF: %s\n",
			cam->dev_name, strerror(errno));
		return -1;
	}
	return 0;
}

static int __camera_frame_get_mmap(struct camera *cam, struct v4l2_buffer *buf)
{
	buf->type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	buf->memory = V4L2_MEMORY_MMAP;

	if (-1 == ioctl(cam->fd, VIDIOC_DQBUF, buf)) {
		fprintf(stderr, "%s: VIDIOC_DQBUF: %s\n",
			cam->dev_name, strerror(errno));
		return -1;
	}

	return 0;
}

static int __camera_stop_mmap(struct camera *cam)
{
        enum v4l2_buf_type type;

	type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	if (-1 == ioctl(cam->fd, VIDIOC_STREAMOFF, &type)) {
		fprintf(stderr, "%s: VIDIOC_STREAMOFF: %s\n",
			cam->dev_name, strerror(errno));
		return -1;
        }
	return 0;
}

static int __camera_start_mmap(struct camera *cam)
{
        enum v4l2_buf_type type;
        unsigned i;

	for (i = 0; i < cam->n_buffers; ++i) {
		struct v4l2_buffer buf;

		CLEAR(buf);
		buf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		buf.memory = V4L2_MEMORY_MMAP;
		buf.index = i;

		if (-1 == ioctl(cam->fd, VIDIOC_QBUF, &buf)) {
			fprintf(stderr, "%s: VIDIOC_QBUF: %s\n",
				cam->dev_name, strerror(errno));
			return -1;
                }

	}

	type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	if (-1 == ioctl(cam->fd, VIDIOC_STREAMON, &type)) {
		fprintf(stderr, "%s: VIDIOC_STREAMON: %s\n",
			cam->dev_name, strerror(errno));
		return -1;
	}

	return 0;
}

// IO_METHOD_USERP
#define __camera_buf_release_userp	__camera_buf_release_mmap

static int __camera_buf_alloc_userp(struct camera *cam, unsigned size)
{
	struct v4l2_requestbuffers req;
	long page_size;
	unsigned page_mask, i;
	int retval;

	page_size = sysconf(_SC_PAGE_SIZE);
	if (page_size < 0) {
		perror("sysconf _SC_PAGE_SIZE");
		return -1;
	}

	page_mask = page_size - 1;
	if (page_size & page_mask) {
		fprintf(stderr, "invalid page size\n");
		return -1;
	}

	size = (size + page_mask) & ~page_mask;

	CLEAR(req);

	req.count  = 4;
	req.type   = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	req.memory = V4L2_MEMORY_USERPTR;

	if (-1 == ioctl(cam->fd, VIDIOC_REQBUFS, &req)) {
		if (EINVAL == errno)
			fprintf(stderr, "%s does not support user pointer i/o\n",
				cam->dev_name);
		else
			fprintf(stderr, "%s: VIDIOC_REQBUFS: %s\n",
				cam->dev_name, strerror(errno));
		return -1;
	}


	retval = __camera_buffers_alloc(cam, req.count);
	if (retval)
		return retval;

        for (i = 0; i < cam->n_buffers; ++i) {
		cam->buffers[i].start = mmap(NULL /* start anywhere */,
					     size,
					     PROT_READ | PROT_WRITE /* required */,
					     MAP_ANONYMOUS | MAP_PRIVATE,
					     -1, 0);
		if (MAP_FAILED == cam->buffers[i].start) {
			cam->buffers[i].start = NULL;
			fprintf(stderr, "Out of memory\n");
			break;
                }
		cam->buffers[i].length = size;
        }

	if (i == cam->n_buffers)
		return 0;

	__camera_buf_release_userp(cam);
	return -1;
}

#define __camera_stop_userp	__camera_stop_mmap

static int __camera_start_userp(struct camera *cam)
{
        enum v4l2_buf_type type;
	unsigned i;

	for (i = 0; i < cam->n_buffers; ++i) {
		struct v4l2_buffer buf;

		CLEAR(buf);
		buf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		buf.memory = V4L2_MEMORY_USERPTR;
		buf.index = i;
		buf.m.userptr = (unsigned long)cam->buffers[i].start;
		buf.length = cam->buffers[i].length;

		if (-1 == ioctl(cam->fd, VIDIOC_QBUF, &buf)) {
			fprintf(stderr, "%s: VIDIOC_QBUF: %s\n",
				cam->dev_name, strerror(errno));
			return -1;
		}

        }

	type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	if (-1 == ioctl(cam->fd, VIDIOC_STREAMON, &type)) {
		fprintf(stderr, "%s: VIDIOC_STREAMON: %s\n",
			cam->dev_name, strerror(errno));
		return -1;
	}

	return 0;
}

#define __camera_frame_put_userp	__camera_frame_put_mmap

static int __camera_frame_get_userp(struct camera *cam, struct v4l2_buffer *buf)
{
	buf->type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	buf->memory = V4L2_MEMORY_USERPTR;

	if (-1 == ioctl(cam->fd, VIDIOC_DQBUF, buf)) {
		fprintf(stderr, "%s: VIDIOC_DQBUF: %s\n",
			cam->dev_name, strerror(errno));
		return -1;
	}

	return 0;
}

/* Camera public API */
static inline int camera_stop(struct camera *cam)
{
	if (!cam->stop)
		return 0;

	return cam->stop(cam);
}

static inline int camera_start(struct camera *cam)
{
	if (!cam->start)
		return 0;

	return cam->start(cam);
}

static inline int camera_fd_set(struct camera *cam, int *nfd, fd_set *fds)
{
	if (!nfd || !fds)
		return -1;

	if (*nfd < cam->fd)
		*nfd = cam->fd;

	FD_SET(cam->fd, fds);
	return 0;
}

static inline int camera_fd_isset(struct camera *cam, fd_set *fds)
{
	return FD_ISSET(cam->fd, fds);
}

const struct window *camera_frame_window_set(struct camera *cam,
					     struct v4l2_buffer *buf)
{
	if (buf->index > cam->n_buffers)
		return NULL;

	if (window_bind_buffers(&cam->window, &cam->buffers[buf->index], NULL))
		return NULL;

	return &cam->window;
}

static inline int camera_frame_dump(struct camera *cam, struct v4l2_buffer *buf,
				    FILE *fp)
{
	if (buf->index > cam->n_buffers)
		return -1;

	if (!fp)
		return -1;

	return fwrite(cam->buffers[buf->index].start,
		      cam->buffers[buf->index].length, 1, fp);
}

static inline int camera_frame_put(struct camera *cam, struct v4l2_buffer *buf)
{
	return cam->frame_put(cam, buf);
}

static inline int camera_frame_get(struct camera *cam, struct v4l2_buffer *buf)
{
	int retval;

	CLEAR(*buf);

	retval = cam->frame_get(cam, buf);
	if (retval < 0)
		return retval;

	if (cam->fps) {
		int fps = fps_count(cam->fps);

		if (fps >= 0)
			fprintf(stderr, "%s: FPS: %3u.%1u\n", cam->dev_name, fps / 10, fps % 10);
	}
	return 0;
}

void camera_release(struct camera *cam)
{
	if (!cam)
		return;

	fps_count_release(cam->fps);

	if (cam->fd >= 0) {
		camera_stop(cam);
		if (cam->buf_release)
			cam->buf_release(cam);
		close(cam->fd);
	}
	free(cam->buffers);
	free(cam->dev_name);
	free(cam);
}

struct camera *camera_create(const char *dev_name,
			     unsigned left, unsigned top,
			     unsigned width, unsigned height,
			     unsigned format, enum io_method io,
			     unsigned framerate, int fps_count)
{
	struct camera *cam;
	struct v4l2_format fmt;
        struct v4l2_capability cap;
        struct v4l2_cropcap cropcap;
        struct v4l2_crop crop;
        struct stat st;
	char fourcc[FORMAT_STR_MAX];

	cam = calloc(1, sizeof(*cam));
	if (!cam)
		return NULL;

	cam->dev_name = strdup(dev_name);
	if (!cam->dev_name) {
		fprintf(stderr, "%s: strdup: %s\n", dev_name, strerror(errno));
		goto err_free;
	}

	if (-1 == stat(cam->dev_name, &st)) {
		fprintf(stderr, "Cannot identify '%s': %d, %s\n",
			cam->dev_name, errno, strerror(errno));
		goto err_free;
        }

	if (!S_ISCHR(st.st_mode)) {
		fprintf(stderr, "%s is no device\n", cam->dev_name);
		goto err_free;
        }

	cam->fd = open(cam->dev_name, O_RDWR /* required */ | O_NONBLOCK, 0);
	if (-1 == cam->fd) {
		fprintf(stderr, "Cannot open '%s': %d, %s\n",
			cam->dev_name, errno, strerror(errno));
		goto err_free;
        }

	if (-1 == ioctl(cam->fd, VIDIOC_QUERYCAP, &cap)) {
		if (EINVAL == errno)
			fprintf(stderr, "%s is no V4L2 device\n", cam->dev_name);
		else
			fprintf(stderr, "%s: VIDIOC_QUERYCAP: %s\n",
				cam->dev_name, strerror(errno));
		goto err_close;
        }

	if (!(cap.capabilities & V4L2_CAP_VIDEO_CAPTURE)) {
                fprintf(stderr, "%s is no video capture device\n", cam->dev_name);
		goto err_close;
        }

	if (io == IO_METHOD_READ) {
		if (!(cap.capabilities & V4L2_CAP_READWRITE)) {
			fprintf(stderr, "%s does not support read i/o\n",
				cam->dev_name);
			goto err_close;
		}
	} else {
                if (!(cap.capabilities & V4L2_CAP_STREAMING)) {
                        fprintf(stderr, "%s does not support streaming i/o\n",
				cam->dev_name);
			goto err_close;
                }
	}

	switch (io) {
	case IO_METHOD_READ:
		cam->buf_alloc = __camera_buf_alloc_read;
		cam->buf_release = __camera_buf_release_read;
		cam->frame_get = __camera_frame_get_read;
		cam->frame_put = __camera_frame_put_read;
		cam->start = __camera_start_read;
		cam->stop = __camera_stop_read;
		break;
	case IO_METHOD_MMAP:
		cam->buf_alloc = __camera_buf_alloc_mmap;
		cam->buf_release = __camera_buf_release_mmap;
		cam->frame_get = __camera_frame_get_mmap;
		cam->frame_put = __camera_frame_put_mmap;
		cam->start = __camera_start_mmap;
		cam->stop = __camera_stop_mmap;
		break;
	case IO_METHOD_USERPTR:
		cam->buf_alloc = __camera_buf_alloc_userp;
		cam->buf_release = __camera_buf_release_userp;
		cam->frame_get = __camera_frame_get_userp;
		cam->frame_put = __camera_frame_put_userp;
		cam->start = __camera_start_userp;
		cam->stop = __camera_stop_userp;
		break;
	default:
		goto err_close;
	}

	/* Select video input, video standard and tune here. */
	CLEAR(cropcap);
	cropcap.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	if (0 == ioctl(cam->fd, VIDIOC_CROPCAP, &cropcap)) {
		crop.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		crop.c = cropcap.defrect; /* reset to default */

		crop.c.left = left;
		crop.c.top = top;
		crop.c.width = width;
		crop.c.height = height;

		if (-1 == ioctl(cam->fd, VIDIOC_S_CROP, &crop)) {
			switch (errno) {
			case EINVAL:
				/* Cropping not supported. */
				break;
			default:
				/* Errors ignored. */
				break;
			}
		}
	} else {
		/* Errors ignored. */
	}

	if (framerate) {
		struct v4l2_streamparm parm;

		parm.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		if (-1 == ioctl(cam->fd, VIDIOC_G_PARM, &parm)) {
			fprintf(stderr, "%s: VIDIOC_G_PARM: %s\n",
				cam->dev_name, strerror(errno));
			goto err_close;
		}

		parm.parm.capture.timeperframe.numerator = 1;
		parm.parm.capture.timeperframe.denominator = framerate;
		if (-1 == ioctl(cam->fd, VIDIOC_S_PARM, &parm)) {
			fprintf(stderr, "%s: VIDIOC_S_PARM: %s\n",
				cam->dev_name, strerror(errno));
			goto err_close;
		}
        }

	fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	fmt.fmt.pix.width  = width;
	fmt.fmt.pix.height = height;
	fmt.fmt.pix.field  = FIELD;
	fmt.fmt.pix.pixelformat = format;

	if (!fmt.fmt.pix.pixelformat) {
		/* Preserve original settings as set by v4l2-ctl for example */
		if (-1 == ioctl(cam->fd, VIDIOC_G_FMT, &fmt)) {
			fprintf(stderr, "%s: VIDIOC_G_FMT: %s\n",
				cam->dev_name, strerror(errno));
			goto err_close;
		}
	}

	if (-1 == ioctl(cam->fd, VIDIOC_S_FMT, &fmt)) {
		fprintf(stderr, "%s: VIDIOC_S_FMT: %s\n",
			cam->dev_name, strerror(errno));
		goto err_close;
	}

	if (window_init(&cam->window, fmt.fmt.pix.width, fmt.fmt.pix.height,
			fmt.fmt.pix.bytesperline, fmt.fmt.pix.pixelformat))
		goto err_close;

	if (!cam->buf_alloc)
		goto err_close;

	if (cam->buf_alloc(cam, fmt.fmt.pix.sizeimage) < 0)
		goto err_close;

	if (fps_count)
		cam->fps = fps_count_create();

	format_to_str(fourcc, cam->window.fourcc);
	fprintf(stderr, "%s: %u x %u, %s\n", cam->dev_name,
		cam->window.width, cam->window.height, fourcc);

	return cam;

err_close:
	close(cam->fd);
err_free:
	free(cam->dev_name);
	free(cam);
	return NULL;
}

/* Framebuffer public API */
struct window *fb_windows_create(struct fb *fb, unsigned n_windows)
{
	unsigned x0, nx, ny, n, i;
	unsigned xres, yres, top, left;
	struct window *windows;
	int retval;

	if (!n_windows)
		return NULL;

	windows = calloc(n_windows, sizeof(*windows));
	if (!windows) {
		fprintf(stderr, "Out of memory\n");
		return NULL;
	}

	/* Single window */
	n = n_windows;
	/* Calculate integer square root of n_windows */
	if (n > 1) {
		i = n_windows / 2;
		do {
			n = i;
			i = (n + n_windows / n) / 2;
		} while (i < n);
	}

	/* Calculate number of window columns and rows */
	i = (n_windows + n - 1) / n;
	if (fb->window.width > fb->window.height) {
		nx = i;
		ny = n;
	} else {
		ny = i;
		nx = n;
	}

	/* Calculate window sizes */
	xres = fb->window.width / nx;
	yres = fb->window.height / ny;

	top = (fb->window.height % ny) / 2;
	x0 = (fb->window.width % nx) / 2;

	i = 0;
	while (ny-- > 0) {
		/* Adjust the number and position of the last row windows */
		if (!ny) {
			nx = n_windows - i;
			x0 = (fb->window.width - xres * nx) / 2;
		}

		left = x0;
		for (n = 0; n < nx; n++) {
			int retval;

			retval = window_subwindow_get(&fb->window, &windows[i],
						      top, left, xres, yres);
			if (retval) {
				fprintf(stderr, "FB window error\n");
				goto err_free;
			}

			left += xres;
			i++;
		}
		top += yres;
	}

	return windows;
err_free:
	free(windows);
	return NULL;
}

#ifdef DRM_ENABLE
static inline unsigned pix_format_to_drm_format(unsigned fourcc)
{
	switch (fourcc) {
	case V4L2_PIX_FMT_UYVY:
		return DRM_FORMAT_UYVY;
	case V4L2_PIX_FMT_YUYV:
		return DRM_FORMAT_YUYV;
	case V4L2_PIX_FMT_RGB565:
                return DRM_FORMAT_RGB565;
	case V4L2_PIX_FMT_Y10:
		return DRM_FORMAT_Y210;
	case V4L2_PIX_FMT_Y12:
		return DRM_FORMAT_Y212;
	case V4L2_PIX_FMT_Y16:
		return DRM_FORMAT_Y216;
	case V4L2_PIX_FMT_NV12:
                return DRM_FORMAT_NV12;
	case V4L2_PIX_FMT_NV16:
                return DRM_FORMAT_NV16;
	default:
                return DRM_FORMAT_XRGB8888;
	}
}

static inline uint32_t drm_format_to_pix_format(uint32_t fourcc)
{
	switch (fourcc) {
	case DRM_FORMAT_UYVY:
		return V4L2_PIX_FMT_UYVY;
	case DRM_FORMAT_YUYV:
		return V4L2_PIX_FMT_YUYV;
	case DRM_FORMAT_RGB565:
		return V4L2_PIX_FMT_RGB565;
	case DRM_FORMAT_Y210:
		return V4L2_PIX_FMT_Y10;
	case DRM_FORMAT_Y212:
		return V4L2_PIX_FMT_Y12;
	case DRM_FORMAT_Y216:
		return V4L2_PIX_FMT_Y16;
	case DRM_FORMAT_NV12:
		return V4L2_PIX_FMT_NV12;
        case DRM_FORMAT_NV16:
		return V4L2_PIX_FMT_NV16;
	default:
                return V4L2_PIX_FMT_XBGR32;
	}
}

void drmfb_release(struct fb *fb)
{
	struct drmfb *drmfb = to_drmfb(fb);
	unsigned i;

	if (drmfb->saved_crtc) {
		drmModeSetCrtc(drmfb->fb.fd, drmfb->saved_crtc->crtc_id,
			       drmfb->saved_crtc->buffer_id,
			       drmfb->saved_crtc->x, drmfb->saved_crtc->y,
			       &drmfb->conn_id, 1, &drmfb->saved_crtc->mode);
		drmModeFreeCrtc(drmfb->saved_crtc);
		drmfb->saved_crtc = NULL;
	}

	drmModeRmFB(drmfb->fb.fd, drmfb->fb_id);
	for (i = 0; i < drmfb->num_planes; i++) {
		struct drm_mode_destroy_dumb dreq = {
			.handle = drmfb->planes[i].handle,
		};

		munmap(drmfb->planes[i].buffer.start, drmfb->planes[i].buffer.length);
		drmIoctl(drmfb->fb.fd, DRM_IOCTL_MODE_DESTROY_DUMB, &dreq);
	}
	free(drmfb->planes);
	free(drmfb);
}

struct drmfb *drmfb_setup(int fd, drmModeConnector *conn, unsigned format)
{
	char fmtstr[FORMAT_STR_MAX];
	uint32_t drmfmt, crtc_id;
	unsigned i, num_planes;
	struct drmfb_info info;
	struct drmplane *planes;
	struct drmfb *drmfb;
	drmModeModeInfo mode;
	drmModeEncoder *enc;
	int retval;

	drmfb = calloc(1, sizeof(*drmfb));
	if (!drmfb) {
		fprintf(stderr, "Out of memory\n");
		return NULL;
	}

	memcpy(&mode, &conn->modes[0], sizeof(drmModeModeInfo));
	memset(&info, 0, sizeof(info));

	drmfmt = pix_format_to_drm_format(format);
	format = drm_format_to_pix_format(drmfmt);

	info.xsub[0] = 1;
	info.ysub[0] = 1;
	info.bpp[0] = format_to_bpp(format) * 8;
	if (info.bpp[0] < 8)
		return NULL;

	num_planes = 1;
	switch (drmfmt) {
        case DRM_FORMAT_NV12:
		num_planes = 2;
		info.xsub[1] = 2;
		info.ysub[1] = 2;
		info.bpp[1] = info.bpp[0] * 2;
		break;
        case DRM_FORMAT_NV16:
		num_planes = 2;
		info.xsub[1] = 2;
		info.ysub[1] = 1;
		info.bpp[1] = info.bpp[0] * 2;
		break;
	default:
		break;
	}
	for (i = 0; i < num_planes; i++) {
		struct drm_mode_create_dumb creq = {
			.width = mode.hdisplay / info.xsub[i],
			.height = mode.vdisplay / info.ysub[i],
			.bpp = info.bpp[i],
		};

		retval = drmIoctl(fd, DRM_IOCTL_MODE_CREATE_DUMB, &creq);
		if (retval) {
			perror("drmIoctl DRM_IOCTL_MODE_CREATE_DUMB");
			break;
		}

		info.pitches[i] = creq.pitch;
		info.size[i] = creq.size;
		info.bo_handles[i] = creq.handle;
	}

	if (i < num_planes)
		goto out_destroy;

	planes = calloc(num_planes, sizeof(*drmfb->planes));
	if (!planes) {
		fprintf(stderr, "Out of memory\n");
		goto out_destroy;
	}

	for (i = 0; i < num_planes; i++) {
		struct drm_mode_map_dumb mreq = {
			.handle = info.bo_handles[i],
		};

		retval = drmIoctl(fd, DRM_IOCTL_MODE_MAP_DUMB, &mreq);
		if (retval) {
			perror("drmIoctl DRM_IOCTL_MODE_MAP_DUMB");
			break;
		}

		planes[i].buffer.start = (char *)mmap(0, info.size[i],
						      PROT_READ | PROT_WRITE, MAP_SHARED,
						      fd, mreq.offset);
		if (planes[i].buffer.start == MAP_FAILED) {
			perror("Error: failed to map DRM framebuffer device to memory");
			planes[i].buffer.start = NULL;
			break;
		}
		planes[i].buffer.length = info.size[i];
		planes[i].handle = mreq.handle;
	}

	if (i < num_planes)
		goto out_unmap;

	retval = drmModeAddFB2(fd, mode.hdisplay, mode.vdisplay, drmfmt,
			       info.bo_handles, info.pitches,
			       info.offsets, &drmfb->fb_id, 0);
	if (retval) {
		perror("drmModeAddFB2");
		goto out_unmap;
	}

	enc = drmModeGetEncoder(fd, conn->encoder_id);
	if (!enc) {
		fprintf(stderr, "drmModeGetEncoder failed\n");
		goto out_unmap;
	}

	crtc_id = enc->crtc_id;
	drmModeFreeEncoder(enc);

	drmfb->saved_crtc = drmModeGetCrtc(fd, enc->crtc_id);
	retval = drmModeSetCrtc(fd, crtc_id, drmfb->fb_id,
				0, 0, &conn->connector_id, 1, &mode);
	if (retval) {
		fprintf(stderr, "drmModeSetCrtc() failed\n");
		goto out_unmap;
	}

	retval = window_init(&drmfb->fb.window, mode.hdisplay, mode.vdisplay,
			     info.pitches[0], format);
	if (retval) {
		fprintf(stderr, "DRM FB window format error\n");
		goto out_unmap;
	}

	retval = window_bind_buffers(&drmfb->fb.window, &planes[0].buffer,
				     num_planes ? &planes[1].buffer : NULL);
	if (retval) {
		fprintf(stderr, "DRM FB window buffer error\n");
		goto out_unmap;
	}

	drmfb->fb.release = drmfb_release;
	drmfb->planes = planes;
	drmfb->num_planes = num_planes;
	drmfb->crtc_id = crtc_id;
	drmfb->conn_id = conn->connector_id;

	format_to_str(fmtstr, drmfb->fb.window.fourcc);
	fprintf(stderr, "DRM FB (conn: %u, crtc: %u): %u x %u, %s\n",
		drmfb->conn_id, drmfb->crtc_id,
		drmfb->fb.window.width, drmfb->fb.window.height,
		fmtstr);
	return drmfb;

out_unmap:
	while (i-- > 0) {
		if (planes[i].buffer.start)
			munmap(planes[i].buffer.start, planes[i].buffer.length);
	}

	free(planes);
	i = num_planes;
out_destroy:
	while (i-- > 0) {
		struct drm_mode_destroy_dumb dreq = {
			.handle = info.bo_handles[i],
		};

		drmIoctl(fd, DRM_IOCTL_MODE_DESTROY_DUMB, &dreq);
	}

	free(drmfb);
	return NULL;
}

struct fb *drmfb_create(int fd, unsigned format)
{
	struct drmfb *drmfb = NULL;
	drmModeRes *res;
	unsigned i;

	res = drmModeGetResources(fd);
	if (!res) {
		fprintf(stderr, "drmModeGetResources failed");
		goto err_free;
	}

	/* scan all available connectors */
	for (i = 0; i < res->count_connectors; i++) {
		drmModeConnector *conn;

		conn = drmModeGetConnector(fd, res->connectors[i]);
		if (!conn)
			continue;

		if ((conn->connection == DRM_MODE_CONNECTED) && conn->count_modes)
			drmfb = drmfb_setup(fd, conn, format);

		drmModeFreeConnector(conn);
		if (drmfb)
			break;
	}

	drmModeFreeResources(res);
	if (drmfb)
		return &drmfb->fb;

err_free:
	free(drmfb);
	return NULL;
}
#endif /* DRM_ENABLE */

void fb_release(struct fb *fb)
{
	int fd;

	if (!fb)
		return;

	fd = fb->fd;
	if (fb->release)
		fb->release(fb);
	else
		free(fb);
	close(fd);
}

void genfb_release(struct fb *fb)
{
	struct genfb *genfb = to_genfb(fb);

	munmap(genfb->buffer.start, genfb->buffer.length);
	free(genfb);
}

struct fb *genfb_create(int fd, unsigned format)
{
	struct fb_var_screeninfo vinfo;
	struct fb_fix_screeninfo finfo;
	struct genfb *genfb;
	int retval;

	// Get fixed screen information
	retval = ioctl(fd, FBIOGET_FSCREENINFO, &finfo);
	if (retval) {
		perror("Error reading fixed information");
		return NULL;
	}

	// Get variable screen information
	retval = ioctl(fd, FBIOGET_VSCREENINFO, &vinfo);
	if (retval) {
		perror("Error reading variable information");
		return NULL;
	}

	fprintf(stderr, "FB Fixed Info:\n"
		"   %s  @ 0x%lx, len=%d, line=%d bytes,\n",
		finfo.id,
		finfo.smem_start,
		finfo.smem_len,
		finfo.line_length);

	fprintf(stderr, "   Geometry - %u x %u, %u bpp%s\n",
		vinfo.xres,
		vinfo.yres,
		vinfo.bits_per_pixel,
		vinfo.grayscale ? ", greyscale" : "");

	if (vinfo.bits_per_pixel != 32 || vinfo.grayscale) {
		fprintf(stderr, "Unsupported FB format\n");
		return NULL;
	}

	genfb = calloc(1, sizeof(*genfb));
	if (!genfb) {
		fprintf(stderr, "Out of memory\n");
		return NULL;
	}

	// Map the device to memory
	genfb->buffer.start = (char *)mmap(0, finfo.smem_len,
					PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
	if (genfb->buffer.start == MAP_FAILED) {
		perror("Error: failed to map framebuffer device to memory");
		goto err_free;
	}

	genfb->buffer.length = finfo.smem_len;
	if (window_init(&genfb->fb.window, vinfo.xres, vinfo.yres,
			finfo.line_length, V4L2_PIX_FMT_XBGR32)) {
		fprintf(stderr, "FB window format error\n");
		goto err_unmap;
	}

	if (window_bind_buffers(&genfb->fb.window, &genfb->buffer, NULL)) {
		fprintf(stderr, "FB window buffer error\n");
		goto err_unmap;
	}

	genfb->fb.release = genfb_release;
	return &genfb->fb;
err_unmap:
	munmap(genfb->buffer.start, genfb->buffer.length);
err_free:
	free(genfb);
	return NULL;
}

static struct fbtype fbtypes[] = {
	{
		.major = 29,
		.create = genfb_create,
#ifdef DRM_ENABLE
	}, {
		.major = 226,
		.create = drmfb_create,
#endif /* DRM_ENABLE */
	},
};

struct fb *fb_create(const char *fbdev_name, unsigned format)
{
	struct fb *fb = NULL;
	struct stat st;
	unsigned major, i;
	int fd, retval;

	fd = open(fbdev_name, O_RDWR, 0);
	if (fd < 0) {
		fprintf(stderr, "Cannot open '%s': %d, %s\n",
			fbdev_name, errno, strerror(errno));
		return NULL;
	}

	retval = fstat(fd, &st);
	if (retval < 0) {
		perror("fstat");
		goto out_close;
	}

	if (st.st_mode & S_IFMT != S_IFCHR) {
		fprintf(stderr, "invalid device type\n");
		goto out_close;
	}

	major = st.st_rdev >> 8;
	for (i = 0; i < ARRAY_SIZE(fbtypes); i++) {
		if (major == fbtypes[i].major) {
			fb = fbtypes[i].create(fd, format);
			break;
		}
	}

	if (!fb) {
		fprintf(stderr, "framebuffer failed\n");
		goto out_close;
	}

	fb->fd = fd;
	return fb;

out_close:
	close(fd);
	return NULL;
}

/* Params private functions and data */
#define PARAMS_DEF_FBDEV_NAME	"/dev/fb0"
#define PARAMS_DEF_DEV_NAME	"/dev/video0"
#define PARAMS_DEF_FORMAT_NAME	"uyvy"
#define PARAMS_DEF_FRAME_COUNT	70
#define PARAMS_DEF_TIMEOUT	60
#define PARAMS_DEF_N_DEVS	1
#define PARAMS_DEF_IO		IO_METHOD_MMAP
#define PARAMS_DEF_TOP		0
#define PARAMS_DEF_LEFT		0
#define PARAMS_DEF_WIDTH	1920
#define PARAMS_DEF_HEIGHT	1080

static const struct option __long_options[] = {
	{ "device",	required_argument, NULL, 'd' },
	{ "ndev",	required_argument, NULL, 'D' },
	{ "help",	no_argument,       NULL, 'h' },
	{ "mmap",	no_argument,       NULL, 'm' },
	{ "read",	no_argument,       NULL, 'r' },
	{ "userp",	no_argument,       NULL, 'u' },
	{ "output",	no_argument,       NULL, 'o' },
	{ "output_fb",	optional_argument, NULL, 'F' },
	{ "format",	required_argument, NULL, 'f' },
	{ "count", 	required_argument, NULL, 'c' },
	{ "fps_count",	required_argument, NULL, 'z' },
	{ "framerate",	required_argument, NULL, 's' },
	{ "left",	required_argument, NULL, 'L' },
	{ "top",	required_argument, NULL, 'T' },
	{ "width",	required_argument, NULL, 'W' },
	{ "height",	required_argument, NULL, 'H' },
	{ "timeout",	required_argument, NULL, 't' },
	{ 0, 0, 0, 0 }
};

const char __short_options[] = "d:D:hmruoF::f:c:zs:L:T:W:H:t:";

static void __params_usage(struct params *params, const char *name)
{
	fprintf(stderr,
		"Usage: %s [options]\n\n"
		"Version: 1.5 "
#ifdef DRM_ENABLE
		"(DRM+)"
#else
		"(DRM-)"
#endif /* DRM_ENABLE */
		"\nOptions:\n"
		" -d | --device <path>       Set video device path (can be used repeatedly) [%s]\n"
		" -D | --ndev <number>       Use <number> devices starting from /dev/video0 [%i]\n"
		" -h | --help                Print this message and exit\n"
		" -m | --mmap                Use memory mapped buffers%s\n"
		" -r | --read                Use read() calls%s\n"
		" -u | --userp               Use application allocated buffers%s\n"
		" -o | --output              Output video stream to stdout\n"
		" -F | --output_fb [path]    Output video stream to framebuffer [%s]\n"
		" -f | --format <name>       Set pixel format [%s]\n"
		" -c | --count <number>      Set number of frames to grab [%i]\n"
		" -z | --fps_count           Show FPS\n"
		" -s | --framerate <number>  Set framerate\n"
		" -L | --left <number>       Set video left crop [%i]\n"
		" -T | --top <number>        Set video top crop [%i]\n"
		" -W | --width <number>      Set video width [%i]\n"
		" -H | --height <number>     Set video height [%i]\n"
		" -t | --timeout <number>    Set select() timeout, Sec [%i]\n",
		name, PARAMS_DEF_DEV_NAME, PARAMS_DEF_N_DEVS,
		PARAMS_DEF_IO == IO_METHOD_MMAP ? " [default]" : "",
		PARAMS_DEF_IO == IO_METHOD_READ ? " [default]" : "",
		PARAMS_DEF_IO == IO_METHOD_USERPTR ? " [default]" : "",
		PARAMS_DEF_FBDEV_NAME, PARAMS_DEF_FORMAT_NAME,
		PARAMS_DEF_FRAME_COUNT, PARAMS_DEF_LEFT, PARAMS_DEF_TOP,
		PARAMS_DEF_WIDTH, PARAMS_DEF_HEIGHT, PARAMS_DEF_TIMEOUT);
	fprintf(stderr, "Pixel formats:\n");
	format_list_supported(stderr, 1, 27);
}

/* Params public API */
void params_release(struct params *params)
{
	if (!params)
		return;

	free(params->dev_names);
	free(params);
}

struct params *params_create(void)
{
	struct params *params = calloc(1, sizeof(*params));

	if (!params) {
		fprintf(stderr, "Out of memory\n");
		return NULL;
	}

	params->left = PARAMS_DEF_LEFT;
	params->top = PARAMS_DEF_TOP;
	params->width = PARAMS_DEF_WIDTH;
	params->height = PARAMS_DEF_HEIGHT;
	params->frame_count = PARAMS_DEF_FRAME_COUNT;
	params->io = PARAMS_DEF_IO;
	params->n_devs = PARAMS_DEF_N_DEVS;
	params->timeout = PARAMS_DEF_TIMEOUT; /* seconds */
	params->format_name = PARAMS_DEF_FORMAT_NAME;
	return params;
}

int params_parse(struct params *params, int argc, char *argv[])
{
	size_t dev = 0;

        for (;;) {
		int idx, c;

		c = getopt_long(argc, argv, __short_options, __long_options, &idx);
		if (-1 == c)
                        break;

		switch (c) {
		case 0: /* getopt_long() flag */
			break;

		case 'd':
			if (dev >= params->n_dev_names) {
				const char **tmp;

				params->n_dev_names += 16;
				tmp = realloc(params->dev_names,
					      params->n_dev_names * sizeof(*params->dev_names));
				if (!tmp)
					return -1;
				params->dev_names = tmp;
			}
			params->dev_names[dev++] = optarg;
			break;

		case 'D':
			errno = 0;
			params->n_devs = strtol(optarg, NULL, 0);
			if (errno)
				return -1;
                        break;

		case 'm':
			params->io = IO_METHOD_MMAP;
			break;

		case 'r':
			params->io = IO_METHOD_READ;
			break;

		case 'u':
			params->io = IO_METHOD_USERPTR;
			break;

		case 'o':
			params->out_fp = stdout;
			break;

		case 'F':
			params->fbdev_name = optarg ? optarg : PARAMS_DEF_FBDEV_NAME;
			break;

		case 'f':
			params->format_name = optarg;
			break;

		case 'c':
			errno = 0;
			params->frame_count = strtol(optarg, NULL, 0);
			if (errno)
				return -1;
                        break;

		case 'z':
			params->fps_count = 1;
			break;

		case 's':
			errno = 0;
			params->framerate = strtol(optarg, NULL, 0);
			if (errno)
				return -1;
			break;

		case 'L':
			errno = 0;
			params->left = strtol(optarg, NULL, 0);
			if (errno)
				return -1;
			break;

		case 'T':
			errno = 0;
			params->top = strtol(optarg, NULL, 0);
			if (errno)
				return -1;
			break;

		case 'W':
			errno = 0;
			params->width = strtol(optarg, NULL, 0);
			if (errno)
				return -1;
			break;

		case 'H':
			errno = 0;
			params->height = strtol(optarg, NULL, 0);
			if (errno)
				return -1;
			break;

		case 't':
			errno = 0;
			params->timeout = strtol(optarg, NULL, 0);
			if (errno)
				return -1;
			break;

		default: 
			__params_usage(params, argv[0]);
			return 1;
		}
	}

	params->n_dev_names = dev;

	if (dev > params->n_devs)
		params->n_devs = dev;

	if (!params->n_devs)
		return -1;

	return 0;
}

const char *params_get_dev_name(struct params *params, unsigned idx)
{
	if (idx >= params->n_dev_names)
		return NULL;

	return params->dev_names[idx];
}

/****************** Application ******************/

static volatile sig_atomic_t signal_caught;

void sig_handler(int sig)
{
	signal_caught = sig;
}

static int main_loop(struct camera **cams, struct window *windows, unsigned n_devs,
		     unsigned frame_count, unsigned timeout, FILE *fp)
{
	/* Give time to queue buffers at start streaming by VIN module */
	// usleep(34000*3);

	while (frame_count) {
		struct timeval tv;
		fd_set fds;
		unsigned i;
		int retval, nfd = 0;

		if (signal_caught)
			return -EINTR;

		FD_ZERO(&fds);
		for (i = 0; i < n_devs; i++)
			camera_fd_set(cams[i], &nfd, &fds);

		/* Timeout. */
		tv.tv_sec = timeout;
		tv.tv_usec = 0;

		retval = select(nfd + 1, &fds, NULL, NULL, &tv);
		if (-1 == retval) {
			if (EINTR == errno)
				return -EINTR;

			perror("select");
			return -1;
		}

		if (0 == retval) {
			fprintf(stderr, "select timeout\n");
			return -1;
		}

                retval = 0;
                for (i = 0; i < n_devs; i++) {
			if (signal_caught)
				return -EINTR;

			if (camera_fd_isset(cams[i], &fds)) {
				struct v4l2_buffer buf;
				const struct window *src;

				if (camera_frame_get(cams[i], &buf) < 0)
					continue;

				retval++;

				if (windows) {
					src = camera_frame_window_set(cams[i], &buf);
					if (src)
						window_copy(src, &windows[i]);
				}

				camera_frame_dump(cams[i], &buf, fp);

				camera_frame_put(cams[i], &buf);
			}
		}

                if (retval)
			frame_count--;
	}

	return 0;
}

#define DEVNAME_SIZE	64

int main(int argc, char **argv)
{
	struct params *params;
	struct fb *fb;
	struct camera **cams;
	struct window *windows;
	unsigned i, format;
	int retval;

	signal(SIGINT, sig_handler);
	signal(SIGTERM, sig_handler);

	params = params_create();
	if (!params)
		return -1;

	retval = params_parse(params, argc, argv);
	if (retval)
		goto err_params;

	format = format_find(params->format_name);
	fb = NULL;
	windows = NULL;
	if (params->fbdev_name) {
		unsigned n_windows = params->n_devs;

		fb = fb_create(params->fbdev_name, format);
		if (!fb) {
			retval = -1;
			goto err_params;
		}

		n_windows = params->n_devs;
		windows = fb_windows_create(fb, n_windows);
		if (!windows)
			goto err_fb;
	}

	cams = calloc(params->n_devs, sizeof(*cams));
	if (!cams) {
		fprintf(stderr, "Out of memory\n");
		retval = -1;
		goto err_fb;
	}

	for (i = 0; i < params->n_devs; i++) {
		char buf[DEVNAME_SIZE];
		const char *dev_name = params_get_dev_name(params, i);

		if (!dev_name) {
			snprintf(buf, DEVNAME_SIZE, "/dev/video%u", i);
			dev_name = buf;
		}
		cams[i] = camera_create(dev_name,
					params->left, params->top,
					params->width, params->height,
					format, params->io,
					params->framerate, params->fps_count);
		if (!cams[i])
			break;
        }

	if (i == params->n_devs) {
		for (i = 0; i < params->n_devs; i++) {
			if (camera_start(cams[i]) < 0)
				break;
		}
	}

	if (i == params->n_devs)
		retval = main_loop(cams, windows, params->n_devs,
				   params->frame_count, params->timeout,
				   params->out_fp);
	else
		retval = -1;

	for (i = 0; i < params->n_devs; i++)
		camera_release(cams[i]);

	free(cams);
	free(windows);
err_fb:
	fb_release(fb);
err_params:
	params_release(params);
        return retval;
}
