import sys
import os
import argparse
from kconfiglib import Kconfig

# Adapted from edk2-nvidia/Silicon/NVIDIA/edk2nv/stuart/builder.py BuildConfigFile method
def build_config_file(kconf_path, configs, config_out):
    kconf = Kconfig(kconf_path, warn_to_stderr=False,
                    suppress_traceback=True)
    kconf.warn_assign_undef = True
    kconf.warn_assign_override = False
    kconf.warn_assign_redun = False

    print(kconf.load_config(configs[0]))
    for config in configs[1:]:
        # replace=False creates a merged configuration
        print(kconf.load_config(config, replace=False))

    kconf.write_config(os.devnull)

    if kconf.warnings:
        # Put a blank line between warnings to make them easier to read
        for warning in kconf.warnings:
            print("\n" + warning, file=sys.stderr)

        # Turn all warnings into errors, so that e.g. assignments to undefined
        # Kconfig symbols become errors.
        #
        # A warning is generated by this script whenever a symbol gets a
        # different value than the one it was assigned. Keep that one as just a
        # warning for now.
        raise ValueError("Aborting due to Kconfig warnings")

    # Write the merged configuration
    print(kconf.write_config(config_out))
    return 0

def main():
    parser = argparse.ArgumentParser(description="Configuration merge tool for NVIDIA UEFI/EDK2 builds")
    parser.add_argument("--kconfig-path", help="path to base Kconfig file", required=True)
    parser.add_argument("--output-directory", help="directory for writing .config and config.dsc.inc files")
    parser.add_argument("config_files", nargs="+")
    args = parser.parse_args()
    if args.output_directory:
        output_file = os.path.join(args.output_directory, ".config")
        output_dsc = os.path.join(args.output_directory, "config.dsc.inc")
    else:
        output_file = ".config"
        output_dsc = "config.dsc.inc"
    build_config_file(args.kconfig_path, args.config_files, output_file)
    with open(output_file, "r") as f, open(output_dsc, "w") as fo:
        for line in f:
            fo.write(line.replace('"', '').replace("'", ""))
    return 0

if __name__ == '__main__':
    try:
        ret = main()
    except Exception:
        ret = 1
        import traceback
        traceback.print_exc()
    sys.exit(ret)
