# Checks if atomic operations are supported natively or if linking against
# libatomic is needed.

# Check inspired by LLVMs cmake/modules/CheckAtomic.cmake

INCLUDE(CheckCXXSourceCompiles)
INCLUDE(CheckLibraryExists)

function(check_working_cxx_atomics varname)
  set(OLD_CMAKE_REQUIRED_FLAGS ${CMAKE_REQUIRED_FLAGS})
  set(CMAKE_REQUIRED_FLAGS "${CMAKE_REQUIRED_FLAGS} -std=c++11")
  CHECK_CXX_SOURCE_COMPILES("
#include <atomic>
std::atomic<int> x;
int main() {
  return x;
}
" ${varname})
  set(CMAKE_REQUIRED_FLAGS ${OLD_CMAKE_REQUIRED_FLAGS})
endfunction(check_working_cxx_atomics)

function(check_working_cxx_atomics64 varname)
  set(OLD_CMAKE_REQUIRED_FLAGS ${CMAKE_REQUIRED_FLAGS})
  set(CMAKE_REQUIRED_FLAGS "-std=c++11 ${CMAKE_REQUIRED_FLAGS}")
  CHECK_CXX_SOURCE_COMPILES("
#include <atomic>
#include <cstdint>
std::atomic<uint64_t> x (0);
std::atomic<double> y (0);
int main() {
  uint64_t i = x.load(std::memory_order_relaxed);
  return int(y);
}
" ${varname})
  set(CMAKE_REQUIRED_FLAGS ${OLD_CMAKE_REQUIRED_FLAGS})
endfunction(check_working_cxx_atomics64)

# Check if atomics work without libatomic
check_working_cxx_atomics(HAVE_CXX_ATOMICS_WITHOUT_LIB)

if(NOT HAVE_CXX_ATOMICS_WITHOUT_LIB)
  check_library_exists(atomic __atomic_fetch_add_4 "" HAVE_LIBATOMIC)
  if( HAVE_LIBATOMIC )
    list(APPEND CMAKE_REQUIRED_LIBRARIES "atomic")
    check_working_cxx_atomics(HAVE_CXX_ATOMICS_WITH_LIB)
    if (NOT HAVE_CXX_ATOMICS_WITH_LIB)
      message(FATAL_ERROR "Host compiler must support std::atomic!")
    endif()
  else()
    message(FATAL_ERROR "Host compiler appears to require libatomic, but cannot find it.")
  endif()
endif()

# Check if 64bit atomics work without libatomic
check_working_cxx_atomics64(HAVE_CXX_ATOMICS64_WITHOUT_LIB)

if(NOT HAVE_CXX_ATOMICS64_WITHOUT_LIB)
  check_library_exists(atomic __atomic_load_8 "" HAVE_CXX_LIBATOMICS64)
  if(HAVE_CXX_LIBATOMICS64)
    list(APPEND CMAKE_REQUIRED_LIBRARIES "atomic")
    check_working_cxx_atomics64(HAVE_CXX_ATOMICS64_WITH_LIB)
    if (NOT HAVE_CXX_ATOMICS64_WITH_LIB)
      message(FATAL_ERROR "Host compiler must support std::atomic!")
    endif()
  else()
    message(FATAL_ERROR "Host compiler appears to require libatomic, but cannot find it.")
  endif()
endif()

